declare module 'seisplotjs/src/animatedseismograph' {
  import { DateTime, Duration, Interval } from "luxon";
  import { DataLinkPacket } from "seisplotjs/src/datalink";
  import { OrganizedDisplay } from "seisplotjs/src/organizeddisplay";
  import { AlignmentLinkedTimeScale } from "seisplotjs/src/scale";
  import { SeisPlotElement } from "seisplotjs/src/spelement";
  import { Network } from "seisplotjs/src/stationxml";
  export class AnimatedTimeScaler {
      alignmentTime: DateTime;
      timeScale: AlignmentLinkedTimeScale;
      minRedrawMillis: number;
      goAnimation: boolean;
      previousStep: DOMHighResTimeStamp;
      _animationId: number;
      constructor(timeScale: AlignmentLinkedTimeScale, alignmentTime?: DateTime, minRedrawMillis?: number);
      animate(): void;
      animateOnce(): void;
      pause(): void;
      stepper(timestamp: DOMHighResTimeStamp): void;
      step(): void;
  }
  export type RTDisplayContainer = {
      organizedDisplay: OrganizedDisplay;
      animationScaler: AnimatedTimeScaler;
      packetHandler: (packet: DataLinkPacket) => void;
      config: RTConfig;
  };
  export type RTConfig = {
      duration: Duration;
      alignmentTime: DateTime;
      offset: Duration;
      minRedrawMillis: number;
      networkList: Array<Network>;
  };
  /**
   * Validates the object for typescript and adds any missing required
   * fields. Passing in an empty object is allowed.
   * @param  configObj  config object
   * @returns true if object has correct structure, with defaults populated
   */
  export function isValidRTConfig(configObj: unknown): configObj is RTConfig;
  export function createRealtimeDisplay(config: unknown): RTDisplayContainer;
  export function internalCreateRealtimeDisplay(config: RTConfig): RTDisplayContainer;
  /**
   * Trims all seismograms to the given window in place. This should not
   * cause a redraw.
   *
   * @param orgDisplay OrganizedDisplay to trim
   * @param  timeRange  time window to coarse trim the data to
   */
  export function trim(orgDisplay: OrganizedDisplay, timeRange: Interval): void;
  /**
   * Calculates the duration covered by one pixel on the time axis. This is the optimal
   * time interval for updating the animation of a real time display as updates more
   * frequently than this tend to cause more flikering than actual movement of the seismogram.
   * @param seismograph the seismograph to get pixel size from
   * @returns the pixel width as a Duration
   */
  export function calcOnePixelDuration(seismograph: SeisPlotElement): Duration;

}
declare module 'seisplotjs/src/areautil' {
  export interface Location {
      latitude: number;
      longitude: number;
  }
  export function inArea(bounds: Array<Location>, point: Location): boolean;
  export function polygonPointCheck(lonA: number, latA: number, lonB: number, latB: number): number;

}
declare module 'seisplotjs/src/axisutil' {
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  import "d3-transition";
  import type { ScaleTime } from "d3-scale";
  import { DateTime, Interval } from "luxon";
  export interface HandlebarsInput {
      seisConfig?: SeismographConfig;
      seisDataList?: unknown;
      seisXData?: unknown;
      seisYData?: unknown;
  }
  export class LuxonTimeScale {
      interval: Interval;
      range: [number, number];
      constructor(interval: Interval, range: [number, number]);
      for(d: DateTime): number;
      invert(v: number): DateTime;
      domain(): Interval;
      get d3scale(): ScaleTime<number, number, never>;
      millisPerPixel(): number;
  }
  export function drawXLabel(svgEl: SVGElement, seismographConfig: SeismographConfig, height: number, width: number, handlebarsInput?: HandlebarsInput): void;
  export function drawXSublabel(svgEl: SVGElement, seismographConfig: SeismographConfig, height: number, width: number, // eslint-disable-next-line no-unused-vars
  handlebarsInput?: HandlebarsInput): void;
  export function drawYLabel(svgEl: SVGElement, seismographConfig: SeismographConfig, height: number, width: number, handlebarsInput?: HandlebarsInput): void;
  /**
   * Draws Y axis sublabel, possibly reflecting units of seismograph.
   *
   * @param  svgEl             svg to draw to
   * @param  seismographConfig  config options
   * @param  height             height of svg
   * @param  width              width of svg
   * @param  handlebarsInput    optional key-values for handlebars
   * @param  unitsLabel          option label representing units if config.ySublabelIsUnit
   */
  export function drawYSublabel(svgEl: SVGElement, seismographConfig: SeismographConfig, height: number, width: number, // eslint-disable-next-line no-unused-vars
  handlebarsInput?: HandlebarsInput, unitsLabel?: string): void;
  export function drawTitle(svgEl: SVGElement, seismographConfig: SeismographConfig, height: number, width: number, handlebarsInput?: HandlebarsInput): void;
  /**
   * Draws axis labels and title, possibly reflecting units of seismograph.
   *
   * @param  svgEl             svg to draw to
   * @param  seismographConfig  config options
   * @param  height             height of svg
   * @param  width              width of svg
   * @param  handlebarsInput    optional key-values for handlebars
   * @param  unitsLabel          option label representing units if config.ySublabelIsUnit
   */
  export function drawAxisLabels(svgEl: SVGElement, seismographConfig: SeismographConfig, height: number, width: number, handlebarsInput?: HandlebarsInput, unitsLabel?: string): void;

}
declare module 'seisplotjs/src/components' {
  import { Channel } from "seisplotjs/src/stationxml";
  import { LatLonBox, LatLonRadius } from "seisplotjs/src/fdsncommon";
  import { FDSNSourceId } from "seisplotjs/src/fdsnsourceid";
  export const SOURCEID_LIST_ELEMENT = "sp-sourceid-list";
  export const CHANNEL_LIST_ELEMENT = "sp-channel-list";
  export const CHANNEL_CODE_ELEMENT = "sp-channel-code-input";
  export const MINMAX_ELEMENT = "sp-minmax";
  export const LATLONRADIUS_ELEMENT = "sp-latlon-radius";
  export const LATLONBOX_ELEMENT = "sp-latlon-box";
  export const LATLON_CHOICE_ELEMENT = "sp-latlon-choice";
  export function numberOrNaN(a: string | number | null): number;
  export function numberFromInput(root: ShadowRoot | Element | null, query: string): number;
  export function labeledTextInput(label: string, defaultVal: string, classname?: string | null): HTMLElement;
  export function labeledNumberInput(label: string, defaultVal: string): HTMLElement;
  export class ChannelCodeInput extends HTMLElement {
      constructor();
      attributeChangedCallback(name: string, oldValue: string, newValue: string): void;
      static get observedAttributes(): string[];
      get network(): string;
      set network(n: string);
      get station(): string;
      set station(n: string);
      get location(): string;
      set location(n: string);
      get channel(): string;
      set channel(n: string);
      _getInputValue(name: string): string;
      _setInputValue(name: string, val: string): void;
  }
  export class ChannelListChooser extends HTMLElement {
      channels: Array<Channel>;
      selected_channels: Set<Channel>;
      constructor();
      draw_element(): void;
      setChannels(channels: Array<Channel>): void;
      appendChannels(channels: Array<Channel>): void;
      get type(): string;
      set type(s: string);
      attributeChangedCallback(_name: string, _oldValue: string, _newValue: string): void;
      selectedChannels(): Array<Channel>;
  }
  export class SourceIdListChooser extends HTMLElement {
      sourceIdList: Array<FDSNSourceId>;
      selected_sourceIds: Set<FDSNSourceId>;
      constructor();
      draw_element(): void;
      setSourceIds(sourceIdList: Array<FDSNSourceId>): void;
      appendSourceIds(sourceIdList: Array<FDSNSourceId>): void;
      get type(): string;
      set type(s: string);
      attributeChangedCallback(_name: string, _oldValue: string, _newValue: string): void;
      selectedSourceIds(): Array<FDSNSourceId>;
  }
  export class LabeledMinMax extends HTMLElement {
      default_min: number;
      default_max: number;
      constructor();
      attributeChangedCallback(name: string, oldValue: string, newValue: string): void;
      static get observedAttributes(): string[];
      draw_element(): void;
      validate(lastChanged?: string): void;
      get lowerbound(): number;
      set lowerbound(v: number);
      get upperbound(): number;
      set upperbound(v: number);
      get min(): number;
      set min(v: number);
      get max(): number;
      set max(v: number);
      updateMinMax(): void;
      connectedCallback(): void;
  }
  /**
   * ensures input number is -90 <= value <= 90
   *
   * @param  value              input latitude
   * @returns       output latitude in range, zero if NaN
   */
  export function validateLatitude(value: number): number;
  /**
   * ensures input number is -180 <= value <= 360
   *
   * @param  value              input longitude
   * @returns       output longitude in range, zero if NaN
   */
  export function validateLongitude(value: number): number;
  export class LatLonRadiusEl extends HTMLElement {
      constructor();
      draw(): void;
      attributeChangedCallback(_name: string, _oldValue: string, _newValue: string): void;
      static get observedAttributes(): string[];
      get latitude(): number;
      set latitude(v: number);
      get longitude(): number;
      set longitude(v: number);
      get minRadius(): number;
      set minRadius(v: number);
      get maxRadius(): number;
      set maxRadius(v: number);
      asLatLonRadius(): LatLonRadius;
      toString(): string;
      _doUpdateCallback(): void;
  }
  export class LatLonBoxEl extends HTMLElement {
      constructor();
      attributeChangedCallback(_name: string, _oldValue: string, _newValue: string): void;
      static get observedAttributes(): string[];
      draw(): void;
      get south(): number;
      set south(value: number);
      get north(): number;
      set north(value: number);
      get west(): number;
      set west(value: number);
      get east(): number;
      set east(value: number);
      asLatLonBox(): LatLonBox;
      validate(lastChanged: string): void;
      toString(): string;
  }
  export class LatLonChoice extends HTMLElement {
      constructor();
      static get observedAttributes(): string[];
      attributeChangedCallback(name: string, oldValue: string, newValue: string): void;
      draw(): void;
      /**
       * returns choosen sub element, either LatLonBox or LatLonRadius or
       * null if choosen is "all"
       *
       * @returns choosen element
       */
      choosen(): null | LatLonBoxEl | LatLonRadiusEl;
      get latLonBox(): LatLonBoxEl;
      get latLonRadius(): LatLonRadiusEl;
  }

}
declare module 'seisplotjs/src/cssutil' {
  export const AUTO_CLASSED = "autoseisplotjs";
  export const AUTO_COLOR_SELECTOR = "seisplotjsautocolor";
  export const G_DATA_SELECTOR = "seisplotjsdata";
  /**
   * Inserts text as css into the head of an html document. No checking
   * as to validity of the css is done, just inserts a style
   * element at the beginning of the head.
   *
   * @param  cssText textual css for insertion
   * @param id optional id for style element
   * @returns the style html element inserted
   */
  export function insertCSS(cssText: string, id: string): HTMLElement;
  export function isCSSInserted(id: string): boolean;
  export function isIdStyleElement(c: Element, id: string): boolean;

}
declare module 'seisplotjs/src/datalink' {
  import * as miniseed from "seisplotjs/src/miniseed";
  import * as mseed3 from "seisplotjs/src/mseed3";
  import { DateTime } from "luxon";
  /** const for datalink protocol for web sockets, DataLink1.0 */
  export const DATALINK_PROTOCOL = "DataLink1.0";
  /** enum for modes, either QUERY or STREAM */
  export enum MODE {
      Query = "QUERY",
      Stream = "STREAM"
  }
  /** const for query mode, QUERY */
  export const QUERY_MODE = MODE.Query;
  /** const for streaming mode, STREAM */
  export const STREAM_MODE = MODE.Stream;
  /** const for maximum process number to create fake number for datalink id, 2^16-2 */
  export const MAX_PROC_NUM: number;
  /** const for fake user name for datalink id, browser */
  export const USER_BROWSER = "browser";
  /** const for fake program name for datalink id, seisplotjs */
  export const DEFAULT_PROGRAM = "seisplotjs";
  /** const for fake architecture for datalink id, javascript */
  export const DEFAULT_ARCH = "javascript";
  /** const for error response, ERROR */
  export const ERROR = "ERROR";
  /** const for ok response, OK */
  export const OK = "OK";
  /** const for info response, INFO */
  export const INFO = "INFO";
  /** const for id response, ID */
  export const ID = "ID";
  export const PACKET = "PACKET";
  export const STREAM = "STREAM";
  export const ENDSTREAM = "ENDSTREAM";
  export const MSEED_TYPE = "/MSEED";
  export const MSEED3_TYPE = "/MSEED3";
  export const IRIS_RINGSERVER_URL = "ws://rtserve.iris.washington.edu/datalink";
  /**
   * A websocket based Datalink connection.
   *
   * Note this cannot connect directly to a native TCP socket, instead it
   * sends the datalink protocol over a websocket.
   *
   * Currently only the IRIS
   * ringserver, https://github.com/iris-edu/ringserver,
   * supports websockets, but it may be possible to use thrid party
   * tools to proxy the websocket to a TCP datalink socket.
   *
   * The datalink protocol is documented here
   *  https://earthscope.github.io/libdali/datalink-protocol.html
   *
   * @param url websocket url to the ringserver
   * @param packetHandler callback for packets as they arrive
   * @param errorHandler callback for errors
   */
  export class DataLinkConnection {
      url: string;
      /** @private */
      _mode: MODE;
      packetHandler: (packet: DataLinkPacket) => void;
      errorHandler: (error: Error) => void;
      closeHandler: null | ((close: CloseEvent) => void);
      serverId: string | null;
      clientIdNum: number;
      programname: string;
      username: string;
      architecture: string;
      /** @private */
      _responseResolve: null | ((response: DataLinkResponse) => void);
      /** @private */
      _responseReject: null | ((error: Error) => void);
      webSocket: WebSocket | null;
      constructor(url: string, packetHandler: (packet: DataLinkPacket) => void, errorHandler: (error: Error) => void);
      /**
       * Set a callback function called when the connection is closed.
       *
       * @param  closeHandler callback function
       */
      setOnClose(closeHandler: (close: CloseEvent) => void): void;
      /**
       * creates the websocket connection and sends the client ID.
       *
       *  @returns a Promise that resolves to the server's ID.
       */
      connect(): Promise<string>;
      /**
       * @returns true if the websocket is connected (non-null)
       */
      isConnected(): boolean;
      /**
       * @returns the current mode, QUERY_MODE or STREAM_MODE
       */
      get mode(): string;
      /**
       * Switches to streaming mode to receive data packets from the ringserver.
       *
       * @returns promise to the response
       */
      stream(): Promise<DataLinkResponse>;
      /**
       * Switches back to query mode to enable commands to be sent to the ringserver.
       */
      endStream(): void;
      /**
       * Closes the connection and the underlying websocket. No communication
       * is possible until connect() is called again.
       */
      close(): void;
      /**
       * Send a ID Command. Command is a string.
       *
       * @returns a Promise that resolves to the response from the ringserver.
       */
      sendId(): Promise<string>;
      /**
       * encodes as a Datalink packet, header with optional data section as
       * binary Uint8Array. Size of the binary data is appended
       * to the header if present.
       *
       * @param header the command/header string
       * @param data optional data portion
       * @returns datalink packet as an ArrayBuffer
       */
      encodeDL(header: string, data?: Uint8Array): ArrayBuffer;
      /**
       * sends the header with optional binary data
       * as the data section. Size of the data is appended
       * to the header before sending if present.
       *
       * @param header header to send
       * @param data optional data to send
       */
      sendDLBinary(header: string, data?: Uint8Array): void;
      /**
       * sends the command as header with optional dataString
       * as the data section. Size of the dataString is appended
       * to the header before sending.
       *
       * @param command the command/header string
       * @param dataString optional data portion of packet
       */
      sendDLCommand(command: string, dataString?: string): void;
      /**
       * Send a DataLink Command and await the response. Command is a string.
       *
       * @param header packet header
       * @param data optional data portion of packet
       * @returns a Promise that resolves with the webSocket MessageEvent.
       */
      awaitDLBinary(header: string, data?: Uint8Array): Promise<DataLinkResponse | DataLinkPacket>;
      /**
       * Send a DataLink Command and await the response. Command is a string.
       * Returns a Promise that resolves with the webSocket MessageEvent.
       *
       * @param command the command/header string
       * @param dataString optional data portion of packet
       * @returns promise to server's response
       */
      awaitDLCommand(command: string, dataString?: string): Promise<DataLinkResponse | DataLinkPacket>;
      /**
       * Writes data to the ringserver and awaits a acknowledgement.
       *
       * @param   streamid    stream id for packet header
       * @param   hpdatastart start of timewindow the packet covers
       * @param   hpdataend   end of timewindow the packet covers
       * @param   data        optional data to send
       * @returns             promise to server's response
       */
      writeAck(streamid: string, hpdatastart: DateTime, hpdataend: DateTime, data?: Uint8Array): Promise<DataLinkResponse | DataLinkPacket>;
      /**
       * Makes sure a response actually is a DataLinkResponse
       *
       * @param   dl datalink packet/response
       * @returns DataLinkResponse after checking instanceof
       * @throws Error if not a DataLinkResponse
       */
      static ensureDataLinkResponse(dl: DataLinkResponse | DataLinkPacket): DataLinkResponse;
      /**
       * Makes sure a response actually is a DataLinkPacket
       *
       * @param   dl datalink packet/response
       * @returns DataLinkPacket after checking instanceof
       * @throws Error if not a DataLinkPacket
       */
      static ensureDataLinkPacket(dl: DataLinkResponse | DataLinkPacket): DataLinkPacket;
      /**
       * Send id and await server's response. All of these are can more or less
       * be filled with dummy values. They are mostly used for logging and debugging
       * on the server side.
       *
       * @param programname name of program, ex seisplotjs
       * @param username name of user, ex browser
       * @param processid process number, used to differentiate between multiple running instances
       * @param architecture cpu architecture, ex javascript
       * @returns promise to servers response
       */
      id(programname: string, username: string, processid: string, architecture: string): Promise<DataLinkResponse>;
      /**
       * Send info command for infoType.
       *
       * @param infoType type to get info for
       * @returns promise to server's response
       */
      info(infoType: string): Promise<DataLinkResponse>;
      infoStatus(): Promise<StatusResponse>;
      infoStreams(): Promise<StreamsResponse>;
      infoConnections(): Promise<ConnectionsResponse>;
      /**
       * Send position after command.
       *
       * @param time time to position after
       * @returns promise to server's response
       */
      positionAfter(time: DateTime): Promise<DataLinkResponse>;
      /**
       * Send position after command.
       *
       * @param hpTime time to position after
       * @returns promise to server's response
       */
      positionAfterHPTime(hpTime: number): Promise<DataLinkResponse>;
      /**
       * Send match command.
       *
       * @param pattern regular expression to match streams
       * @returns promise to server's response
       */
      match(pattern: string): Promise<DataLinkResponse>;
      /**
       * Send reject command.
       *
       * @param pattern regular expression to reject streams
       * @returns promise to server's response
       */
      reject(pattern: string): Promise<DataLinkResponse>;
      /**
       * Read a single packet for the given id.
       *
       * @param packetId id of the packet of interest
       * @returns promise to server's response
       */
      read(packetId: string): Promise<DataLinkPacket>;
      /**
       * Handles a web socket message from the data link connection.
       *
       * @private
       * @param wsEvent web socket event to handle
       */
      handle(wsEvent: MessageEvent): void;
      handleArrayBuffer(rawData: ArrayBuffer): void;
      /**
       * handle errors that arise
       *
       * @private
       * @param   error the error
       */
      handleError(error: Error): void;
  }
  /**
   * Datalink response, used for ID, INFO, OK and ERROR responses.
   */
  export class DataLinkResponse {
      type: string;
      value: string;
      message: string;
      constructor(type: string, value: string, message: string);
      isError(): boolean;
      toString(): string;
      static parse(header: string, data?: DataView): DataLinkResponse;
  }
  /**
   * Represents a Datalink packet from the ringserver.
   *
   */
  export class DataLinkPacket {
      header: string;
      data: DataView;
      streamId: string;
      pktid: string;
      hppackettime: string;
      hppacketstart: string;
      hppacketend: string;
      dataSize: number;
      _miniseed: null | miniseed.DataRecord;
      _mseed3: null | mseed3.MSeed3Record;
      constructor(header: string, dataview: DataView);
      /**
       * Packet start time as a DateTime.
       *
       * @returns start time
       */
      get packetStart(): DateTime;
      /**
       * Packet end time as a DateTime.
       *
       * @returns end time
       */
      get packetEnd(): DateTime;
      /**
       * Packet time as a DateTime.
       *
       * @returns packet time
       */
      get packetTime(): DateTime;
      /**
       * is this packet a miniseed packet
       *
       * @returns          true if it is miniseed
       */
      isMiniseed(): boolean;
      /**
       * Parsed payload as a miniseed data record, if the streamid
       * ends with '/MSEED', null otherwise.
       *
       * @returns miniseed DataRecord or null
       */
      asMiniseed(): miniseed.DataRecord | null;
      /**
       * is this packet a miniseed3 packet
       *
       * @returns          true if it is miniseed3
       */
      isMiniseed3(): boolean;
      /**
       * Parsed payload as a miniseed3 data record, if the data format is 3, null otherwise.
       *
       * @returns miniseed3 DataRecord or null
       */
      asMiniseed3(): mseed3.MSeed3Record | null;
  }
  export class DataLinkIdStats {
      version: string;
      serverId: string;
      capabilities: Array<string>;
      constructor(version: string, serverId: string, capabilities: Array<string>);
      /**
       * Parses the attributes of a <DataLink> xml element.
       *
       * @param  statusEl               DataLink XML element
       * @returns  the id stats
       */
      static parseXMLAttributes(statusEl: Element): DataLinkIdStats;
      toString(): string;
  }
  export class DataLinkStats {
      startTime: DateTime;
      ringVersion: string;
      ringSize: number;
      packetSize: number;
      maximumPacketID: number;
      maximumPackets: number;
      memoryMappedRing: boolean;
      volatileRing: boolean;
      totalConnections: number;
      totalStreams: number;
      txPacketRate: number;
      txByteRate: number;
      rxPacketRate: number;
      rxByteRate: number;
      earliestPacketID: number;
      earliestPacketCreationTime: DateTime;
      earliestPacketDataStartTime: DateTime;
      earliestPacketDataEndTime: DateTime;
      latestPacketID: number;
      latestPacketCreationTime: DateTime;
      latestPacketDataStartTime: DateTime;
      latestPacketDataEndTime: DateTime;
      constructor(startTime: DateTime, ringVersion: string, ringSize: number, packetSize: number, maximumPacketID: number, maximumPackets: number, memoryMappedRing: boolean, volatileRing: boolean, totalConnections: number, totalStreams: number, txPacketRate: number, txByteRate: number, rxPacketRate: number, rxByteRate: number, earliestPacketID: number, earliestPacketCreationTime: DateTime, earliestPacketDataStartTime: DateTime, earliestPacketDataEndTime: DateTime, latestPacketID: number, latestPacketCreationTime: DateTime, latestPacketDataStartTime: DateTime, latestPacketDataEndTime: DateTime);
      /**
       * Parses the attributes of a <Status> xml element.
       *
       * @param  statusEl   DataLink <Status> XML element
       * @returns  the stats
       */
      static parseXMLAttributes(statusEl: Element): DataLinkStats;
      toString(): string;
  }
  export class ThreadStat {
      flags: Array<string>;
      type: Array<string>;
      port: number;
      constructor(flags: Array<string>, type: Array<string>, port: number);
      /**
       * Parses the attributes of a <Status> xml element.
       *
       * @param  statusEl   DataLink <Status> XML element
       * @returns  the stats
       */
      static parseXMLAttributes(statusEl: Element): ThreadStat;
      toString(): string;
  }
  export class StatusResponse {
      idStats: DataLinkIdStats;
      datalinkStats: DataLinkStats;
      threadStats: Array<ThreadStat>;
      rawXml: string;
      constructor(idStats: DataLinkIdStats, datalinkStats: DataLinkStats, threadStats: Array<ThreadStat>);
      static fromDatalinkResponse(daliResp: DataLinkResponse): StatusResponse;
      static fromXML(daliXML: Element): StatusResponse;
      toString(): string;
  }
  export class StreamStat {
      name: string;
      earliestPacketID: number;
      earliestPacketDataStartTime: DateTime;
      earliestPacketDataEndTime: DateTime;
      latestPacketID: number;
      latestPacketDataStartTime: DateTime;
      latestPacketDataEndTime: DateTime;
      dataLatency: number;
      constructor(name: string, earliestPacketID: number, earliestPacketDataStartTime: DateTime, earliestPacketDataEndTime: DateTime, latestPacketID: number, latestPacketDataStartTime: DateTime, latestPacketDataEndTime: DateTime, dataLatency: number);
      static parseXMLAttributes(statusEl: Element): StreamStat;
      toString(): string;
  }
  export class StreamsResponse {
      datalinkStats: DataLinkStats;
      streams: Array<StreamStat>;
      constructor(datalinkStats: DataLinkStats, streams: Array<StreamStat>);
      static fromDatalinkResponse(daliResp: DataLinkResponse): StreamsResponse;
      static fromXML(daliXML: Element): StreamsResponse;
      toString(): string;
  }
  /**
   * Non implementation, just stores xml as a string. Unlikely
   * to be useful remotely as ringserver doesn't allow.
   *
   * @param daliXML  raw xml form server
   */
  export class ConnectionsResponse {
      daliXML: string;
      constructor(daliXML: string);
      static fromDatalinkResponse(daliResp: DataLinkResponse): ConnectionsResponse;
      static fromXML(daliXML: Element): ConnectionsResponse;
      toString(): string;
  }
  /**
   * Convert DataLink style dates, like "2022-10-04 15:11:24.786990"
   * to ISO form for DateTime
   *
   * @param  dalitime       datalink time
   * @returns   DateTime
   */
  export function daliDateTime(dalitime: string): DateTime;
  /**
   * Convert DateTime to a HPTime number.
   *
   * @param   m DateTime to convert
   * @returns  microseconds since epoch
   */
  export function dateTimeToHPTime(m: DateTime): number;
  /**
   * Convert hptime number to a DateTime.
   *
   * @param   hptime hptime to convert
   * @returns  DateTime in utc for the hptime
   */
  export function hpTimeToDateTime(hptime: number): DateTime;
  /**
   * Encode string into a Uint8Array.
   *
   * @param   dataString String to encode.
   * @returns             String as bytes in Uint8Array.
   */
  export function stringToUint8Array(dataString?: string): Uint8Array;

}
declare module 'seisplotjs/src/dataset' {
  import * as mseed3 from "seisplotjs/src/mseed3";
  import { Quake } from "seisplotjs/src/quakeml";
  import { Network } from "seisplotjs/src/stationxml";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import JSZip from "jszip";
  export const DATASET_DIR = "dataset";
  export const DOT_ZIP_EXT = ".zip";
  export const ZIP_FILENAME: string;
  export const SEISMOGRAM_DIR = "seismograms";
  export const CATALOG_FILE = "catalog.quakeml";
  export const INVENTORY_FILE = "inventory.staxml";
  export class Dataset {
      name: string;
      catalog: Array<Quake>;
      inventory: Array<Network>;
      waveforms: Array<SeismogramDisplayData>;
      processedWaveforms: Array<SeismogramDisplayData>;
      extra: Map<string, unknown>;
      constructor();
      saveToZipFile(filename?: string): Promise<void>;
      waveformsToMSeed3(): Map<string, ArrayBuffer>;
      merge(other: Dataset): Dataset;
      associateQuakes(timeOverlapSecs?: number): void;
      associateChannels(): void;
  }
  export function load(url: string): Promise<Dataset>;
  export function loadFromFile(file: File): Promise<Dataset>;
  export function loadFromZip(zip: JSZip): Promise<Dataset>;
  export function sddFromMSeed3(ms3records: Array<mseed3.MSeed3Record>, ds?: Dataset): Array<SeismogramDisplayData>;
  export function insertExtraHeaders(eh: Record<string, unknown>, sdd: SeismogramDisplayData, key: string, ds?: Dataset): void;
  export function createExtraHeaders(key: string, sdd: SeismogramDisplayData): Record<string, unknown>;

}
declare module 'seisplotjs/src/datechooser' {
  import { DateTime, Duration, Interval } from "luxon";
  export const HOURMIN_ELEMENT = "sp-hourmin";
  export const DATETIME_ELEMENT = "sp-datetime";
  export const TIMERANGE_ELEMENT = "sp-timerange";
  export const hourMinRegEx: RegExp;
  export const HOUR_MIN_24 = "HH:mm";
  export const START_LABEL = "startlabel";
  export const DEFAULT_START_LABEL = "Start:";
  export const END_LABEL = "endlabel";
  export const DEFAULT_END_LABEL = "End:";
  export const DUR_LABEL = "durLabel";
  export const DEFAULT_DUR_LABEL = "Dur:";
  /**
   * Hour and Minute chooser.
   * Use as '<sp-hourmin></sp-hourmin>'
   */
  export class HourMinChooser extends HTMLElement {
      _time: DateTime;
      updateCallback: (time: DateTime) => void;
      popupDiv: HTMLDivElement;
      constructor();
      /**
       * Shows or hides the popup based on current visibility style
       */
      showHide(): void;
      hide(): void;
      show(): void;
      /** @private */
      xxx_adjustPopupPosition(): void;
      /** @private */
      _adjustPopupPosition(): void;
      /**
       * Get hours and minutes as Duration instead of as a DateTime. Useful for
       * relative times.
       *
       * @returns hours, minutes as Duration
       */
      get asDuration(): Duration;
      get time(): DateTime;
      set time(dt: DateTime);
      _internalSetTime(dt: DateTime): void;
  }
  /**
   * Date and Time chooser using native date chooser and the above
   * HourMinChooser for the hour and minute of time.
   */
  export class DateTimeChooser extends HTMLElement {
      _time: DateTime;
      updateCallback: (time: DateTime) => void;
      hourMin: HourMinChooser;
      constructor(time?: DateTime);
      /**
       * Updates the time without triggering the callback function.
       *
       * @param  newTime new time to update sliders
       */
      updateTime(newTime: DateTime): void;
      /**
       * triggers the callback function.
       */
      timeModified(): void;
      get time(): DateTime;
      set time(dt: DateTime);
      /**
       * internal time set
       *
       * @private
       * @param  newTime new time to update
       */
      _internalSetTime(newTime: DateTime): void;
      attributeChangedCallback(name: string, oldValue: string, newValue: string): void;
      static get observedAttributes(): string[];
  }
  export const START_CHANGED = "start";
  export const END_CHANGED = "end";
  export const DURATION_CHANGED = "duration";
  /**
   * Combination of two DateTimeChoosers to specify a start and end time.
   * A "change" event is fired when the times are modified.
   *
   * Initial values can be set via the start, end and duration attributes.
   * Start and end are ISO8601 dates, duration may either be a number of
   * seconds or an ISO8601 duration string.
   * Also, if the prev-next attribute is true, then previous, next and now
   * buttons are added to shift the time range earlier, later or so that
   * the end is the current time.
   *
   * The component remembers the last changed, so if you modify duration
   * and then modify start, the end is adjusted to keep duration the same.
   *
   */
  export class TimeRangeChooser extends HTMLElement {
      updateCallback: (timerange: Interval) => void;
      _duration: Duration;
      startChooser: DateTimeChooser;
      endChooser: DateTimeChooser;
      _mostRecentChanged: string;
      constructor();
      toInterval(): Interval;
      getTimeRange(): Interval;
      /**
       * Updates the times without triggering the callback function.
       *
       * @param  timeRange new time interval
       */
      updateTimeRange(timeRange: Interval): void;
      get startLabel(): string;
      get endLabel(): string;
      get durationLabel(): string;
      get start(): DateTime;
      set start(time: DateTime | string);
      get end(): DateTime;
      set end(time: DateTime);
      set duration(duration: Duration);
      get duration(): Duration;
      _updateDuration(duration: Duration | string): void;
      resyncValues(curChanged: string): void;
      attributeChangedCallback(name: string, oldValue: string, newValue: string): void;
      static get observedAttributes(): string[];
  }
  /**
   * extracts duration from either string as ISO or number as seconds.
   *
   * @param  value               ISO string or number
   * @returns       duration
   */
  export function extractDuration(value: string): Duration;

}
declare module 'seisplotjs/src/distaz' {
  /** const for kilometers per degree on the earth, 111.19 */
  export const kmPerDeg = 111.19;
  /**
   * Convert degrees into kilometers along the earth surface
   *
   * @param deg degrees to convert
   * @returns kilometers
   */
  export function degtokm(deg: number): number;
  /**
   * Convert kilometers into degrees along the earth surface
   *
   * @param km kilometers to convert
   * @returns degrees
   */
  export function kmtodeg(km: number): number;
  export class DistAzOutput {
      delta: number;
      az: number;
      baz: number;
      stalat: number | undefined;
      stalon: number | undefined;
      evtlat: number | undefined;
      evtlon: number | undefined;
      constructor(delta: number, az: number, baz: number);
      get distance(): number;
      get distanceKm(): number;
      get distanceDeg(): number;
      get azimuth(): number;
      get backazimuth(): number;
  }
  /**
   *
   * Returns a simple object (DistAzOutput) with:
   *```
   *     delta       => Great Circle Arc distance in degrees
   *     az          => Azimuth of pt. 1 wrt pt. 2 in degrees
   *     baz         => Azimuth of pt. 2 wrt pt. 1 in degrees
   *```
   *
   * azimuth is if you stand at point 2 and measure angle between north
   *   and point 1. I.E. point 1 is the station and point 2 is the event.
   *
   * @param lat1 Latitude of first point (station) (+N, -S) in degrees
   * @param lon1 Longitude of first point(station) (+E, -W) in degrees
   * @param lat2 Latitude of second point (event)
   * @param lon2 Longitude of second point (event)
   * @returns delta, az, baz in a DistAzOutput
   */
  export function distaz(lat1: number, lon1: number, lat2: number, lon2: number): DistAzOutput;

}
declare module 'seisplotjs/src/fdsnavailability' {
  import { FDSNCommon, IRIS_HOST } from "seisplotjs/src/fdsncommon";
  import { DateTime, Interval } from "luxon";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  /** const for json format, json */
  export const FORMAT_JSON = "json";
  /** const for text format, text */
  export const FORMAT_TEXT = "text";
  /** const for geocsv format, geocsv */
  export const FORMAT_GEOCSV = "geocsv";
  /** const for request format, request */
  export const FORMAT_REQUEST = "request";
  /** const of completely empty json, \{\} */
  export const EMPTY_JSON: RootType;
  /**
   * Major version of the FDSN spec supported here.
   * Currently is 1.
   */
  export const SERVICE_VERSION = 1;
  /**
   * Service name as used in the FDSN DataCenters registry,
   * https://www.fdsn.org/datacenters
   */
  export const SERVICE_NAME = "fdsnws-availability-1";
  /** const for the default IRIS web service host, service.iris.edu */
  export { IRIS_HOST };
  /**
   * Query to a FDSN Availability web service.
   *
   * ```
   * const avail = AvailabilityQuery()
   *    .networkCode("CO")
   *    .stationCode("BIRD")
   *    .startTime("2021-12-27T19:18:54Z")
   *    .endTime("2021-12-27T19:22:54Z");
   * avail.query().then(sddList => {
   *   sddList.forEach(sdd => console.log(sdd))
   * });
   * ```
   *
   * @see https://www.fdsn.org/webservices/
   * @param host optional host to connect to, defaults to IRIS
   */
  export class AvailabilityQuery extends FDSNCommon {
      /** @private */
      _networkCode: string | undefined;
      /** @private */
      _stationCode: string | undefined;
      /** @private */
      _locationCode: string | undefined;
      /** @private */
      _channelCode: string | undefined;
      /** @private */
      _startTime: DateTime | undefined;
      /** @private */
      _endTime: DateTime | undefined;
      /** @private */
      _quality: string | undefined;
      /** @private */
      _merge: string | undefined;
      /** @private */
      _show: string | undefined;
      /** @private */
      _mergeGaps: number | undefined;
      /** @private */
      _limit: number | undefined;
      /** @private */
      _orderby: string | undefined;
      /** @private */
      _includerestricted: boolean | undefined;
      /** @private */
      _format: string | undefined;
      constructor(host?: string);
      /**
       * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
       *  Setting this is probably a bad idea as the code may not be compatible with
       *  the web service.
       *
       * @param value spec version, usually 1
       * @returns the query when setting, the current value when no argument
       */
      specVersion(value?: string): AvailabilityQuery;
      getSpecVersion(): string;
      /**
       * Gets/Sets the protocol, http or https. This should match the protocol
       *  of the page loaded, but is autocalculated and generally need not be set.
       *
       * @param value protocol, usually http or https
       * @returns the query when setting, the current value when no argument
       */
      protocol(value?: string): AvailabilityQuery;
      getProtocol(): string | undefined;
      /**
       * Gets/Sets the remote host to connect to.
       *
       * @param value host
       * @returns the query when setting, the current value when no argument
       */
      host(value?: string): AvailabilityQuery;
      getHost(): string;
      /**
       * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
       * the status code when no matching data is found by the service.
       *
       * @param value number for nodata, usually 404 or 204
       * @returns the query when setting, the current value when no argument
       */
      nodata(value?: number): AvailabilityQuery;
      getNodata(): number | undefined;
      /**
       * Gets/Sets the port, not usually set unless not on standard http or https ports
       *
       * @param value port
       * @returns the query when setting, the current value when no argument
       */
      port(value?: number): AvailabilityQuery;
      getPort(): number;
      /**
       * Gets/Sets the network code to check.
       *
       * @param value network code like IU
       * @returns the query when setting, the current value when no argument
       */
      networkCode(value?: string): AvailabilityQuery;
      getNetworkCode(): string | undefined;
      /**
       * Gets/Sets the station code to check.
       *
       * @param value station code like ANMO
       * @returns the query when setting, the current value when no argument
       */
      stationCode(value?: string): AvailabilityQuery;
      getStationCode(): string | undefined;
      /**
       * Gets/Sets the location code to check.
       *
       * @param value location code like 00
       * @returns the query when setting, the current value when no argument
       */
      locationCode(value?: string): AvailabilityQuery;
      getLocationCode(): string | undefined;
      /**
       * Gets/Sets the channel code to check.
       *
       * @param value channel code like BHZ
       * @returns the query when setting, the current value when no argument
       */
      channelCode(value?: string): AvailabilityQuery;
      getChannelCode(): string | undefined;
      /**
       * Gets/Sets the start time parameter for the query.
       *
       * @param value start time
       * @returns the query when setting, the current value when no argument
       */
      startTime(value?: DateTime | string): AvailabilityQuery;
      getStartTime(): DateTime | undefined;
      /**
       * Gets/Sets the end time parameter for the query.
       *
       * @param value end time
       * @returns the query when setting, the current value when no argument
       */
      endTime(value?: DateTime | string): AvailabilityQuery;
      getEndTime(): DateTime | undefined;
      /**
       * Sets startTime and endTime using the given time window
       *
       * @param   se time window
       * @returns    the query
       */
      timeRange(se: Interval): AvailabilityQuery;
      /**
       * Gets/Sets the quality parameter for the query.
       *
       * @param value quality
       * @returns the query when setting, the current value when no argument
       */
      quality(value?: string): AvailabilityQuery;
      getQuality(): string | undefined;
      /**
       * Gets/Sets the merge parameter for the query.
       *
       * @param value merge
       * @returns the query when setting, the current value when no argument
       */
      merge(value?: string): AvailabilityQuery;
      getMerge(): string | undefined;
      /**
       * Gets/Sets the mergegaps parameter for the query.
       *
       * @param value merge gaps
       * @returns the query when setting, the current value when no argument
       */
      mergeGaps(value?: number): AvailabilityQuery;
      getMergeGaps(): number | undefined;
      /**
       * Gets/Sets the show parameter for the query.
       *
       * @param value show
       * @returns the query when setting, the current value when no argument
       */
      show(value?: string): AvailabilityQuery;
      getShow(): string | undefined;
      /**
       * Gets/Sets the limit parameter for the query.
       *
       * @param value limit
       * @returns the query when setting, the current value when no argument
       */
      limit(value?: number): AvailabilityQuery;
      getLimit(): number | undefined;
      /**
       * Gets/Sets the order by parameter for the query.
       *
       * @param value order by
       * @returns the query when setting, the current value when no argument
       */
      orderby(value?: string): AvailabilityQuery;
      getOrderBy(): string | undefined;
      /**
       * Gets/Sets the include restricted data parameter for the query.
       *
       * @param value true to include restricted data
       * @returns the query when setting, the current value when no argument
       */
      includeRestricted(value?: boolean): AvailabilityQuery;
      getIncludeRestricted(): boolean | undefined;
      /**
       * Gets/Sets the format parameter for the query. Usually not needed as is set
       * by the various query methods.
       *
       * @param value format
       * @returns the query when setting, the current value when no argument
       */
      format(value?: string): AvailabilityQuery;
      getFormat(): string | undefined;
      /**
       * Get/Set the timeout in seconds for the request. Default is 30.
       *
       * @param value timeout in seconds
       * @returns the query when setting, the current value when no argument
       */
      timeout(value?: number): AvailabilityQuery;
      getTimeout(): number | undefined;
      /**
       * Calls query on the remote service, using configured parameters. Calls
       * queryJson internally, then unpacks the result into array of SeismogramDisplayData
       * objects.
       *
       * @returns          promise to array of SeismogramDisplayData, each representing
       * a channel-time window
       */
      query(): Promise<Array<SeismogramDisplayData>>;
      /**
       * Calls the query function the remote server and parses the returned data as json.
       *
       * @returns promise to the result as json
       */
      queryJson(): Promise<RootType>;
      /**
       * Calls extent on the remote service, using configured parameters. Calls
       * extentJson internally, then unpacks the result into array of SeismogramDisplayData
       * objects.
       *
       * @returns          promise to array of SeismogramDisplayData, each representing
       * a channel-time window
       */
      extent(): Promise<Array<SeismogramDisplayData>>;
      /**
       * Call the extend function on the remote server and parses the returned data as json.
       *
       * @returns promise to the result as json
       */
      extentJson(): Promise<RootType>;
      /**
       * Calls query on the remote service using POST, using configured parameters
       * and forms the POST body using the channelTimeList. Calls
       * postQueryJson internally, then unpacks the result into array of SeismogramDisplayData
       * objects.
       *
       * @param channelTimeList array of channel-time windows for the request
       * @returns          promise to array of SeismogramDisplayData, each representing
       * a channel-time window
       */
      postQuery(channelTimeList: Array<SeismogramDisplayData>): Promise<Array<SeismogramDisplayData>>;
      postExtent(channelTimeList: Array<SeismogramDisplayData>): Promise<Array<SeismogramDisplayData>>;
      postExtentJson(channelTimeList: Array<SeismogramDisplayData>): Promise<RootType>;
      postQueryJson(channelTimeList: Array<SeismogramDisplayData>): Promise<RootType>;
      postJson(channelTimeList: Array<SeismogramDisplayData>, method: string): Promise<RootType>;
      postRaw(channelTimeList: Array<SeismogramDisplayData>, method: string): Promise<Response>;
      extractFromJson(jsonChanTimes: RootType): Array<SeismogramDisplayData>;
      createPostBody(channelTimeList: Array<SeismogramDisplayData>): string;
      formBaseURL(): string;
      formVersionURL(): string;
      /**
       * Queries the remote web service to get its version
       *
       * @returns Promise to version string
       */
      queryVersion(): Promise<string>;
      makePostParm(name: string, val: unknown): string;
      formURL(method?: string): string;
  }
  /**
   * Root type of availablility json query.
   */
  export type RootType = {
      created?: FdsnDateTime;
      version: Record<string, unknown>;
      datasources: Array<Datasource>;
  } & Record<string, unknown>;
  export type FdsnDateTime = string;
  export type Datasource = ({
      network: string;
      station: string;
      location: string;
      channel: string;
      quality?: string;
      samplerate?: number;
      timespans?: Array<Array<FdsnDateTime>>;
      earliest?: FdsnDateTime;
      latest?: FdsnDateTime;
      updated?: FdsnDateTime;
      timespanCount?: number;
      restriction?: string;
  } & Record<string, unknown>) & (({
      timespans: unknown;
  } & Record<string, unknown>) | ({
      earliest: FdsnDateTime;
      latest: FdsnDateTime;
  } & Record<string, unknown>));
  export function isValidRootType(jsonValue: unknown): jsonValue is RootType;
  export function isValidDatasource(jsonValue: unknown): jsonValue is Datasource;

}
declare module 'seisplotjs/src/fdsncommon' {
  /**
   * commonalities with all types of FDSN Web Services
   */
  export const IRIS_HOST = "service.iris.edu";
  export class FDSNCommon {
      /** @private */
      _specVersion: string;
      /** @private */
      _protocol: string;
      /** @private */
      _host: string;
      /** @private */
      _port: number;
      /** @private */
      _nodata: number | undefined;
      /** @private */
      _timeoutSec: number;
      constructor(host?: string);
  }
  export class LatLonRegion {
  }
  export class LatLonBox extends LatLonRegion {
      west: number;
      east: number;
      south: number;
      north: number;
      constructor(west: number, east: number, south: number, north: number);
      asLeafletBounds(): [[number, number], [number, number]];
  }
  export class LatLonRadius extends LatLonRegion {
      latitude: number;
      longitude: number;
      minRadius: number;
      maxRadius: number;
      constructor(latitude: number, longitude: number, minRadius: number, maxRadius: number);
  }

}
declare module 'seisplotjs/src/fdsndatacenters' {
  import { FDSNCommon } from "seisplotjs/src/fdsncommon";
  import * as fdsnavailability from "seisplotjs/src/fdsnavailability";
  import * as fdsndataselect from "seisplotjs/src/fdsndataselect";
  import * as fdsnevent from "seisplotjs/src/fdsnevent";
  import * as fdsnstation from "seisplotjs/src/fdsnstation";
  /** const for fdsn web service host, www.fdsn.org */
  export const FDSN_HOST = "www.fdsn.org";
  /**
   * Query to a FDSN Data Centers Registry web service.
   *
   * @see https://www.fdsn.org/webservices/
   * @param host optional host to connect to, defaults to FDSN
   */
  export class DataCentersQuery extends FDSNCommon {
      /** @private */
      _name: string | undefined;
      /** @private */
      _services: string | undefined;
      /** @private */
      _includedatasets: boolean | undefined;
      constructor(host?: string);
      /**
       * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
       * Setting this is probably a bad idea as the code may not be compatible with
       * the web service.
       *
       * @param value optional new value if setting
       * @returns the query when setting, the current value os services if no arguments
       */
      specVersion(value?: string): DataCentersQuery;
      getSpecVersion(): string;
      /**
       * Gets/Sets the protocol, http or https. This should match the protocol
       * of the page loaded, but is autocalculated and generally need not be set.
       *
       * @param value optional new value if setting
       * @returns the query when setting, the current value os services if no arguments
       */
      protocol(value?: string): DataCentersQuery;
      getProtocol(): string | undefined;
      /**
       * Gets/Sets the remote host to connect to. This defaults to
       * www.fdsn.org and generally should not be set.
       *
       * @param value optional new value if setting
       * @returns the query when setting, the current value os services if no arguments
       */
      host(value?: string): DataCentersQuery;
      getHost(): string;
      /**
       * Gets/Sets the remote port to connect to. This defaults to
       * the standard port for the protocol and generally should not be set.
       *
       * @param value optional new value if setting
       * @returns the query when setting, the current value os services if no arguments
       */
      port(value?: number): DataCentersQuery;
      getPort(): number | undefined;
      /**
       * limits results to the named data center, default is all data centers
       *
       * @param   value names to search for
       * @returns the query when setting, the current value os services if no arguments
       */
      name(value?: string): DataCentersQuery;
      getName(): string | undefined;
      /**
       * limits results to services that match the glob style pattern
       *
       * @param  value glob style pattern to match against
       * @returns the query when setting, the current value os services if no arguments
       */
      services(value?: string): DataCentersQuery;
      getServices(): string | undefined;
      /**
       * whether the results include detailed information about
       * the data sets offered by each center, default is false
       *
       * @param  value true to include datasets
       * @returns the query when setting, the current value os services if no arguments
       */
      includeDataSets(value?: boolean): DataCentersQuery;
      getIncludeDataSets(): boolean | undefined;
      /**
       * Get/Set the timeout in seconds for the request. Default is 30.
       *
       * @param  value timeout seconds
       * @returns the query when setting, the current value os services if no arguments
       */
      timeout(value?: number): DataCentersQuery;
      getTimeout(): number | undefined;
      /**
       * queries the fdsn registry web service, returning the result as a parsed json object.
       *
       * @returns Promise to the json object.
       */
      queryJson(): Promise<RootType>;
      /**
       * queries the registry to find fdsn availability compatible web services within
       * a datacenter of the given name, optionally within the repository with
       * the repo name.
       *
       * @param   name     datacenter name
       * @param   repoName optional repository name
       * @returns           Promise to Array of fdsnavailability.AvailabilityQuery objects
       */
      findFdsnAvailability(name: string, repoName?: string): Promise<Array<fdsnavailability.AvailabilityQuery>>;
      /**
       * queries the registry to find fdsn dataselect compatible web services within
       * a datacenter of the given name, optionally within the repository with
       * the repo name.
       *
       * @param   name     datacenter name
       * @param   repoName optional repository name
       * @returns           Promise to Array of fdsndataselect.DataSelectQuery objects
       */
      findFdsnDataSelect(name: string, repoName?: string): Promise<Array<fdsndataselect.DataSelectQuery>>;
      /**
       * queries the registry to find a fdsn event compatible web services within
       * a datacenter of the given name, optionally within the repository with
       * the repo name.
       *
       * @param   dcname     datacenter name
       * @param   repoName optional repository name
       * @returns           Promise to Array of fdsnevent.EventQuery objects
       */
      findFdsnEvent(dcname: string, repoName?: string): Promise<Array<fdsnevent.EventQuery>>;
      /**
       * queries the registry to find a fdsn station compatible web services within
       * a datacenter of the given name, optionally within the repository with
       * the repo name.
       *
       * @param   dcname     datacenter name
       * @param   repoName optional repository name
       * @returns           Promise to Array of fdsnstation.StationQuery objects
       */
      findFdsnStation(dcname: string, repoName?: string): Promise<Array<fdsnstation.StationQuery>>;
      /**
       * Extracts services comaptible with the given service name, optionally within
       * the given repository, from the json.
       *
       * @param   json           json containing services
       * @param   compatibleName service name to be compatible with
       * @param   repoName       optional repository within the json to search
       * @returns                array of services found
       */
      extractCompatibleServices(json: RootType, compatibleName: string, repoName?: string): Array<Service>;
      /**
       * Forms the base of the url for accessing the datacenters service.
       *
       * @returns         URL
       */
      formBaseURL(): string;
      /**
       * Forms version url, not part of spec and so may not be supported.
       *
       * @returns         version
       */
      formVersionURL(): string;
      /**
       * Queries the remote web service to get its version
       *
       * @returns Promise to version string
       */
      queryVersion(): Promise<string>;
      /**
       * forms a url to the fdsn registry based on the configured parameters.
       *
       * @returns the url
       */
      formURL(): string;
  }
  /**
   * Root type of fdsn datacenters json query.
   */
  export type RootType = {
      version: Record<string, unknown>;
      datacenters: Array<{
          name: string;
          website: string;
          fullName?: string;
          summary?: string;
          repositories: Array<Repository>;
      }>;
  } & Record<string, unknown>;
  export type Repository = {
      name: string;
      description?: string;
      website?: string;
      services: Array<Service>;
      datasets?: Array<Dataset>;
  };
  export type Dataset = {
      network?: string;
      station?: string;
      location?: string;
      channel?: string;
      starttime?: string;
      endtime?: string;
      priority?: number;
      description?: string;
      url?: string;
      services?: Array<Service>;
  };
  export type Service = {
      name: string;
      description?: string;
      url?: string;
      compatibleWith?: string;
  };
  export function isValidRootType(jsonValue: unknown): jsonValue is RootType;

}
declare module 'seisplotjs/src/fdsndataselect' {
  import { FDSNCommon, IRIS_HOST } from "seisplotjs/src/fdsncommon";
  import { NslcId } from "seisplotjs/src/fdsnsourceid";
  import { DateTime, Interval } from "luxon";
  import * as miniseed from "seisplotjs/src/miniseed";
  import * as mseed3 from "seisplotjs/src/mseed3";
  import { Seismogram, SeismogramDisplayData } from "seisplotjs/src/seismogram";
  /** const for miniseed format, mseed */
  export const FORMAT_MINISEED = "miniseed";
  /** const for miniseed format, mseed */
  export const FORMAT_MINISEED_THREE = "miniseed3";
  /**
   * Major version of the FDSN spec supported here.
   * Currently is 1.
   */
  export const SERVICE_VERSION = 1;
  /**
   * Service name as used in the FDSN DataCenters registry,
   * https://www.fdsn.org/datacenters
   */
  export const SERVICE_NAME = "fdsnws-dataselect-1";
  /** const for the default IRIS web service host, service.iris.edu */
  export { IRIS_HOST };
  /**
   * Query to a FDSN Dataselect web service.
   *
   * @see https://www.fdsn.org/webservices/
   * @param host optional host to connect to, defaults to IRIS
   */
  export class DataSelectQuery extends FDSNCommon {
      /** @private */
      _networkCode: string | undefined;
      /** @private */
      _stationCode: string | undefined;
      /** @private */
      _locationCode: string | undefined;
      /** @private */
      _channelCode: string | undefined;
      /** @private */
      _startTime: DateTime | undefined;
      /** @private */
      _endTime: DateTime | undefined;
      /** @private */
      _quality: string | undefined;
      /** @private */
      _minimumLength: number | undefined;
      /** @private */
      _longestOnly: boolean | undefined;
      /** @private */
      _repository: string | undefined;
      /** @private */
      _format: string | undefined;
      constructor(host?: string);
      /**
       * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
       *  Setting this is probably a bad idea as the code may not be compatible with
       *  the web service.
       *
       * @param value spec version, usually 1
       * @returns new value if getting, this if setting
       */
      specVersion(value?: string): DataSelectQuery;
      getSpecVersion(): string;
      /**
       * Gets/Sets the protocol, http or https. This should match the protocol
       *  of the page loaded, but is autocalculated and generally need not be set.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      protocol(value?: string): DataSelectQuery;
      getProtocol(): string;
      /**
       * Gets/Sets the remote host to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      host(value?: string): DataSelectQuery;
      getHost(): string;
      /**
       * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
       * the status code when no matching data is found by the service.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      nodata(value?: number): DataSelectQuery;
      getNodata(): number | undefined;
      /**
       * Gets/Sets the remote port to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      port(value?: number): DataSelectQuery;
      getPort(): number | undefined;
      /**
       * Get/Set the network query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      networkCode(value?: string): DataSelectQuery;
      getNetworkCode(): string | undefined;
      /**
       * Get/Set the station query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      stationCode(value?: string): DataSelectQuery;
      getStationCode(): string | undefined;
      /**
       * Get/Set the location code query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      locationCode(value?: string): DataSelectQuery;
      getLocationCode(): string | undefined;
      /**
       * Get/Set the channel query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      channelCode(value?: string): DataSelectQuery;
      getChannelCode(): string | undefined;
      nslcCodes(channelId: NslcId): DataSelectQuery;
      /**
       * Get/Set the starttime query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      startTime(value?: DateTime): DataSelectQuery;
      getStartTime(): DateTime | undefined;
      /**
       * Get/Set the endtime query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      endTime(value?: DateTime): DataSelectQuery;
      getEndTime(): DateTime | undefined;
      /**
       * Sets startTime and endTime using the given time range
       *
       * @param   se time range
       * @returns     this
       */
      timeRange(se: Interval): DataSelectQuery;
      /**
       * Get/Set the quality query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      quality(value?: string): DataSelectQuery;
      getQuality(): string | undefined;
      /**
       * Get/Set the minimum length query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minimumLength(value?: number): DataSelectQuery;
      getMinimumLength(): number | undefined;
      /**
       * Get/Set the longest only query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      longestOnly(value?: boolean): DataSelectQuery;
      getLongestOnly(): boolean | undefined;
      /**
       * set or get the repository paramter. This is an IRIS-specific
       * parameter that will not work with other dataselect web services.
       *
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      repository(value?: string): DataSelectQuery;
      getRepository(): string | undefined;
      /**
       * Get/Set the format query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      format(value?: string): DataSelectQuery;
      getFormat(): string | undefined;
      /**
       * Get/Set the timeout in seconds for the request. Default is 30.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      timeout(value?: number): DataSelectQuery;
      getTimeout(): number | undefined;
      /**
       * queries the web service using the configured parameters, parsing the response
       * into miniseed data records.
       *
       * @returns Promise to Array of miniseed.DataRecords
       */
      queryDataRecords(): Promise<Array<miniseed.DataRecord>>;
      /**
       * queries the web service using the configured parameters, parsing the response
       * into miniseed data records.
       *
       * @returns Promise to Array of miniseed.DataRecords
       */
      queryMS3Records(): Promise<Array<mseed3.MSeed3Record>>;
      /**
       * queries the web service using the configured parameters, parsing the response
       * into miniseed data records and then combining the data records into
       * Seismogram objects.
       *
       * @returns Promise to Array of Seismogram objects
       */
      querySeismograms(): Promise<Array<Seismogram>>;
      postQueryDataRecords(channelTimeList: Array<SeismogramDisplayData>): Promise<Array<miniseed.DataRecord>>;
      postQueryMS3Records(channelTimeList: Array<SeismogramDisplayData>): Promise<Array<mseed3.MSeed3Record>>;
      /**
       * query the dataselect server using post, which allows for multiple
       * channel-timeranges at once. This assumes that there are not multiple
       * time ranges for the same channel as the results, encapsulated as
       * SeismogramDisplayData objects, are returned one seismogram
       * per channel, which may contain gaps. The original channel and timerange are
       * also populated with each result.
       *
       * @param   sddList array of SeismogramDisplayData objects
       * that will be filled in with the resulting seismogram
       * @returns Promise to the input Array of SeismogramDisplayData objects, each with the
       * seismogram containing the data returned from the server
       */
      postQuerySeismograms(sddList: Array<SeismogramDisplayData>): Promise<Array<SeismogramDisplayData>>;
      postQueryRaw(sddList: Array<SeismogramDisplayData>): Promise<Response>;
      postQueryRawWithBody(body: string): Promise<Response>;
      static createPostBody(sddList: Array<SeismogramDisplayData>): string;
      formBaseURL(): string;
      formVersionURL(): string;
      /**
       * Queries the remote web service to get its version
       *
       * @returns Promise to version string
       */
      queryVersion(): Promise<string>;
      formPostURL(): string;
      formURL(): string;
  }
  export function createDataSelectQuery(params: Record<string, string>): DataSelectQuery;

}
declare module 'seisplotjs/src/fdsnevent' {
  import { FDSNCommon, LatLonRegion } from "seisplotjs/src/fdsncommon";
  import { DateTime, Interval } from "luxon";
  import { EventParameters, Quake, USGS_HOST } from "seisplotjs/src/quakeml";
  /**
   * Major version of the FDSN spec supported here.
   * Currently is 1.
   */
  export const SERVICE_VERSION = 1;
  /**
   * Service name as used in the FDSN DataCenters registry,
   * https://www.fdsn.org/datacenters
   */
  export const SERVICE_NAME = "fdsnws-event-1";
  export { USGS_HOST };
  /**
   * Query to a FDSN Event web service.
   *
   * @see https://www.fdsn.org/webservices/
   * @param host optional host to connect to, defaults to USGS
   */
  export class EventQuery extends FDSNCommon {
      /** @private */
      _eventId: string | undefined;
      /** @private */
      _startTime: DateTime | undefined;
      /** @private */
      _endTime: DateTime | undefined;
      /** @private */
      _updatedAfter: DateTime | undefined;
      /** @private */
      _minMag: number | undefined;
      /** @private */
      _maxMag: number | undefined;
      /** @private */
      _magnitudeType: string | undefined;
      /** @private */
      _minDepth: number | undefined;
      /** @private */
      _maxDepth: number | undefined;
      /** @private */
      _minLat: number | undefined;
      /** @private */
      _maxLat: number | undefined;
      /** @private */
      _minLon: number | undefined;
      /** @private */
      _maxLon: number | undefined;
      /** @private */
      _latitude: number | undefined;
      /** @private */
      _longitude: number | undefined;
      /** @private */
      _minRadius: number | undefined;
      /** @private */
      _maxRadius: number | undefined;
      /** @private */
      _includeArrivals: boolean | undefined;
      /** @private */
      _includeAllOrigins: boolean | undefined;
      /** @private */
      _includeAllMagnitudes: boolean | undefined;
      /** @private */
      _limit: number | undefined;
      /** @private */
      _offset: number | undefined;
      /** @private */
      _orderBy: string | undefined;
      /** @private */
      _contributor: string | undefined;
      /** @private */
      _catalog: string | undefined;
      /** @private */
      _format: string | undefined;
      constructor(host?: string);
      /**
       * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
       *  Setting this is probably a bad idea as the code may not be compatible with
       *  the web service.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      specVersion(value?: string): EventQuery;
      getSpecVersion(): string;
      /**
       * Gets/Sets the protocol, http or https. This should match the protocol
       *  of the page loaded, but is autocalculated and generally need not be set.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      protocol(value?: string): EventQuery;
      getProtocol(): string | undefined;
      /**
       * Gets/Sets the remote host to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      host(value?: string): EventQuery;
      getHost(): string;
      /**
       * Gets/Sets the remote port to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      port(value?: number): EventQuery;
      getPort(): number | undefined;
      /**
       * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
       * the status code when no matching data is found by the service.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      nodata(value?: number): EventQuery;
      getNodata(): number | undefined;
      /**
       * Get/Set the timeout in seconds for the request. Default is 30.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      timeout(value?: number): EventQuery;
      getTimeout(): number | undefined;
      /**
       * Get/Set the eventid query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      eventId(value?: string): EventQuery;
      getEventId(): string | undefined;
      /**
       * Get/Set the starttime query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      startTime(value?: DateTime): EventQuery;
      getStartTime(): DateTime | undefined;
      /**
       * Get/Set the endtime query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      endTime(value?: DateTime): EventQuery;
      getEndTime(): DateTime | undefined;
      /**
       * Sets startTime and endTime using the given time window
       *
       * @param   se time window
       * @returns     this
       */
      timeRange(se: Interval): EventQuery;
      /**
       * Get/Set the updatedafter query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      updatedAfter(value?: DateTime): EventQuery;
      getUpdatedAfter(): DateTime | undefined;
      /**
       * Get/Set the minmag query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minMag(value?: number): EventQuery;
      getMinMag(): number | undefined;
      /**
       * Get/Set the maxmag query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxMag(value?: number): EventQuery;
      getMaxMag(): number | undefined;
      /**
       * Get/Set the magnitudetype query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      magnitudeType(value?: string): EventQuery;
      getMagnitudeType(): string | undefined;
      /**
       * Get/Set the mindepth query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minDepth(value?: number): EventQuery;
      getMinDepth(): number | undefined;
      /**
       * Get/Set the maxdepth query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxDepth(value?: number): EventQuery;
      getMaxDepth(): number | undefined;
      /**
       * Get/Set the minlat query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minLat(value?: number): EventQuery;
      getMinLat(): number | undefined;
      /**
       * Get/Set the maxlat query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxLat(value?: number): EventQuery;
      getMaxLat(): number | undefined;
      /**
       * Get/Set the minlon query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minLon(value?: number): EventQuery;
      getMinLon(): number | undefined;
      /**
       * Get/Set the maxlon query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxLon(value?: number): EventQuery;
      getMaxLon(): number | undefined;
      /**
       * Get/Set the latitude query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      latitude(value?: number): EventQuery;
      getLatitude(): number | undefined;
      /**
       * Get/Set the longitude query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      longitude(value?: number): EventQuery;
      getLongitude(): number | undefined;
      /**
       * Get/Set the minradius query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minRadius(value?: number): EventQuery;
      getMinRadius(): number | undefined;
      /**
       * Get/Set the maxradius query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxRadius(value?: number): EventQuery;
      getMaxRadius(): number | undefined;
      /**
       * Get/Set the includearrivals query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      includeArrivals(value?: boolean): EventQuery;
      getIncludeArrivals(): boolean | undefined;
      /**
       * Get/Set the includeallorigins query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      includeAllOrigins(value?: boolean): EventQuery;
      getIncludeAllOrigins(): boolean | undefined;
      /**
       * Get/Set the includeallmagnitudes query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      includeAllMagnitudes(value?: boolean): EventQuery;
      getIncludeAllMagnitudes(): boolean | undefined;
      /**
       * Get/Set the format query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      format(value?: string): EventQuery;
      getFormat(): string | undefined;
      /**
       * Get/Set the limit query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      limit(value?: number): EventQuery;
      getLimit(): number | undefined;
      /**
       * Get/Set the offset query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      offset(value?: number): EventQuery;
      getOffset(): number | undefined;
      /**
       * Get/Set the orderby query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      orderBy(value?: string): EventQuery;
      getOrderBy(): string | undefined;
      /**
       * Get/Set the catalog query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      catalog(value?: string): EventQuery;
      getCatalog(): string | undefined;
      /**
       * Get/Set the contributor query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      contributor(value?: string): EventQuery;
      getContributor(): string | undefined;
      latLonRegion(value: LatLonRegion | null): EventQuery;
      /**
       * Checks to see if any parameter that would limit the data
       * returned is set. This is a crude, coarse check to make sure
       * the client doesn't ask for EVERYTHING the server has.
       *
       * @returns true is some parameter is set
       */
      isSomeParameterSet(): boolean;
      /**
       * Queries the remote service and parses the returned xml.
       *
       *  @returns Promise to an Array of Quake objects.
       */
      query(): Promise<Array<Quake>>;
      queryEventParameters(): Promise<EventParameters>;
      /**
       * Queries the remote server, to get QuakeML xml.
       *
       * @returns xml Document
       */
      queryRawXml(): Promise<Document>;
      /**
       * Forms the basic URL to contact the web service, without any query paramters
       *
       * @returns the url
       */
      formBaseURL(): string;
      /**
       * Forms the URL to get catalogs from the web service, without any query paramters
       *
       * @returns the url
       */
      formCatalogsURL(): string;
      /**
       * Queries the remote web service to get known catalogs
       *
       * @returns Promise to Array of catalog names
       */
      queryCatalogs(): Promise<Array<string>>;
      /**
       * Forms the URL to get contributors from the web service, without any query paramters
       *
       * @returns the url
       */
      formContributorsURL(): string;
      /**
       * Queries the remote web service to get known contributors
       *
       * @returns Promise to Array of contributor names
       */
      queryContributors(): Promise<Array<string>>;
      /**
       * Forms the URL to get version from the web service, without any query paramters
       *
       * @returns the url
       */
      formVersionURL(): string;
      /**
       * Queries the remote web service to get its version
       *
       * @returns Promise to version string
       */
      queryVersion(): Promise<string>;
      /**
       * Form URL to query the remote web service, encoding the query parameters.
       *
       * @returns url
       */
      formURL(): string;
  }

}
declare module 'seisplotjs/src/fdsneventcomponent' {
  import { EventQuery } from "seisplotjs/src/fdsnevent";
  export class EarthquakeSearch extends HTMLElement {
      constructor();
      _registerEvent(wrapper: HTMLElement, sel: string): void;
      draw_element(shadow: ShadowRoot): void;
      populateQuery(query?: EventQuery): EventQuery;
  }

}
declare module 'seisplotjs/src/fdsnsourceid' {
  export const FDSN_PREFIX = "FDSN:";
  export const SEP = "_";
  export class FDSNSourceId {
      networkCode: string;
      stationCode: string;
      locationCode: string;
      bandCode: string;
      sourceCode: string;
      subsourceCode: string;
      constructor(networkCode: string, stationCode: string, locationCode: string, bandCode: string, sourceCode: string, subsourceCode: string);
      static createUnknown(sampRate?: number, source?: string, subsource?: string): FDSNSourceId;
      static parse(id: string): FDSNSourceId;
      static fromNslc(net: string, sta: string, loc: string, channelCode: string): FDSNSourceId;
      static fromNslcId(nslcId: NslcId): FDSNSourceId;
      static parseNslc(nslc: string, sep?: string): FDSNSourceId;
      stationSourceId(): StationSourceId;
      networkSourceId(): NetworkSourceId;
      asNslc(): NslcId;
      /**
       * returns a channel code. If this is an old style NSLC, it will be 3 chars,
       * but if either source or subsouce is more than one char, it will be
       * three fields delimited by underscores.
       *
       * @returns the channel code part of the id
       */
      formChannelCode(): string;
      toString(): string;
      toStringNoPrefix(): string;
      equals(other: FDSNSourceId | null): boolean;
      clone(): FDSNSourceId;
  }
  export class NetworkSourceId {
      networkCode: string;
      constructor(networkCode: string);
      static parse(id: string): NetworkSourceId;
      toString(): string;
      equals(other: NetworkSourceId): boolean;
  }
  export class StationSourceId {
      networkCode: string;
      stationCode: string;
      constructor(networkCode: string, stationCode: string);
      static parse(id: string): StationSourceId;
      toString(): string;
      networkSourceId(): NetworkSourceId;
      equals(other: StationSourceId): boolean;
  }
  export class LocationSourceId {
      networkCode: string;
      stationCode: string;
      locationCode: string;
      constructor(networkCode: string, stationCode: string, locationCode: string);
      toString(): string;
      equals(other: LocationSourceId): boolean;
  }
  /**
   * Generates the best band code for a channel based on the sample rate and
   * optionally the response lower bound period, which is mostly useful for
   * separating broadband from short period seismometers.
   *
   * @param  sampRate  sample rate in samples per second
   * @param  resp_lb   response long period bound in seconds
   * @returns          single character band code
   */
  export function bandCodeForRate(sampRate?: number, resp_lb?: number): string;
  export const EMPTY_LOC_CODE = "--";
  export class NslcId {
      networkCode: string;
      stationCode: string;
      locationCode: string;
      channelCode: string;
      constructor(net: string, sta: string, loc: string, chan: string);
      static parse(nslc: string, sep?: string): NslcId;
      toString(): string;
      equals(other: NslcId): boolean;
  }
  export function parseSourceId(id: string): FDSNSourceId | NetworkSourceId | StationSourceId;
  export function SourceIdSorter(aSid: FDSNSourceId, bSid: FDSNSourceId): number;

}
declare module 'seisplotjs/src/fdsnstation' {
  import { FDSNCommon, IRIS_HOST, LatLonRegion } from "seisplotjs/src/fdsncommon";
  import { NslcId } from "seisplotjs/src/fdsnsourceid";
  import { DateTime, Interval } from "luxon";
  import { Network } from "seisplotjs/src/stationxml";
  export const LEVEL_NETWORK = "network";
  export const LEVEL_STATION = "station";
  export const LEVEL_CHANNEL = "channel";
  export const LEVEL_RESPONSE = "response";
  export const LEVELS: string[];
  /**
   * Major version of the FDSN spec supported here.
   * Currently is 1.
   */
  export const SERVICE_VERSION = 1;
  /**
   * Service name as used in the FDSN DataCenters registry,
   * https://www.fdsn.org/datacenters
   */
  export const SERVICE_NAME = "fdsnws-station-1";
  /** const for the default IRIS web service host, service.iris.edu */
  export { IRIS_HOST };
  /**
   * Query to a FDSN Station web service.
   *
   * @see https://www.fdsn.org/webservices/
   * @param host optional host to connect to, defaults to IRIS
   */
  export class StationQuery extends FDSNCommon {
      /** @private */
      _networkCode: string | undefined;
      /** @private */
      _stationCode: string | undefined;
      /** @private */
      _locationCode: string | undefined;
      /** @private */
      _channelCode: string | undefined;
      /** @private */
      _startTime: DateTime | undefined;
      /** @private */
      _endTime: DateTime | undefined;
      /** @private */
      _startBefore: DateTime | undefined;
      /** @private */
      _endBefore: DateTime | undefined;
      /** @private */
      _startAfter: DateTime | undefined;
      /** @private */
      _endAfter: DateTime | undefined;
      /** @private */
      _minLat: number | undefined;
      /** @private */
      _maxLat: number | undefined;
      /** @private */
      _minLon: number | undefined;
      /** @private */
      _maxLon: number | undefined;
      /** @private */
      _latitude: number | undefined;
      /** @private */
      _longitude: number | undefined;
      /** @private */
      _minRadius: number | undefined;
      /** @private */
      _maxRadius: number | undefined;
      /** @private */
      _includeRestricted: boolean | undefined;
      /** @private */
      _includeAvailability: boolean | undefined;
      /** @private */
      _format: string | undefined;
      /** @private */
      _updatedAfter: DateTime | undefined;
      /** @private */
      _matchTimeseries: boolean | undefined;
      /**
       * Construct a query
       *
       * @param host the host to connect to , defaults to service.iris.edu
       */
      constructor(host?: string);
      /**
       * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
       * Setting this is probably a bad idea as the code may not be compatible with
       * the web service.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      specVersion(value?: string): StationQuery;
      getSpecVersion(): string;
      /**
       * Gets/Sets the protocol, http or https. This should match the protocol
       * of the page loaded, but is autocalculated and generally need not be set.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      protocol(value?: string): StationQuery;
      getProtocol(): string | undefined;
      /**
       * Gets/Sets the remote host to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      host(value?: string): StationQuery;
      getHost(): string | undefined;
      /**
       * Gets/Sets the remote port to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      port(value?: number): StationQuery;
      getPort(): number | undefined;
      /**
       * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
       * the status code when no matching data is found by the service.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      nodata(value?: number): StationQuery;
      getNodata(): number | undefined;
      /**
       * Get/Set the network query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      networkCode(value?: string): StationQuery;
      getNetworkCode(): string | undefined;
      /**
       * Get/Set the station query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      stationCode(value?: string): StationQuery;
      getStationCode(): string | undefined;
      /**
       * Get/Set the location code query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      locationCode(value?: string): StationQuery;
      getLocationCode(): string | undefined;
      /**
       * Get/Set the channel query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      channelCode(value?: string): StationQuery;
      getChannelCode(): string | undefined;
      /**
       * Get/Set the starttime query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      startTime(value?: DateTime): StationQuery;
      getStartTime(): DateTime | undefined;
      /**
       * Get/Set the endtime query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      endTime(value?: DateTime): StationQuery;
      getEndTime(): DateTime | undefined;
      /**
       * Sets startTime and endTime using the given time window
       *
       * @param   se time window
       * @returns     this
       */
      timeRange(se: Interval): StationQuery;
      /**
       * Get/Set the startbefore query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      startBefore(value?: DateTime): StationQuery;
      getStartBefore(): DateTime | undefined;
      /**
       * Get/Set the endbefore query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      endBefore(value?: DateTime): StationQuery;
      getEndBefore(): DateTime | undefined;
      /**
       * Get/Set the startafter query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      startAfter(value?: DateTime): StationQuery;
      getStartAfter(): DateTime | undefined;
      /**
       * Get/Set the endafter query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      endAfter(value?: DateTime): StationQuery;
      getEndAfter(): DateTime | undefined;
      /**
       * Get/Set the minlat query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minLat(value?: number): StationQuery;
      getMinLat(): number | undefined;
      /**
       * Get/Set the maxlon query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxLat(value?: number): StationQuery;
      getMaxLat(): number | undefined;
      /**
       * Get/Set the minlon query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minLon(value?: number): StationQuery;
      getMinLon(): number | undefined;
      /**
       * Get/Set the maxlon query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxLon(value?: number): StationQuery;
      getMaxLon(): number | undefined;
      /**
       * Get/Set the latitude query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      latitude(value?: number): StationQuery;
      getLatitude(): number | undefined;
      /**
       * Get/Set the longitude query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      longitude(value?: number): StationQuery;
      getLongitude(): number | undefined;
      /**
       * Get/Set the minradius query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minRadius(value?: number): StationQuery;
      getMinRadius(): number | undefined;
      /**
       * Get/Set the maxradius query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxRadius(value?: number): StationQuery;
      getMaxRadius(): number | undefined;
      /**
       * Get/Set the includerestricted query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      includeRestricted(value?: boolean): StationQuery;
      getIncludeRestricted(): boolean | undefined;
      /**
       * Get/Set the includeavailability query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      includeAvailability(value?: boolean): StationQuery;
      getIncludeAvailability(): boolean | undefined;
      /**
       * Get/Set the format query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      format(value?: string): StationQuery;
      getFormat(): string | undefined;
      /**
       * Get/Set the updatedafter query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      updatedAfter(value?: DateTime): StationQuery;
      getUpdatedAfter(): DateTime | undefined;
      /**
       * Get/Set the matchtimeseries query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      matchTimeseries(value?: boolean): StationQuery;
      getMatchTimeseries(): boolean | undefined;
      latLonRegion(value: LatLonRegion | null): StationQuery;
      nslcCodes(channelId: NslcId): StationQuery;
      /**
       * Get/Set the timeout in seconds for the request. Default is 30.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      timeout(value?: number): StationQuery;
      getTimeout(): number | undefined;
      /**
       * Checks to see if any parameter that would limit the data
       * returned is set. This is a crude, coarse check to make sure
       * the client doesn't ask for EVERYTHING the server has.
       *
       * @returns true if some parameter set
       */
      isSomeParameterSet(): boolean;
      /**
       * Queries the remote web service for networks.
       *
       * @returns a Promise to an Array of Network objects.
       */
      queryNetworks(): Promise<Array<Network>>;
      /**
       * Queries the remote web service for stations. The stations
       * are contained within their respective Networks.
       *
       * @returns a Promise to an Array of Network objects.
       */
      queryStations(): Promise<Array<Network>>;
      /**
       * Queries the remote web service for channels. The Channels
       * are contained within their respective Stations which are in Networks.
       *
       * @returns a Promise to an Array of Network objects.
       */
      queryChannels(): Promise<Array<Network>>;
      /**
       * Queries the remote web service for responses. The Responses
       * are contained within their respective Channels,
       * which are in Stations which are in Networks.
       *
       * @returns a Promise to an Array of Network objects.
       */
      queryResponses(): Promise<Array<Network>>;
      /**
       * Queries the remote web service at the given level.
       *
       * @param level the level to query at, networ, station, channel or response.
       * @returns a Promise to an Array of Network objects.
       */
      query(level: string): Promise<Array<Network>>;
      /**
       * Execute POST request for networks, using params defined in this, and with
       * channel lines of the form:
       *
       * NET STA LOC CHA STARTTIME ENDTIME
       *
       * Note that empty LOC should be encoded as dash-dash
       *
       * @param  postLines array of channel selection lines
       * @returns a Promise to an Array of Network objects.
       */
      postQueryNetworks(postLines: Array<string>): Promise<Array<Network>>;
      /**
       * Execute POST request for stations, using params defined in this, and with
       * channel lines of the form:
       *
       * NET STA LOC CHA STARTTIME ENDTIME
       *
       * Note that empty LOC should be encoded as dash-dash
       *
       * @param  postLines array of channel selection lines
       * @returns a Promise to an Array of Network objects.
       */
      postQueryStations(postLines: Array<string>): Promise<Array<Network>>;
      /**
       * Execute POST request for channels, using params defined in this, and with
       * channel lines of the form:
       *
       * NET STA LOC CHA STARTTIME ENDTIME
       *
       * Note that empty LOC should be encoded as dash-dash
       *
       * @param  postLines array of channel selection lines
       * @returns a Promise to an Array of Network objects.
       */
      postQueryChannels(postLines: Array<string>): Promise<Array<Network>>;
      /**
       * Execute POST request for responses, using params defined in this, and with
       * channel lines of the form:
       *
       * NET STA LOC CHA STARTTIME ENDTIME
       *
       * Note that empty LOC should be encoded as dash-dash
       *
       * @param  postLines array of channel selection lines
       * @returns a Promise to an Array of Network objects.
       */
      postQueryResponses(postLines: Array<string>): Promise<Array<Network>>;
      /**
       * Execute POST request using params defined in this, for given level, and with
       * channel lines of the form:
       *
       * NET STA LOC CHA STARTTIME ENDTIME
       *
       * Note that empty LOC should be encoded as dash-dash
       *
       * @param  level     level to request, one of network, station, channel, response
       * @param  postLines array of channel selection lines
       * @returns a Promise to an Array of Network objects.
       */
      postQuery(level: string, postLines: Array<string>): Promise<Array<Network>>;
      /**
       * Queries the remote web service at the given level for raw xml.
       * Note that in the case of a nodata status code, xml that represents a
       * valid stationxml but with zero &lt;Network&gt; elements will be returned
       * as this simplifies parsing.
       *
       * @param level the level to query at, network, station, channel or response.
       * @returns a Promise to an xml Document.
       */
      queryRawXml(level: string): Promise<Document>;
      /**
       * Queries the remote web service at the given level for unparsed xml as text.
       * Note that in the case of a nodata status code, text that represents a
       * valid stationxml but with zero &lt;Network&gt; elements will be returned
       * as this simplifies parsing.
       *
       * @param level the level to query at, network, station, channel or response.
       * @returns a Promise to string.
       */
      queryRawXmlText(level: string): Promise<string>;
      /**
       * Execute POST request using params defined in this, for given level, and with
       * channel lines of the form:
       *
       * NET STA LOC CHA STARTTIME ENDTIME
       *
       * Note that empty LOC should be encoded as dash-dash
       *
       * @param  level     level to request, one of network, station, channel, response
       * @param  postLines array of channel selection lines
       * @returns           string suitable for POST to fdsn station web service.
       */
      postQueryRawXml(level: string, postLines: Array<string>): Promise<Document>;
      /**
       * Creates post body using params defined in this, for given level, and with
       * optional channel lines of the form:
       *
       * NET STA LOC CHA STARTTIME ENDTIME
       *
       * Note that empty LOC should be encoded as dash-dash
       *
       * @param  level     level to request, one of network, station, channel, response
       * @param  postLines optional array of channel selection lines
       * @returns           string suitable for POST to fdsn station web service.
       */
      createPostBody(level: string, postLines?: Array<string>): string;
      /**
       * Forms the URL to get version from the web service, without any query paramters
       *
       * @returns the url
       */
      formVersionURL(): string;
      /**
       * Queries the remote web service to get its version
       *
       * @returns Promise to version string
       */
      queryVersion(): Promise<string>;
      /**
       * Forms the basic URL to contact the web service, without any query paramters
       *
       * @returns the url
       */
      formBaseURL(): string;
      formPostURL(): string;
      /**
       * Form URL to query the remote web service, encoding the query parameters.
       *
       * @param level network, station, channel or response
       * @returns url
       */
      formURL(level: string): string;
  }

}
declare module 'seisplotjs/src/fdsnstationcomponent' {
  import { StationQuery } from "seisplotjs/src/fdsnstation";
  import { LatLonChoice } from "seisplotjs/src/components";
  export class ChannelSearch extends HTMLElement {
      constructor();
      _registerEvent(wrapper: HTMLElement, sel: string): void;
      draw_element(shadow: ShadowRoot): void;
      populateQuery(query?: StationQuery): StationQuery;
      getGeoChoiceElement(): LatLonChoice;
  }
  export const CHANNEL_SEARCH_ELEMENT = "sp-channel-search";

}
declare module 'seisplotjs/src/fft' {
  import { Seismogram, SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { Complex } from "seisplotjs/src/oregondsputil";
  /**
   * A higher level function to calculate DFT. Returns a
   * FFTResult for easier access to the result as
   * complex, amp, phase arrays. Calls calcDFT internally.
   * Inverse FFT is available as FFTResult.fftInverse().
   *
   * @param seis seismogram or SeismogramDisplayData to transform
   * @returns fft of seismogram
   */
  export function fftForward(seis: Seismogram | SeismogramDisplayData): FFTResult;
  /**
   * Calculates the discrete fourier transform using the OregonDSP library.
   *
   * This is a lower level function, fftForward is better for most uses.
   *
   * @param   timeseries timeseries array
   * @returns           DFT as packed array Float32Array
   */
  export function calcDFT(timeseries: Int32Array | Float32Array | Float64Array): Float32Array;
  /**
   * Calculates the inverse discrete fourier transform using the oregondsp library.
   *
   * @param   packedFreq DFT as packed array Float32Array
   * @param   numPoints     number of points in original timeseries array.
   * @returns           inverse of DFT as a timeseries array
   */
  export function inverseDFT(packedFreq: Float32Array, numPoints: number): Float32Array;
  /**
   * Finds smallest power of two >= input number.
   *
   * @param  fftlength               input number
   * @returns           tuple of N and log2N, like [16,4]
   */
  export function findPowerTwo(fftlength: number): [number, number];
  /**
   * Results of FFT calculation. Allows convertion of the packed real/imag array output from calcDFT into
   * amplitude and phase.
   */
  export class FFTResult {
      /** number of points in the original timeseries, may be less than fft size. */
      origLength: number;
      packedFreq: Float32Array;
      /** number of points in the fft, usually power of 2 larger than origLength. */
      numPoints: number;
      /** sample rate of the original time series, maybe be null. */
      sampleRate: number;
      /** optional units of the original data for display purposes. */
      inputUnits: string | undefined;
      /**
       * optional reference to SeismogramDisplayData when calculated from a seismogram.
       *  Useful for creating title, etc.
       */
      seismogramDisplayData: SeismogramDisplayData | undefined;
      constructor(origLength: number, sampleRate: number);
      /**
       * Factory method to create FFTResult from packed array.
       *
       * @param   packedFreq real and imag values in packed format
       * @param   origLength length of the original timeseries before padding.
       * @param   sampleRate sample rate of original data
       * @returns            FFTResult
       */
      static createFromPackedFreq(packedFreq: Float32Array, origLength: number, sampleRate: number): FFTResult;
      /**
       * Factory method to create from array of complex numbers.
       *
       * @param   complexArray real and imag values as array of Complex objects.
       * @param   origLength   length of the original timeseries before padding.
       * @param   sampleRate sample rate of original data
       * @returns               FFTResult
       */
      static createFromComplex(complexArray: Array<InstanceType<typeof Complex>>, origLength: number, sampleRate: number): FFTResult;
      /**
       * Factory method to create from amp and phase arrays
       *
       * @param   amp        amplitude values
       * @param   phase      phase values
       * @param   origLength length of the original timeseries before padding.
       * @param   sampleRate sample rate of original data
       * @returns             FFTResult
       */
      static createFromAmpPhase(amp: Float32Array, phase: Float32Array, origLength: number, sampleRate: number): FFTResult;
      /**
       * The minimum non-zero frequency in the fft
       *
       * @returns fundamental frequency
       */
      get fundamentalFrequency(): number;
      asComplex(): Array<InstanceType<typeof Complex>>;
      asAmpPhase(): [Float32Array, Float32Array];
      /**
       * calculates the inverse fft of this.packedFreq
       *
       * @returns time domain representation
       */
      fftInverse(): Float32Array;
      frequencies(): Float32Array;
      get numFrequencies(): number;
      get minFrequency(): number;
      get maxFrequency(): number;
      amplitudes(): Float32Array;
      phases(): Float32Array;
      clone(): FFTResult;
  }

}
declare module 'seisplotjs/src/filter' {
  import { DateTime } from "luxon";
  import { Seismogram } from "seisplotjs/src/seismogram";
  import { InstrumentSensitivity } from "seisplotjs/src/stationxml";
  import { Butterworth, ChebyshevI, ChebyshevII, IIRFilter, LOWPASS, BANDPASS, HIGHPASS } from "seisplotjs/src/oregondsputil";
  /**
   * Constant for bandpass OregonDSP filter creation.
   */
  export const BAND_PASS = "BANDPASS";
  /**
   * Constant for lowpass OregonDSP filter creation.
   */
  export const LOW_PASS = "LOWPASS";
  /**
   * Constant for highpass OregonDSP filter creation.
   */
  export const HIGH_PASS = "HIGHPASS";
  export function amplitude(real: number, imag: number): number;
  /**
   * Remove the mean from a seismogram. Subtract the mean from each data point.
   *
   * @param   seis input seismogram
   * @returns       seismogram with mean of zero
   */
  export function rMean(seis: Seismogram): Seismogram;
  export type LineFitType = {
      slope: number;
      intercept: number;
      reference_time: DateTime;
      sigma: number;
      sigma_a: number;
      sigma_b: number;
      correlation: number;
  };
  /**
   * Calculate best fit line to seismogram. Limited to contiguous data currently.
   * Code derived from scm/lifite.c in SAC.
   * Original version from Steve Taylor.
   *
   * @param  seis                        [description]
   * @param  referenceTime               [description]
   * @returns               best fit line
   */
  export function lineFit(seis: Seismogram, referenceTime?: DateTime): LineFitType;
  /**
   * Returns a new Seismogram with the trend removed by
   * subtracting the trend line from each data point.
   *
   * @param   seis input seismogram
   * @param fitLine optional fit type
   * @returns       seismogram with mean of zero and best fit line horizontal
   */
  export function removeTrend(seis: Seismogram, fitLine?: LineFitType): Seismogram;
  /**
   * Apply the frequency independent overall gain to a seismogram. This does not
   * do a full transfer using poles and zero, this only applies the scalar conversion
   * factor to convert counts back to original real world units and update the units.
   *
   * @param   seis                  the seismogram to correct
   * @param   instrumentSensitivity overall gain object, usually pulled from stationxml
   * @returns                        new seismogram with original units, like m/s and gain applied.
   */
  export function gainCorrect(seis: Seismogram, instrumentSensitivity: InstrumentSensitivity): Seismogram;
  export function mul(seis: Seismogram, factor: number): Seismogram;
  export function add(seis: Seismogram, factor: number): Seismogram;
  export function getPassband(type: string): typeof LOWPASS | typeof BANDPASS | typeof HIGHPASS;
  /**
   * Creates a Butterworth IIR filter using the OregonDSP library.
   *
   * @param   numPoles       number of poles
   * @param   passband       type, use constants of BAND_PASS, LOW_PASS, HIGH_PASS
   * @param   lowFreqCorner  low corner frequency
   * @param   highFreqCorner high corner frequency
   * @param   delta          delta, period, of timeseries
   * @returns                 Butterworth IIR filter
   */
  export function createButterworth(numPoles: number, passband: string, lowFreqCorner: number, highFreqCorner: number, delta: number): InstanceType<typeof Butterworth>;
  /**
   * Creates a Chebyshev I IIR filter using the OregonDSP library.
   *
   * @param   numPoles       number of poles
   * @param   epsilon        Chebyshev epsilon value
   * @param   passband       type, use constants of BAND_PASS, LOW_PASS, HIGH_PASS
   * @param   lowFreqCorner  low corner frequency
   * @param   highFreqCorner high corner frequency
   * @param   delta          delta, period, of timeseries
   * @returns                 Chebyshev I IIR filter
   */
  export function createChebyshevI(numPoles: number, epsilon: number, passband: string, lowFreqCorner: number, highFreqCorner: number, delta: number): InstanceType<typeof ChebyshevI>;
  /**
   * Creates a Chebyshev II IIR filter using the OregonDSP library.
   *
   * @param   numPoles       number of poles
   * @param   epsilon        Chebyshev epsilon value
   * @param   passband       type, use constants of BAND_PASS, LOW_PASS, HIGH_PASS
   * @param   lowFreqCorner  low corner frequency
   * @param   highFreqCorner high corner frequency
   * @param   delta          delta, period, of timeseries
   * @returns                 Chebyshev II IIR filter
   */
  export function createChebyshevII(numPoles: number, epsilon: number, passband: string, lowFreqCorner: number, highFreqCorner: number, delta: number): InstanceType<typeof ChebyshevII>;
  /**
   * Applies the filter to the given seismogram.
   *
   * @param   iirFilter filter to apply
   * @param   seis      seismogram to apply filter to
   * @returns            filtered seismogram
   */
  export function applyFilter(iirFilter: InstanceType<typeof IIRFilter>, seis: Seismogram): Seismogram;
  /**
   * Calculates the envelope, y_i = sqrt( y_i * y_i + h_i * h_i)
   *  where h is the hilber transform of y. The default configuration
   *  for the hilbet transform is n=100, lowEdge=.05 and highEdge = 0.95
   *
   * @param seis seismogram to apply envelope to
   * @returns seismogram cloned but with data as the envelope
   */
  export function envelope(seis: Seismogram): Seismogram;
  /**
   * Calculates the hilbert transform using the OregonDSP library
   *  with default number of points, n=10 (to yield a 21 pt FIR transform)
   *  and default low and high edge of 0.05 and 0.95. Low and high edge are
   *  given normalized 0 to 1.
   *
   * Note this uses Float32Array, other array types will be converted,
   * possibly losing precision.
   *
   * @param seis seismogram to calculate from
   * @param n optional number of points in transform, default is 10
   * @param lowEdge low edge of filter, normailized to 0-1, default is 0.05
   * @param highEdge high edge of filter, normailized to 0-1, default is 0.95
   * @returns hilbert transformed data
   */
  export function hilbert(seis: Seismogram, n?: number, lowEdge?: number, highEdge?: number): Seismogram;
  /**
   * Differentiate a seismogram.
   *
   * @param   seis input seismogram
   * @returns       differentiated seismogram
   */
  export function differentiate(seis: Seismogram): Seismogram;
  /**
   * Integrate a seismogram.
   *
   * @param   seis input seismogram
   * @param   integrationConst integration constant, optional, defaults to 0
   * @returns       integrated seismogram
   */
  export function integrate(seis: Seismogram, integrationConst?: number): Seismogram;

}
declare module 'seisplotjs/src/handlebarshelpers' {
  import * as Handlebars from "handlebars";
  export { Handlebars };
  /**
   * Registers Handlebars helpers like onlyChangesChannel and other formatters
   *
   */
  export function registerHelpers(): void;

}
declare module 'seisplotjs/src/helicorder' {
  import { DateTime, Duration, Interval, Zone } from "luxon";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { SeismogramSegment } from "seisplotjs/src/seismogramsegment";
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  import { SeisPlotElement } from "seisplotjs/src/spelement";
  export const HELICORDER_ELEMENT = "sp-helicorder";
  /**
   * A helicorder-like multi-line seismogram display usually covering 24 hours
   *
   * @param inSvgParent the parent element, usually a div tag
   * @param heliConfig configuration object
   * @param seisData the data to display
   */
  export class Helicorder extends SeisPlotElement {
      constructor(seisData?: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      get heliConfig(): HelicorderConfig;
      set heliConfig(config: HelicorderConfig);
      get width(): number;
      get height(): number;
      appendSegment(segment: SeismogramSegment): void;
      /**
       * draws the helicorder.
       */
      draw(): void;
      /**
       * draws or redraws the seismograms in the helicorder
       *
       * @private
       */
      drawSeismograms(): void;
      cutForLine(singleSeisData: SeismogramDisplayData, lineInterval: Interval): SeismogramDisplayData;
      /**
       * Calculates the time range covered by each line of the display
       *
       * @param   startTime      start of display
       * @param   secondsPerLine seconds covered by each line
       * @param   numberOfLines  number of lines
       * @returns Array of HeliTimeRange, one per line
       */
      calcTimesForLines(startTime: DateTime, secondsPerLine: number, numberOfLines: number): Array<HeliTimeRange>;
      calcDetailForEvent(evt: MouseEvent): HeliMouseEventType;
  }
  export const DEFAULT_MAX_HEIGHT = 600;
  /**
   * Configuration of the helicorder
   *
   * Note that setting maxVariation=0 and fixedAmplitudeScale=[0,0] will scale the
   * data to max
   *
   * @param timeRange the time range covered by the helicorder, required
   */
  export class HelicorderConfig extends SeismographConfig {
      lineSeisConfig: SeismographConfig;
      overlap: number;
      numLines: number;
      maxVariation: number;
      detrendLines: boolean;
      yLabelTimeZone: string | Zone<boolean>;
      yLabelRightTimeZone: string | Zone<boolean>;
      constructor(timeRange: Interval);
      static fromSeismographConfig(seisConfig: SeismographConfig): HelicorderConfig;
  }
  /**
   * Time range for a single line of the helicorder, extends Interval
   * to add the line number
   */
  export class HeliTimeRange {
      lineNumber: number;
      interval: Interval;
      constructor(startTime: DateTime, duration: Duration, lineNumber: number);
  }
  export function nameForTimeZone(zone: string | null | Zone): string;
  /** default styling for helicorder plots. */
  export const helicorder_css = "\n:host {\n  display: block;\n  min-height: 200px;\n  height: 100%;\n  cursor: crosshair;\n}\n";
  export const HELICORDER_SELECTOR = "helicorder";
  export const HELI_COLOR_CSS_ID = "helicordercolors";
  export type HeliMouseEventType = {
      mouseevent: MouseEvent;
      time: DateTime;
      lineNum: number;
  };

}
declare module 'seisplotjs/src/index' {
  import * as animatedseismograph from "seisplotjs/src/animatedseismograph";
  import * as axisutil from "seisplotjs/src/axisutil";
  import * as components from "seisplotjs/src/components";
  import * as cssutil from "seisplotjs/src/cssutil";
  import * as datalink from "seisplotjs/src/datalink";
  import * as dataset from "seisplotjs/src/dataset";
  import * as datechooser from "seisplotjs/src/datechooser";
  import * as distaz from "seisplotjs/src/distaz";
  import * as fdsnavailability from "seisplotjs/src/fdsnavailability";
  import * as fdsncommon from "seisplotjs/src/fdsncommon";
  import * as fdsndatacenters from "seisplotjs/src/fdsndatacenters";
  import * as fdsnevent from "seisplotjs/src/fdsnevent";
  import * as fdsneventcomponent from "seisplotjs/src/fdsneventcomponent";
  import * as fdsnstation from "seisplotjs/src/fdsnstation";
  import * as fdsnstationcomponent from "seisplotjs/src/fdsnstationcomponent";
  import * as fdsndataselect from "seisplotjs/src/fdsndataselect";
  import * as fdsnsourceid from "seisplotjs/src/fdsnsourceid";
  import * as filter from "seisplotjs/src/filter";
  import * as fft from "seisplotjs/src/fft";
  import * as spectraplot from "seisplotjs/src/spectraplot";
  import * as handlebarshelpers from "seisplotjs/src/handlebarshelpers";
  import * as helicorder from "seisplotjs/src/helicorder";
  import * as infotable from "seisplotjs/src/infotable";
  import * as leafletutil from "seisplotjs/src/leafletutil";
  import * as miniseed from "seisplotjs/src/miniseed";
  import * as mseed3 from "seisplotjs/src/mseed3";
  import * as mseed3eh from "seisplotjs/src/mseed3eh";
  import * as mseedarchive from "seisplotjs/src/mseedarchive";
  import * as oregondsputil from "seisplotjs/src/oregondsputil";
  import * as organizeddisplay from "seisplotjs/src/organizeddisplay";
  import * as particlemotion from "seisplotjs/src/particlemotion";
  import * as quakeml from "seisplotjs/src/quakeml";
  import * as ringserverweb from "seisplotjs/src/ringserverweb";
  import * as sacPoleZero from "seisplotjs/src/sacpolezero";
  import * as scale from "seisplotjs/src/scale";
  import * as seedcodec from "seisplotjs/src/seedcodec";
  import * as seedlink from "seisplotjs/src/seedlink";
  import * as seedlink4 from "seisplotjs/src/seedlink4";
  import * as seismogram from "seisplotjs/src/seismogram";
  import * as seismogramloader from "seisplotjs/src/seismogramloader";
  import * as seismogramsegment from "seisplotjs/src/seismogramsegment";
  import * as seismograph from "seisplotjs/src/seismograph";
  import * as seismographmarker from "seisplotjs/src/seismographmarker";
  import * as seismographutil from "seisplotjs/src/seismographutil";
  import * as seismographconfig from "seisplotjs/src/seismographconfig";
  import * as seismographconfigeditor from "seisplotjs/src/seismographconfigeditor";
  import * as sorting from "seisplotjs/src/sorting";
  import * as stationxml from "seisplotjs/src/stationxml";
  import * as taper from "seisplotjs/src/taper";
  import * as transfer from "seisplotjs/src/transfer";
  import * as traveltime from "seisplotjs/src/traveltime";
  import * as usgsgeojson from "seisplotjs/src/usgsgeojson";
  import * as util from "seisplotjs/src/util";
  import { version } from "seisplotjs/src/version";
  import * as vector from "seisplotjs/src/vector";
  import * as OregonDSP from "oregondsp";
  import * as luxon from "luxon";
  import * as leaflet from "leaflet";
  export { animatedseismograph, axisutil, components, cssutil, datalink, dataset, datechooser, distaz, infotable, fdsnavailability, fdsncommon, fdsndatacenters, fdsnevent, fdsneventcomponent, fdsnstation, fdsnstationcomponent, fdsndataselect, fdsnsourceid, fft, handlebarshelpers, helicorder, filter, leafletutil, miniseed, mseed3, mseed3eh, mseedarchive, oregondsputil, organizeddisplay, particlemotion, quakeml, ringserverweb, sacPoleZero, scale, seedcodec, seedlink, seedlink4, seismogram, seismogramloader, seismogramsegment, seismograph, seismographmarker, seismographutil, seismographconfig, seismographconfigeditor, sorting, spectraplot, stationxml, taper, transfer, traveltime, usgsgeojson, util, vector, version, OregonDSP, leaflet, luxon, };

}
declare module 'seisplotjs/src/index_node' {
  import * as datalink from "seisplotjs/src/datalink";
  import * as dataset from "seisplotjs/src/dataset";
  import * as distaz from "seisplotjs/src/distaz";
  import * as fdsnavailability from "seisplotjs/src/fdsnavailability";
  import * as fdsncommon from "seisplotjs/src/fdsncommon";
  import * as fdsndatacenters from "seisplotjs/src/fdsndatacenters";
  import * as fdsnevent from "seisplotjs/src/fdsnevent";
  import * as fdsnstation from "seisplotjs/src/fdsnstation";
  import * as fdsndataselect from "seisplotjs/src/fdsndataselect";
  import * as fdsnsourceid from "seisplotjs/src/fdsnsourceid";
  import * as filter from "seisplotjs/src/filter";
  import * as fft from "seisplotjs/src/fft";
  import * as miniseed from "seisplotjs/src/miniseed";
  import * as mseed3 from "seisplotjs/src/mseed3";
  import * as mseedarchive from "seisplotjs/src/mseedarchive";
  import * as oregondsputil from "seisplotjs/src/oregondsputil";
  import * as quakeml from "seisplotjs/src/quakeml";
  import * as ringserverweb from "seisplotjs/src/ringserverweb";
  import * as sacPoleZero from "seisplotjs/src/sacpolezero";
  import * as scale from "seisplotjs/src/scale";
  import * as seedcodec from "seisplotjs/src/seedcodec";
  import * as seedlink from "seisplotjs/src/seedlink";
  import * as seedlink4 from "seisplotjs/src/seedlink4";
  import * as seismogram from "seisplotjs/src/seismogram";
  import * as seismogramloader from "seisplotjs/src/seismogramloader";
  import * as seismogramsegment from "seisplotjs/src/seismogramsegment";
  import * as sorting from "seisplotjs/src/sorting";
  import * as stationxml from "seisplotjs/src/stationxml";
  import * as taper from "seisplotjs/src/taper";
  import * as transfer from "seisplotjs/src/transfer";
  import * as traveltime from "seisplotjs/src/traveltime";
  import * as usgsgeojson from "seisplotjs/src/usgsgeojson";
  import * as util from "seisplotjs/src/util";
  import { version } from "seisplotjs/src/version";
  import * as vector from "seisplotjs/src/vector";
  import * as OregonDSP from "oregondsp";
  import * as luxon from "luxon";
  const axisutil: null;
  const animatedseismograph: null;
  const components: null;
  const cssutil: null;
  const datechooser: null;
  const infotable: null;
  const fdsneventcomponent: null;
  const fdsnstationcomponent: null;
  const handlebarshelpers: null;
  const helicorder: null;
  const organizeddisplay: null;
  const particlemotion: null;
  const seismograph: null;
  const seismographconfig: null;
  const seismographutil: null;
  const seismographconfigeditor: null;
  const spectraplot: null;
  const leaflet: null;
  const leafletutil: null;
  export { animatedseismograph, axisutil, components, cssutil, datalink, dataset, datechooser, distaz, infotable, fdsnavailability, fdsncommon, fdsndatacenters, fdsnevent, fdsneventcomponent, fdsnstation, fdsnstationcomponent, fdsndataselect, fdsnsourceid, fft, handlebarshelpers, helicorder, filter, leafletutil, miniseed, mseed3, mseedarchive, oregondsputil, organizeddisplay, particlemotion, quakeml, ringserverweb, sacPoleZero, scale, seedcodec, seedlink, seedlink4, seismogram, seismogramloader, seismogramsegment, seismograph, seismographutil, seismographconfig, seismographconfigeditor, sorting, spectraplot, stationxml, taper, transfer, traveltime, usgsgeojson, util, vector, version, OregonDSP, leaflet, luxon, };

}
declare module 'seisplotjs/src/infotable' {
  import { Quake } from "seisplotjs/src/quakeml";
  import { Channel, Station } from "seisplotjs/src/stationxml";
  import { SeisPlotElement } from "seisplotjs/src/spelement";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  export const INFO_ELEMENT = "sp-station-quake-table";
  export const QUAKE_INFO_ELEMENT = "sp-quake-table";
  export enum QUAKE_COLUMN {
      LAT = "Lat",
      LON = "Lon",
      TIME = "Time",
      MAG = "Mag",
      MAGTYPE = "MagType",
      DEPTH = "Depth",
      DESC = "Description",
      EVENTID = "EventId"
  }
  export enum CHANNEL_COLUMN {
      LAT = "Lat",
      LON = "Lon",
      AZIMUTH = "Az",
      DIP = "Dip",
      START = "Start",
      END = "End",
      ELEVATION = "Elev",
      DEPTH = "Depth",
      SOURCEID = "SourceId",
      CODE = "Code",
      NETWORK_CODE = "NetworkCode",
      STATION_CODE = "StationCode",
      LOCATION_CODE = "LocationCode",
      CHANNEL_CODE = "ChannelCode"
  }
  export enum STATION_COLUMN {
      LAT = "Lat",
      LON = "Lon",
      START = "Start",
      END = "End",
      ELEVATION = "Elev",
      SOURCEID = "SourceId",
      CODE = "Code",
      NETWORK_CODE = "NetworkCode",
      STATION_CODE = "StationCode",
      DESCRIPTION = "Description"
  }
  export enum SEISMOGRAM_COLUMN {
      START = "Start",
      DURATION = "Duration",
      END = "End",
      NUM_POINTS = "Num Pts",
      SAMPLE_RATE = "Sample Rate",
      SAMPLE_PERIOD = "Sample Period",
      SEGMENTS = "Segments",
      SOURCEID = "SourceId",
      CODE = "Codes",
      NETWORK_CODE = "NetworkCode",
      STATION_CODE = "StationCode"
  }
  export const DEFAULT_TEMPLATE = "\n  <table>\n  <thead>\n      <tr>\n        <th colspan=\"7\">Waveform</th>\n        <th colspan=\"4\">Channel</th>\n        <th colspan=\"6\">Event</th>\n        <th colspan=\"4\">DistAz</th>\n      </tr>\n      <tr>\n        <th>Codes</th>\n        <th>Start</th>\n        <th>Duration</th>\n        <th>End</th>\n        <th>Num Pts</th>\n        <th>Sample Rate</th>\n        <th>YUnit</th>\n        <th>Seg</th>\n\n        <th>Lat</th>\n        <th>Lon</th>\n        <th>Elev</th>\n        <th>Depth</th>\n\n        <th>Time</th>\n        <th>Lat</th>\n        <th>Lon</th>\n        <th colspan=\"2\">Mag</th>\n        <th>Depth</th>\n\n        <th>Dist deg</th>\n        <th>Dist km</th>\n        <th>Azimuth</th>\n        <th>Back Azimuth</th>\n      </tr>\n    </thead>\n    <tbody>\n    {{#each seisDataList as |sdd|}}\n      <tr>\n        <td>{{sdd.nslc}}</td>\n        <td>{{formatIsoDate sdd.seismogram.startTime}}</td>\n        <td>{{formatDuration sdd.seismogram.timeRange}}</td>\n        <td>{{formatIsoDate sdd.seismogram.endTime}}</td>\n        <td>{{sdd.seismogram.numPoints}}</td>\n        <td>{{sdd.seismogram.sampleRate}}</td>\n        <td>{{sdd.seismogram.yUnit}}</td>\n        <td>{{sdd.seismogram.segments.length}}</td>\n\n        {{#if sdd.channel}}\n          <td>{{sdd.channel.latitude}}</td>\n          <td>{{sdd.channel.longitude}}</td>\n          <td>{{sdd.channel.elevation}}</td>\n          <td>{{sdd.channel.depth}}</td>\n        {{else}}\n          <td>no channel</td>\n          <td/>\n          <td/>\n          <td/>\n        {{/if}}\n\n        {{#if sdd.quake }}\n          <td>{{formatIsoDate sdd.quake.time}}</td>\n          <td>{{sdd.quake.latitude}}</td>\n          <td>{{sdd.quake.longitude}}</td>\n          <td>{{sdd.quake.magnitude.mag}}</td>\n          <td>{{sdd.quake.magnitude.type}}</td>\n          <td>{{sdd.quake.depthKm}}</td>\n        {{else}}\n          <td>no quake</td>\n          <td/>\n          <td/>\n          <td/>\n          <td/>\n          <td/>\n        {{/if}}\n        {{#if sdd.quake }}\n          {{#if sdd.channel }}\n            <td>{{formatNumber sdd.distaz.distanceDeg 2}}</td>\n            <td>{{formatNumber sdd.distaz.distanceKm 0}}</td>\n            <td>{{formatNumber sdd.distaz.az 2}}</td>\n            <td>{{formatNumber sdd.distaz.baz 2}}</td>\n          {{/if}}\n        {{/if}}\n      </tr>\n    {{/each}}\n    </tbody>\n  </table>\n";
  export const TABLE_CSS = "\ntbody tr:nth-child(even)\n{\n  background: var(--even-row-background, Cornsilk);\n}\ntbody tr:nth-child(odd)\n{\n  background: var(--odd-row-background);\n}\ntable {\n    display: block;\n    overflow-x: auto;\n    white-space: nowrap;\n}\n";
  /**
   * Table displaying information about waveforms, quakes, channels and stations.
   *
   * The CSS vars --even-row-background and --odd-row-background will change
   * the color of even and odd rows. Default for odd is nothing, even is Cornsilk.
   *
   * @param seisData    Array of SeismogramDisplayData for display
   * @param seisConfig  configuration
   */
  export class QuakeStationTable extends SeisPlotElement {
      _template: string;
      constructor(seisData?: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      get template(): string;
      set template(t: string);
      draw(): void;
  }
  export class QuakeTable extends HTMLElement {
      _columnLabels: Map<QUAKE_COLUMN, string>;
      _quakeList: Array<Quake>;
      _rowToQuake: Map<HTMLTableRowElement, Quake>;
      lastSortAsc: boolean;
      lastSortCol: QUAKE_COLUMN | undefined;
      constructor(quakeList?: Array<Quake>, columnLabels?: Map<QUAKE_COLUMN, string>);
      get quakeList(): Array<Quake>;
      set quakeList(ql: Array<Quake>);
      get columnLabels(): Map<QUAKE_COLUMN, string>;
      set columnLabels(cols: Map<QUAKE_COLUMN, string>);
      addStyle(css: string, id?: string): HTMLStyleElement;
      findRowForQuake(q: Quake): HTMLTableRowElement | null;
      draw(): void;
      headers(): Array<QUAKE_COLUMN>;
      populateRow(q: Quake, row: HTMLTableRowElement, index: number): void;
      static getQuakeValue(q: Quake, h: QUAKE_COLUMN): string;
      sort(h: QUAKE_COLUMN, _headerCell: HTMLTableCellElement): void;
  }
  export class ChannelTable extends HTMLElement {
      _columnLabels: Map<CHANNEL_COLUMN, string>;
      _channelList: Array<Channel>;
      _rowToChannel: Map<HTMLTableRowElement, Channel>;
      lastSortAsc: boolean;
      lastSortCol: CHANNEL_COLUMN | undefined;
      constructor(channelList?: Array<Channel>, columnLabels?: Map<CHANNEL_COLUMN, string>);
      get channelList(): Array<Channel>;
      set channelList(ql: Array<Channel>);
      get columnLabels(): Map<CHANNEL_COLUMN, string>;
      set columnLabels(cols: Map<CHANNEL_COLUMN, string>);
      addStyle(css: string, id?: string): HTMLStyleElement;
      draw(): void;
      headers(): Array<CHANNEL_COLUMN>;
      populateRow(q: Channel, row: HTMLTableRowElement, index: number): void;
      static getChannelValue(q: Channel, h: CHANNEL_COLUMN): string;
      sort(h: CHANNEL_COLUMN, _headerCell: HTMLTableCellElement): void;
  }
  export const CHANNEL_INFO_ELEMENT = "sp-channel-table";
  export class StationTable extends HTMLElement {
      _columnLabels: Map<STATION_COLUMN, string>;
      _stationList: Array<Station>;
      _rowToStation: Map<HTMLTableRowElement, Station>;
      lastSortAsc: boolean;
      lastSortCol: STATION_COLUMN | undefined;
      constructor(stationList?: Array<Station>, columnLabels?: Map<STATION_COLUMN, string>);
      get stationList(): Array<Station>;
      set stationList(ql: Array<Station>);
      get columnLabels(): Map<STATION_COLUMN, string>;
      set columnLabels(cols: Map<STATION_COLUMN, string>);
      addStyle(css: string, id?: string): HTMLStyleElement;
      draw(): void;
      headers(): Array<STATION_COLUMN>;
      populateRow(q: Station, row: HTMLTableRowElement, index: number): void;
      static getStationValue(q: Station, h: STATION_COLUMN): string;
      sort(h: STATION_COLUMN, _headerCell: HTMLTableCellElement): void;
  }
  export const STATION_INFO_ELEMENT = "sp-station-table";
  export class SeismogramTable extends HTMLElement {
      _columnLabels: Map<SEISMOGRAM_COLUMN, string>;
      _sddList: Array<SeismogramDisplayData>;
      _rowToSDD: Map<HTMLTableRowElement, SeismogramDisplayData>;
      lastSortAsc: boolean;
      lastSortCol: SEISMOGRAM_COLUMN | undefined;
      constructor(sddList?: Array<SeismogramDisplayData>, columnLabels?: Map<SEISMOGRAM_COLUMN, string>);
      get seisData(): Array<SeismogramDisplayData>;
      set seisData(ql: Array<SeismogramDisplayData>);
      get columnLabels(): Map<SEISMOGRAM_COLUMN, string>;
      set columnLabels(cols: Map<SEISMOGRAM_COLUMN, string>);
      addStyle(css: string, id?: string): HTMLStyleElement;
      draw(): void;
      headers(): Array<SEISMOGRAM_COLUMN>;
      populateRow(q: SeismogramDisplayData, row: HTMLTableRowElement, index: number): void;
      static getSeismogramValue(q: SeismogramDisplayData, h: SEISMOGRAM_COLUMN): string;
      sort(h: SEISMOGRAM_COLUMN, _headerCell: HTMLTableCellElement): void;
  }
  export const SDD_INFO_ELEMENT = "sp-seismogram-table";
  export const latlonFormat: Intl.NumberFormat;
  export const magFormat: Intl.NumberFormat;
  export const depthFormat: Intl.NumberFormat;
  export const depthMeterFormat: Intl.NumberFormat;

}
declare module 'seisplotjs/src/irisfedcatalog' {
  import { FDSNCommon } from "seisplotjs/src/fdsncommon";
  import { DateTime, Interval } from "luxon";
  import { Network } from "seisplotjs/src/stationxml";
  import { StationQuery } from "seisplotjs/src/fdsnstation";
  import { NslcId } from "seisplotjs/src/fdsnsourceid";
  import { DataSelectQuery } from "seisplotjs/src/fdsndataselect";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  /**
   * Major version of the IRIS web service supported here.
   * Currently is 1.
   */
  export const SERVICE_VERSION = 1;
  /**
   * Service name as used in the FDSN DataCenters registry,
   * https://www.fdsn.org/datacenters
   */
  export const SERVICE_NAME = "irisws-fedcatalog-1";
  export const IRIS_HOST = "service.iris.edu";
  export const TARGET_DATASELECT = "dataselect";
  export const TARGET_STATION = "station";
  /** a fake, completely empty stationxml document in case of no data. */
  export const FAKE_EMPTY_TEXT = "\n";
  /**
   * Represents the part of the results of a fedcatalog query for a single
   * datacenter, a section of the larger FedCatalogResult.
   *
   */
  export class FedCatalogDataCenter {
      dataCenter: string;
      services: Map<string, string>;
      stationService: string;
      dataSelectService: string;
      postLines: Array<string>;
      stationQuery: StationQuery | null;
      dataSelectQuery: DataSelectQuery | null;
      level: string;
      constructor();
      /**
       * Uses the response from the FedCat server to make the actual FDSNStation
       * query that returns StationXML. If the original FedCat query did not return
       * a Station service, or it was not asked for, then the array will be empty.
       *
       * @returns promise to networks
       */
      queryNetworkList(): Promise<Array<Network>>;
      queryStationRawXml(): Promise<Document>;
      querySDDList(): Promise<Array<SeismogramDisplayData>>;
  }
  /**
   * Represents the results of a fedcatalog query.
   *
   */
  export class FedCatalogResult {
      params: Map<string, string>;
      queries: Array<FedCatalogDataCenter>;
      constructor();
  }
  /**
   * Query to a IRIS FedCatalog web service.
   *
   * @see https://service.iris.edu/irisws/fedcatalog/1/
   * @param host optional host to connect to, defaults to IRIS
   */
  export class FedCatalogQuery extends FDSNCommon {
      /** @private */
      _targetService: string | undefined;
      /** @private */
      _level: string | undefined;
      /** @private */
      _networkCode: string | undefined;
      /** @private */
      _stationCode: string | undefined;
      /** @private */
      _locationCode: string | undefined;
      /** @private */
      _channelCode: string | undefined;
      /** @private */
      _startTime: DateTime | undefined;
      /** @private */
      _endTime: DateTime | undefined;
      /** @private */
      _startBefore: DateTime | undefined;
      /** @private */
      _endBefore: DateTime | undefined;
      /** @private */
      _startAfter: DateTime | undefined;
      /** @private */
      _endAfter: DateTime | undefined;
      /** @private */
      _minLat: number | undefined;
      /** @private */
      _maxLat: number | undefined;
      /** @private */
      _minLon: number | undefined;
      /** @private */
      _maxLon: number | undefined;
      /** @private */
      _latitude: number | undefined;
      /** @private */
      _longitude: number | undefined;
      /** @private */
      _minRadius: number | undefined;
      /** @private */
      _maxRadius: number | undefined;
      /** @private */
      _includeRestricted: boolean | undefined;
      /** @private */
      _includeAvailability: boolean | undefined;
      /** @private */
      _format: string | undefined;
      /** @private */
      _updatedAfter: DateTime | undefined;
      /** @private */
      _matchTimeseries: boolean | undefined;
      fedCatResult: Promise<FedCatalogResult> | null;
      /**
       * Construct a query
       *
       * @param host the host to connect to , defaults to service.iris.edu
       */
      constructor(host?: string);
      /**
       * Constructs a station FedCatalogQuery using the parameters in a StationQuery.
       *
       * @param   stationQuery query to pull parameters from
       * @returns               fedcatalog query
       */
      static fromStationQuery(stationQuery: StationQuery): FedCatalogQuery;
      /**
       * Gets/Sets the version of the fdsnws spec, 1 is currently the only value.
       *  Setting this is probably a bad idea as the code may not be compatible with
       *  the web service.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      specVersion(value?: string): FedCatalogQuery;
      getSpecVersion(): string;
      /**
       * Gets/Sets the protocol, http or https. This should match the protocol
       *  of the page loaded, but is autocalculated and generally need not be set.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      protocol(value?: string): FedCatalogQuery;
      getProtocol(): string | undefined;
      /**
       * Gets/Sets the remote host to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      host(value?: string): FedCatalogQuery;
      getHost(): string | undefined;
      /**
       * Gets/Sets the remote port to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      port(value?: number): FedCatalogQuery;
      getPort(): number | undefined;
      /**
       * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
       * the status code when no matching data is found by the service.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      nodata(value?: number): FedCatalogQuery;
      getNodata(): number | undefined;
      /**
       * Get/Set the targetservice query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      targetService(value?: string): FedCatalogQuery;
      getTargetService(): string | undefined;
      /**
       * Get/Set the network query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      networkCode(value?: string): FedCatalogQuery;
      getNetworkCode(): string | undefined;
      /**
       * Get/Set the station query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      stationCode(value?: string): FedCatalogQuery;
      getStationCode(): string | undefined;
      /**
       * Get/Set the location code query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      locationCode(value?: string): FedCatalogQuery;
      getLocationCode(): string | undefined;
      /**
       * Get/Set the channel query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      channelCode(value?: string): FedCatalogQuery;
      getChannelCode(): string | undefined;
      nslcCodes(channelId: NslcId): FedCatalogQuery;
      /**
       * Get/Set the starttime query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      startTime(value?: DateTime | string): FedCatalogQuery;
      getStartTime(): DateTime | undefined;
      /**
       * Get/Set the endtime query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      endTime(value?: DateTime | string): FedCatalogQuery;
      getEndTime(): DateTime | undefined;
      /**
       * Sets startTime and endTime using the given time window
       *
       * @param   se time window
       * @returns     this
       */
      timeRange(se: Interval): FedCatalogQuery;
      /**
       * Get/Set the startbefore query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      startBefore(value?: DateTime | string): FedCatalogQuery;
      getStartBefore(): DateTime | undefined;
      /**
       * Get/Set the endbefore query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      endBefore(value?: DateTime | string): FedCatalogQuery;
      getEndBefore(): DateTime | undefined;
      /**
       * Get/Set the startafter query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      startAfter(value?: DateTime | string): FedCatalogQuery;
      getStartAfter(): DateTime | undefined;
      /**
       * Get/Set the endafter query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      endAfter(value?: DateTime | string): FedCatalogQuery;
      getEndAfter(): DateTime | undefined;
      /**
       * Get/Set the minlat query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minLat(value?: number): FedCatalogQuery;
      getMinLat(): number | undefined;
      /**
       * Get/Set the maxlon query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxLat(value?: number): FedCatalogQuery;
      getMaxLat(): number | undefined;
      /**
       * Get/Set the minlon query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minLon(value?: number): FedCatalogQuery;
      getMinLon(): number | undefined;
      /**
       * Get/Set the maxlon query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxLon(value?: number): FedCatalogQuery;
      getMaxLon(): number | undefined;
      /**
       * Get/Set the latitude query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      latitude(value?: number): FedCatalogQuery;
      getLatitude(): number | undefined;
      /**
       * Get/Set the longitude query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      longitude(value?: number): FedCatalogQuery;
      getLongitude(): number | undefined;
      /**
       * Get/Set the minradius query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      minRadius(value?: number): FedCatalogQuery;
      getMinRadius(): number | undefined;
      /**
       * Get/Set the maxradius query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      maxRadius(value?: number): FedCatalogQuery;
      getMaxRadius(): number | undefined;
      /**
       * Get/Set the includerestricted query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      includeRestricted(value?: boolean): FedCatalogQuery;
      getIncludeRestricted(): boolean | undefined;
      /**
       * Get/Set the includeavailability query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      includeAvailability(value?: boolean): FedCatalogQuery;
      getIncludeAvailability(): boolean | undefined;
      /**
       * Get/Set the format query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      format(value?: string): FedCatalogQuery;
      getFormat(): string | undefined;
      /**
       * Get/Set the updatedafter query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      updatedAfter(value?: DateTime): FedCatalogQuery;
      getUpdatedAfter(): DateTime | undefined;
      /**
       * Get/Set the matchtimeseries query parameter.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      matchTimeseries(value?: boolean): FedCatalogQuery;
      getMatchTimeseries(): boolean | undefined;
      /**
       * Get/Set the timeout in seconds for the request. Default is 30.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      timeout(value?: number): FedCatalogQuery;
      getTimeout(): number | undefined;
      /**
       * Checks to see if any parameter that would limit the data
       * returned is set. This is a crude, coarse check to make sure
       * the client doesn't ask for EVERYTHING the server has.
       *
       * @returns true if some parameter set
       */
      isSomeParameterSet(): boolean;
      /**
       * Queries the remote web service for networks.
       *
       * @returns a Promise to an Array of Network objects.
       */
      queryNetworks(): Promise<Array<Network>>;
      /**
       * Queries the remote web service for stations. The stations
       * are contained within their respective Networks.
       *
       * @returns a Promise to an Array of Network objects.
       */
      queryStations(): Promise<Array<Network>>;
      /**
       * Queries the remote web service for channels. The Channels
       * are contained within their respective Stations which are in Networks.
       *
       * @returns a Promise to an Array of Network objects.
       */
      queryChannels(): Promise<Array<Network>>;
      /**
       * Queries the remote web service for responses. The Responses
       * are contained within their respective Channels,
       * which are in Stations which are in Networks.
       *
       * @returns a Promise to an Array of Network objects.
       */
      queryResponses(): Promise<Array<Network>>;
      /**
       * Queries the remote station web service at the given level.
       *
       * @param level the level to query at, networ, station, channel or response.
       * @returns a Promise to an Array of Network objects.
       */
      queryFdsnStation(level: string): Promise<Array<Network>>;
      setupQueryFdsnStation(level: string): Promise<FedCatalogResult>;
      /**
       * For each item in a parsed result returned from a FedCat service, create a
       * DataSelectQuery with host and port, or url filled in correctly, ready to
       * be called with the result lines.
       *
       * @param   fedCatalogResult result from a FedCat web service
       * @returns               result with dataSelectQuery added to each item
       */
      setupForFdnsDataSelect(fedCatalogResult: FedCatalogResult): FedCatalogResult;
      queryFdsnDataselect(): Promise<Array<SeismogramDisplayData>>;
      postFdsnDataselectForFedCatResult(fedCatalogResult: FedCatalogResult): Promise<Array<SeismogramDisplayData>>;
      /**
       * query the fed cat server for dataselect using post, which allows for multiple
       * channel-timeranges at once. This assumes that there are not multiple
       * time ranges for the same channel as the results, encapsulated as
       * SeismogramDisplayData objects, are returned one seismogram
       * per channel, which may contain gaps. The original channel and timerange are
       * also populated with each result.
       *
       * @param   sddList array of SeismogramDisplayData objects
       * that will be filled in with the resulting seismogram
       * @returns Promise to the input Array of SeismogramDisplayData objects, each with the
       * seismogram containing the data returned from the server
       */
      postQuerySeismograms(sddList: Array<SeismogramDisplayData>): Promise<Array<SeismogramDisplayData>>;
      postQueryRaw(sddList: Array<SeismogramDisplayData>, targetService: string): Promise<FedCatalogResult>;
      postQueryRawWithBody(body: string): Promise<FedCatalogResult>;
      /**
       * Queries the remote web service.
       *
       * @returns a Promise to an parsed result.
       */
      queryRaw(): Promise<FedCatalogResult>;
      handleHttpResponseCodes(response: Response): Promise<string>;
      parseRequest(requestText: string): FedCatalogResult;
      /**
       * Forms the URL to get version from the web service, without any query paramters
       *
       * @returns the url
       */
      formVersionURL(): string;
      /**
       * Queries the remote web service to get its version
       *
       * @returns Promise to version string
       */
      queryVersion(): Promise<string>;
      /**
       * Forms the basic URL to contact the web service, without any query paramters
       *
       * @returns the url
       */
      formBaseURL(): string;
      /**
       * Form URL to post the remote web service. No parameters are added
       * to the URL as those will be in the body of the post.
       *
       * @returns the URL for posting
       */
      formPostURL(): string;
      /**
       * Form URL to query the remote web service, encoding the query parameters.
       *
       * @returns url
       */
      formURL(): string;
  }

}
declare module 'seisplotjs/src/leaflet_css' {
  export const leaflet_css = "\n/* required styles */\n\n.leaflet-pane,\n.leaflet-tile,\n.leaflet-marker-icon,\n.leaflet-marker-shadow,\n.leaflet-tile-container,\n.leaflet-pane > svg,\n.leaflet-pane > canvas,\n.leaflet-zoom-box,\n.leaflet-image-layer,\n.leaflet-layer {\n\tposition: absolute;\n\tleft: 0;\n\ttop: 0;\n\t}\n.leaflet-container {\n\toverflow: hidden;\n\t}\n.leaflet-tile,\n.leaflet-marker-icon,\n.leaflet-marker-shadow {\n\t-webkit-user-select: none;\n\t   -moz-user-select: none;\n\t        user-select: none;\n\t  -webkit-user-drag: none;\n\t}\n/* Prevents IE11 from highlighting tiles in blue */\n.leaflet-tile::selection {\n\tbackground: transparent;\n}\n/* Safari renders non-retina tile on retina better with this, but Chrome is worse */\n.leaflet-safari .leaflet-tile {\n\timage-rendering: -webkit-optimize-contrast;\n\t}\n/* hack that prevents hw layers \"stretching\" when loading new tiles */\n.leaflet-safari .leaflet-tile-container {\n\twidth: 1600px;\n\theight: 1600px;\n\t-webkit-transform-origin: 0 0;\n\t}\n.leaflet-marker-icon,\n.leaflet-marker-shadow {\n\tdisplay: block;\n\t}\n/* .leaflet-container svg: reset svg max-width decleration shipped in Joomla! (joomla.org) 3.x */\n/* .leaflet-container img: map is broken in FF if you have max-width: 100% on tiles */\n.leaflet-container .leaflet-overlay-pane svg {\n\tmax-width: none !important;\n\tmax-height: none !important;\n\t}\n.leaflet-container .leaflet-marker-pane img,\n.leaflet-container .leaflet-shadow-pane img,\n.leaflet-container .leaflet-tile-pane img,\n.leaflet-container img.leaflet-image-layer,\n.leaflet-container .leaflet-tile {\n\tmax-width: none !important;\n\tmax-height: none !important;\n\twidth: auto;\n\tpadding: 0;\n\t}\n\n.leaflet-container img.leaflet-tile {\n\t/* See: https://bugs.chromium.org/p/chromium/issues/detail?id=600120 */\n\tmix-blend-mode: plus-lighter;\n}\n\n.leaflet-container.leaflet-touch-zoom {\n\t-ms-touch-action: pan-x pan-y;\n\ttouch-action: pan-x pan-y;\n\t}\n.leaflet-container.leaflet-touch-drag {\n\t-ms-touch-action: pinch-zoom;\n\t/* Fallback for FF which doesn't support pinch-zoom */\n\ttouch-action: none;\n\ttouch-action: pinch-zoom;\n}\n.leaflet-container.leaflet-touch-drag.leaflet-touch-zoom {\n\t-ms-touch-action: none;\n\ttouch-action: none;\n}\n.leaflet-container {\n\t-webkit-tap-highlight-color: transparent;\n}\n.leaflet-container a {\n\t-webkit-tap-highlight-color: rgba(51, 181, 229, 0.4);\n}\n.leaflet-tile {\n\tfilter: inherit;\n\tvisibility: hidden;\n\t}\n.leaflet-tile-loaded {\n\tvisibility: inherit;\n\t}\n.leaflet-zoom-box {\n\twidth: 0;\n\theight: 0;\n\t-moz-box-sizing: border-box;\n\t     box-sizing: border-box;\n\tz-index: 800;\n\t}\n/* workaround for https://bugzilla.mozilla.org/show_bug.cgi?id=888319 */\n.leaflet-overlay-pane svg {\n\t-moz-user-select: none;\n\t}\n\n.leaflet-pane         { z-index: 400; }\n\n.leaflet-tile-pane    { z-index: 200; }\n.leaflet-overlay-pane { z-index: 400; }\n.leaflet-shadow-pane  { z-index: 500; }\n.leaflet-marker-pane  { z-index: 600; }\n.leaflet-tooltip-pane   { z-index: 650; }\n.leaflet-popup-pane   { z-index: 700; }\n\n.leaflet-map-pane canvas { z-index: 100; }\n.leaflet-map-pane svg    { z-index: 200; }\n\n.leaflet-vml-shape {\n\twidth: 1px;\n\theight: 1px;\n\t}\n.lvml {\n\tbehavior: url(#default#VML);\n\tdisplay: inline-block;\n\tposition: absolute;\n\t}\n\n\n/* control positioning */\n\n.leaflet-control {\n\tposition: relative;\n\tz-index: 800;\n\tpointer-events: visiblePainted; /* IE 9-10 doesn't have auto */\n\tpointer-events: auto;\n\t}\n.leaflet-top,\n.leaflet-bottom {\n\tposition: absolute;\n\tz-index: 1000;\n\tpointer-events: none;\n\t}\n.leaflet-top {\n\ttop: 0;\n\t}\n.leaflet-right {\n\tright: 0;\n\t}\n.leaflet-bottom {\n\tbottom: 0;\n\t}\n.leaflet-left {\n\tleft: 0;\n\t}\n.leaflet-control {\n\tfloat: left;\n\tclear: both;\n\t}\n.leaflet-right .leaflet-control {\n\tfloat: right;\n\t}\n.leaflet-top .leaflet-control {\n\tmargin-top: 10px;\n\t}\n.leaflet-bottom .leaflet-control {\n\tmargin-bottom: 10px;\n\t}\n.leaflet-left .leaflet-control {\n\tmargin-left: 10px;\n\t}\n.leaflet-right .leaflet-control {\n\tmargin-right: 10px;\n\t}\n\n\n/* zoom and fade animations */\n\n.leaflet-fade-anim .leaflet-popup {\n\topacity: 0;\n\t-webkit-transition: opacity 0.2s linear;\n\t   -moz-transition: opacity 0.2s linear;\n\t        transition: opacity 0.2s linear;\n\t}\n.leaflet-fade-anim .leaflet-map-pane .leaflet-popup {\n\topacity: 1;\n\t}\n.leaflet-zoom-animated {\n\t-webkit-transform-origin: 0 0;\n\t    -ms-transform-origin: 0 0;\n\t        transform-origin: 0 0;\n\t}\nsvg.leaflet-zoom-animated {\n\twill-change: transform;\n}\n\n.leaflet-zoom-anim .leaflet-zoom-animated {\n\t-webkit-transition: -webkit-transform 0.25s cubic-bezier(0,0,0.25,1);\n\t   -moz-transition:    -moz-transform 0.25s cubic-bezier(0,0,0.25,1);\n\t        transition:         transform 0.25s cubic-bezier(0,0,0.25,1);\n\t}\n.leaflet-zoom-anim .leaflet-tile,\n.leaflet-pan-anim .leaflet-tile {\n\t-webkit-transition: none;\n\t   -moz-transition: none;\n\t        transition: none;\n\t}\n\n.leaflet-zoom-anim .leaflet-zoom-hide {\n\tvisibility: hidden;\n\t}\n\n\n/* cursors */\n\n.leaflet-interactive {\n\tcursor: pointer;\n\t}\n.leaflet-grab {\n\tcursor: -webkit-grab;\n\tcursor:    -moz-grab;\n\tcursor:         grab;\n\t}\n.leaflet-crosshair,\n.leaflet-crosshair .leaflet-interactive {\n\tcursor: crosshair;\n\t}\n.leaflet-popup-pane,\n.leaflet-control {\n\tcursor: auto;\n\t}\n.leaflet-dragging .leaflet-grab,\n.leaflet-dragging .leaflet-grab .leaflet-interactive,\n.leaflet-dragging .leaflet-marker-draggable {\n\tcursor: move;\n\tcursor: -webkit-grabbing;\n\tcursor:    -moz-grabbing;\n\tcursor:         grabbing;\n\t}\n\n/* marker & overlays interactivity */\n.leaflet-marker-icon,\n.leaflet-marker-shadow,\n.leaflet-image-layer,\n.leaflet-pane > svg path,\n.leaflet-tile-container {\n\tpointer-events: none;\n\t}\n\n.leaflet-marker-icon.leaflet-interactive,\n.leaflet-image-layer.leaflet-interactive,\n.leaflet-pane > svg path.leaflet-interactive,\nsvg.leaflet-image-layer.leaflet-interactive path {\n\tpointer-events: visiblePainted; /* IE 9-10 doesn't have auto */\n\tpointer-events: auto;\n\t}\n\n/* visual tweaks */\n\n.leaflet-container {\n\tbackground: #ddd;\n\toutline-offset: 1px;\n\t}\n.leaflet-container a {\n\tcolor: #0078A8;\n\t}\n.leaflet-zoom-box {\n\tborder: 2px dotted #38f;\n\tbackground: rgba(255,255,255,0.5);\n\t}\n\n\n/* general typography */\n.leaflet-container {\n\tfont-family: \"Helvetica Neue\", Arial, Helvetica, sans-serif;\n\tfont-size: 12px;\n\tfont-size: 0.75rem;\n\tline-height: 1.5;\n\t}\n\n\n/* general toolbar styles */\n\n.leaflet-bar {\n\tbox-shadow: 0 1px 5px rgba(0,0,0,0.65);\n\tborder-radius: 4px;\n\t}\n.leaflet-bar a {\n\tbackground-color: #fff;\n\tborder-bottom: 1px solid #ccc;\n\twidth: 26px;\n\theight: 26px;\n\tline-height: 26px;\n\tdisplay: block;\n\ttext-align: center;\n\ttext-decoration: none;\n\tcolor: black;\n\t}\n.leaflet-bar a,\n.leaflet-control-layers-toggle {\n\tbackground-position: 50% 50%;\n\tbackground-repeat: no-repeat;\n\tdisplay: block;\n\t}\n.leaflet-bar a:hover,\n.leaflet-bar a:focus {\n\tbackground-color: #f4f4f4;\n\t}\n.leaflet-bar a:first-child {\n\tborder-top-left-radius: 4px;\n\tborder-top-right-radius: 4px;\n\t}\n.leaflet-bar a:last-child {\n\tborder-bottom-left-radius: 4px;\n\tborder-bottom-right-radius: 4px;\n\tborder-bottom: none;\n\t}\n.leaflet-bar a.leaflet-disabled {\n\tcursor: default;\n\tbackground-color: #f4f4f4;\n\tcolor: #bbb;\n\t}\n\n.leaflet-touch .leaflet-bar a {\n\twidth: 30px;\n\theight: 30px;\n\tline-height: 30px;\n\t}\n.leaflet-touch .leaflet-bar a:first-child {\n\tborder-top-left-radius: 2px;\n\tborder-top-right-radius: 2px;\n\t}\n.leaflet-touch .leaflet-bar a:last-child {\n\tborder-bottom-left-radius: 2px;\n\tborder-bottom-right-radius: 2px;\n\t}\n\n/* zoom control */\n\n.leaflet-control-zoom-in,\n.leaflet-control-zoom-out {\n\tfont: bold 18px 'Lucida Console', Monaco, monospace;\n\ttext-indent: 1px;\n\t}\n\n.leaflet-touch .leaflet-control-zoom-in, .leaflet-touch .leaflet-control-zoom-out  {\n\tfont-size: 22px;\n\t}\n\n\n/* layers control */\n\n.leaflet-control-layers {\n\tbox-shadow: 0 1px 5px rgba(0,0,0,0.4);\n\tbackground: #fff;\n\tborder-radius: 5px;\n\t}\n.leaflet-control-layers-toggle {\n\tbackground-image: url(images/layers.png);\n\twidth: 36px;\n\theight: 36px;\n\t}\n.leaflet-retina .leaflet-control-layers-toggle {\n\tbackground-image: url(images/layers-2x.png);\n\tbackground-size: 26px 26px;\n\t}\n.leaflet-touch .leaflet-control-layers-toggle {\n\twidth: 44px;\n\theight: 44px;\n\t}\n.leaflet-control-layers .leaflet-control-layers-list,\n.leaflet-control-layers-expanded .leaflet-control-layers-toggle {\n\tdisplay: none;\n\t}\n.leaflet-control-layers-expanded .leaflet-control-layers-list {\n\tdisplay: block;\n\tposition: relative;\n\t}\n.leaflet-control-layers-expanded {\n\tpadding: 6px 10px 6px 6px;\n\tcolor: #333;\n\tbackground: #fff;\n\t}\n.leaflet-control-layers-scrollbar {\n\toverflow-y: scroll;\n\toverflow-x: hidden;\n\tpadding-right: 5px;\n\t}\n.leaflet-control-layers-selector {\n\tmargin-top: 2px;\n\tposition: relative;\n\ttop: 1px;\n\t}\n.leaflet-control-layers label {\n\tdisplay: block;\n\tfont-size: 13px;\n\tfont-size: 1.08333em;\n\t}\n.leaflet-control-layers-separator {\n\theight: 0;\n\tborder-top: 1px solid #ddd;\n\tmargin: 5px -10px 5px -6px;\n\t}\n\n/* Default icon URLs */\n.leaflet-default-icon-path { /* used only in path-guessing heuristic, see L.Icon.Default */\n\tbackground-image: url(images/marker-icon.png);\n\t}\n\n\n/* attribution and scale controls */\n\n.leaflet-container .leaflet-control-attribution {\n\tbackground: #fff;\n\tbackground: rgba(255, 255, 255, 0.8);\n\tmargin: 0;\n\t}\n.leaflet-control-attribution,\n.leaflet-control-scale-line {\n\tpadding: 0 5px;\n\tcolor: #333;\n\tline-height: 1.4;\n\t}\n.leaflet-control-attribution a {\n\ttext-decoration: none;\n\t}\n.leaflet-control-attribution a:hover,\n.leaflet-control-attribution a:focus {\n\ttext-decoration: underline;\n\t}\n.leaflet-attribution-flag {\n\tdisplay: inline !important;\n\tvertical-align: baseline !important;\n\twidth: 1em;\n\theight: 0.6669em;\n\t}\n.leaflet-left .leaflet-control-scale {\n\tmargin-left: 5px;\n\t}\n.leaflet-bottom .leaflet-control-scale {\n\tmargin-bottom: 5px;\n\t}\n.leaflet-control-scale-line {\n\tborder: 2px solid #777;\n\tborder-top: none;\n\tline-height: 1.1;\n\tpadding: 2px 5px 1px;\n\twhite-space: nowrap;\n\t-moz-box-sizing: border-box;\n\t     box-sizing: border-box;\n\tbackground: rgba(255, 255, 255, 0.8);\n\ttext-shadow: 1px 1px #fff;\n\t}\n.leaflet-control-scale-line:not(:first-child) {\n\tborder-top: 2px solid #777;\n\tborder-bottom: none;\n\tmargin-top: -2px;\n\t}\n.leaflet-control-scale-line:not(:first-child):not(:last-child) {\n\tborder-bottom: 2px solid #777;\n\t}\n\n.leaflet-touch .leaflet-control-attribution,\n.leaflet-touch .leaflet-control-layers,\n.leaflet-touch .leaflet-bar {\n\tbox-shadow: none;\n\t}\n.leaflet-touch .leaflet-control-layers,\n.leaflet-touch .leaflet-bar {\n\tborder: 2px solid rgba(0,0,0,0.2);\n\tbackground-clip: padding-box;\n\t}\n\n\n/* popup */\n\n.leaflet-popup {\n\tposition: absolute;\n\ttext-align: center;\n\tmargin-bottom: 20px;\n\t}\n.leaflet-popup-content-wrapper {\n\tpadding: 1px;\n\ttext-align: left;\n\tborder-radius: 12px;\n\t}\n.leaflet-popup-content {\n\tmargin: 13px 24px 13px 20px;\n\tline-height: 1.3;\n\tfont-size: 13px;\n\tfont-size: 1.08333em;\n\tmin-height: 1px;\n\t}\n.leaflet-popup-content p {\n\tmargin: 17px 0;\n\tmargin: 1.3em 0;\n\t}\n.leaflet-popup-tip-container {\n\twidth: 40px;\n\theight: 20px;\n\tposition: absolute;\n\tleft: 50%;\n\tmargin-top: -1px;\n\tmargin-left: -20px;\n\toverflow: hidden;\n\tpointer-events: none;\n\t}\n.leaflet-popup-tip {\n\twidth: 17px;\n\theight: 17px;\n\tpadding: 1px;\n\n\tmargin: -10px auto 0;\n\tpointer-events: auto;\n\n\t-webkit-transform: rotate(45deg);\n\t   -moz-transform: rotate(45deg);\n\t    -ms-transform: rotate(45deg);\n\t        transform: rotate(45deg);\n\t}\n.leaflet-popup-content-wrapper,\n.leaflet-popup-tip {\n\tbackground: white;\n\tcolor: #333;\n\tbox-shadow: 0 3px 14px rgba(0,0,0,0.4);\n\t}\n.leaflet-container a.leaflet-popup-close-button {\n\tposition: absolute;\n\ttop: 0;\n\tright: 0;\n\tborder: none;\n\ttext-align: center;\n\twidth: 24px;\n\theight: 24px;\n\tfont: 16px/24px Tahoma, Verdana, sans-serif;\n\tcolor: #757575;\n\ttext-decoration: none;\n\tbackground: transparent;\n\t}\n.leaflet-container a.leaflet-popup-close-button:hover,\n.leaflet-container a.leaflet-popup-close-button:focus {\n\tcolor: #585858;\n\t}\n.leaflet-popup-scrolled {\n\toverflow: auto;\n\t}\n\n.leaflet-oldie .leaflet-popup-content-wrapper {\n\t-ms-zoom: 1;\n\t}\n.leaflet-oldie .leaflet-popup-tip {\n\twidth: 24px;\n\tmargin: 0 auto;\n\n\t-ms-filter: \"progid:DXImageTransform.Microsoft.Matrix(M11=0.70710678, M12=0.70710678, M21=-0.70710678, M22=0.70710678)\";\n\tfilter: progid:DXImageTransform.Microsoft.Matrix(M11=0.70710678, M12=0.70710678, M21=-0.70710678, M22=0.70710678);\n\t}\n\n.leaflet-oldie .leaflet-control-zoom,\n.leaflet-oldie .leaflet-control-layers,\n.leaflet-oldie .leaflet-popup-content-wrapper,\n.leaflet-oldie .leaflet-popup-tip {\n\tborder: 1px solid #999;\n\t}\n\n\n/* div icon */\n\n.leaflet-div-icon {\n\tbackground: #fff;\n\tborder: 1px solid #666;\n\t}\n\n\n/* Tooltip */\n/* Base styles for the element that has a tooltip */\n.leaflet-tooltip {\n\tposition: absolute;\n\tpadding: 6px;\n\tbackground-color: #fff;\n\tborder: 1px solid #fff;\n\tborder-radius: 3px;\n\tcolor: #222;\n\twhite-space: nowrap;\n\t-webkit-user-select: none;\n\t-moz-user-select: none;\n\t-ms-user-select: none;\n\tuser-select: none;\n\tpointer-events: none;\n\tbox-shadow: 0 1px 3px rgba(0,0,0,0.4);\n\t}\n.leaflet-tooltip.leaflet-interactive {\n\tcursor: pointer;\n\tpointer-events: auto;\n\t}\n.leaflet-tooltip-top:before,\n.leaflet-tooltip-bottom:before,\n.leaflet-tooltip-left:before,\n.leaflet-tooltip-right:before {\n\tposition: absolute;\n\tpointer-events: none;\n\tborder: 6px solid transparent;\n\tbackground: transparent;\n\tcontent: \"\";\n\t}\n\n/* Directions */\n\n.leaflet-tooltip-bottom {\n\tmargin-top: 6px;\n}\n.leaflet-tooltip-top {\n\tmargin-top: -6px;\n}\n.leaflet-tooltip-bottom:before,\n.leaflet-tooltip-top:before {\n\tleft: 50%;\n\tmargin-left: -6px;\n\t}\n.leaflet-tooltip-top:before {\n\tbottom: 0;\n\tmargin-bottom: -12px;\n\tborder-top-color: #fff;\n\t}\n.leaflet-tooltip-bottom:before {\n\ttop: 0;\n\tmargin-top: -12px;\n\tmargin-left: -6px;\n\tborder-bottom-color: #fff;\n\t}\n.leaflet-tooltip-left {\n\tmargin-left: -6px;\n}\n.leaflet-tooltip-right {\n\tmargin-left: 6px;\n}\n.leaflet-tooltip-left:before,\n.leaflet-tooltip-right:before {\n\ttop: 50%;\n\tmargin-top: -6px;\n\t}\n.leaflet-tooltip-left:before {\n\tright: 0;\n\tmargin-right: -12px;\n\tborder-left-color: #fff;\n\t}\n.leaflet-tooltip-right:before {\n\tleft: 0;\n\tmargin-left: -12px;\n\tborder-right-color: #fff;\n\t}\n\n/* Printing */\n\n@media print {\n\t/* Prevent printers from removing background-images of controls. */\n\t.leaflet-control {\n\t\t-webkit-print-color-adjust: exact;\n\t\tprint-color-adjust: exact;\n\t\t}\n\t}\n\n";

}
declare module 'seisplotjs/src/leafletutil' {
  import { Quake } from "seisplotjs/src/quakeml";
  import { Station } from "seisplotjs/src/stationxml";
  import { SeisPlotElement } from "seisplotjs/src/spelement";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  import { LatLonBox, LatLonRadius } from "seisplotjs/src/fdsncommon";
  import * as L from "leaflet";
  export const MAP_ELEMENT = "sp-station-quake-map";
  export const triangle = "\u25B2";
  export const StationMarkerClassName = "stationMapMarker";
  export const InactiveStationMarkerClassName = "inactiveStationMapMarker";
  export const QuakeMarkerClassName = "quakeMapMarker";
  export const stationIcon: L.DivIcon;
  export const inactiveStationIcon: L.DivIcon;
  export const stationMarker_css = "\n\n:host {\n  display: block\n}\n\ndiv.wrapper {\n  height: 100%;\n  min-height: 100px;\n}\n\n.leaflet-container {\n  height: 100%;\n  width: 100%;\n}\n\n.stationMapMarker.inactiveStationMapMarker {\n  color: darkgrey;\n  font-size: large;\n  z-index: 1;\n  text-shadow: 1px 1px 0 dimgrey, -1px 1px 0 dimgrey, -2px 1px 0 dimgrey, -1px -1px 0 dimgrey, 0 -3px 0 dimgrey, 1px -1px 0 dimgrey, 2px 1px 0 dimgrey;\n}\n.inactiveStationMapMarker:after{\n  content: \"\u25B2\";\n}\n.stationMapMarker {\n  color: blue;\n  font-size: large;\n  z-index: 10;\n  text-shadow: 1px 1px 0 dimgrey, -1px 1px 0 dimgrey, -2px 1px 0 dimgrey, -1px -1px 0 dimgrey, 0 -3px 0 dimgrey, 1px -1px 0 dimgrey, 2px 1px 0 dimgrey;\n}\n\n.stationMapMarker:after{\n  content: \"\u25B2\";\n}\n.quakeMapMarker {\n  stroke: red;\n  fill: #f03;\n  fill-opacity: 0.15;\n}\n";
  export function createStationMarker(station: Station, classList?: Array<string>, isactive?: boolean, centerLon?: number): L.Marker<any>;
  /**
   * Create a circle marker for Quake. Radius is linearly scaled by magnitude,
   * with min radius of 1 for very small magnitudes. Longitudes are adjusted
   * by +-360 to draw centered on the given center longitude, eg event at
   * lon=350 may plot at -10 if centerlon < 180.
   * @param  quake                          earthquake
   * @param  magScaleFactor               scale factor
   * @param  classList                      CSS classes to attach
   * @param  centerLon                    center longitude of the map
   * @returns leaflet circleMarker
   */
  export function createQuakeMarker(quake: Quake, magScaleFactor?: number, classList?: Array<string>, centerLon?: number): L.CircleMarker<any>;
  export const leaflet_css = "\n/* required styles */\n\n.leaflet-pane,\n.leaflet-tile,\n.leaflet-marker-icon,\n.leaflet-marker-shadow,\n.leaflet-tile-container,\n.leaflet-pane > svg,\n.leaflet-pane > canvas,\n.leaflet-zoom-box,\n.leaflet-image-layer,\n.leaflet-layer {\n\tposition: absolute;\n\tleft: 0;\n\ttop: 0;\n\t}\n.leaflet-container {\n\toverflow: hidden;\n\t}\n.leaflet-tile,\n.leaflet-marker-icon,\n.leaflet-marker-shadow {\n\t-webkit-user-select: none;\n\t   -moz-user-select: none;\n\t        user-select: none;\n\t  -webkit-user-drag: none;\n\t}\n/* Prevents IE11 from highlighting tiles in blue */\n.leaflet-tile::selection {\n\tbackground: transparent;\n}\n/* Safari renders non-retina tile on retina better with this, but Chrome is worse */\n.leaflet-safari .leaflet-tile {\n\timage-rendering: -webkit-optimize-contrast;\n\t}\n/* hack that prevents hw layers \"stretching\" when loading new tiles */\n.leaflet-safari .leaflet-tile-container {\n\twidth: 1600px;\n\theight: 1600px;\n\t-webkit-transform-origin: 0 0;\n\t}\n.leaflet-marker-icon,\n.leaflet-marker-shadow {\n\tdisplay: block;\n\t}\n/* .leaflet-container svg: reset svg max-width decleration shipped in Joomla! (joomla.org) 3.x */\n/* .leaflet-container img: map is broken in FF if you have max-width: 100% on tiles */\n.leaflet-container .leaflet-overlay-pane svg {\n\tmax-width: none !important;\n\tmax-height: none !important;\n\t}\n.leaflet-container .leaflet-marker-pane img,\n.leaflet-container .leaflet-shadow-pane img,\n.leaflet-container .leaflet-tile-pane img,\n.leaflet-container img.leaflet-image-layer,\n.leaflet-container .leaflet-tile {\n\tmax-width: none !important;\n\tmax-height: none !important;\n\twidth: auto;\n\tpadding: 0;\n\t}\n\n.leaflet-container img.leaflet-tile {\n\t/* See: https://bugs.chromium.org/p/chromium/issues/detail?id=600120 */\n\tmix-blend-mode: plus-lighter;\n}\n\n.leaflet-container.leaflet-touch-zoom {\n\t-ms-touch-action: pan-x pan-y;\n\ttouch-action: pan-x pan-y;\n\t}\n.leaflet-container.leaflet-touch-drag {\n\t-ms-touch-action: pinch-zoom;\n\t/* Fallback for FF which doesn't support pinch-zoom */\n\ttouch-action: none;\n\ttouch-action: pinch-zoom;\n}\n.leaflet-container.leaflet-touch-drag.leaflet-touch-zoom {\n\t-ms-touch-action: none;\n\ttouch-action: none;\n}\n.leaflet-container {\n\t-webkit-tap-highlight-color: transparent;\n}\n.leaflet-container a {\n\t-webkit-tap-highlight-color: rgba(51, 181, 229, 0.4);\n}\n.leaflet-tile {\n\tfilter: inherit;\n\tvisibility: hidden;\n\t}\n.leaflet-tile-loaded {\n\tvisibility: inherit;\n\t}\n.leaflet-zoom-box {\n\twidth: 0;\n\theight: 0;\n\t-moz-box-sizing: border-box;\n\t     box-sizing: border-box;\n\tz-index: 800;\n\t}\n/* workaround for https://bugzilla.mozilla.org/show_bug.cgi?id=888319 */\n.leaflet-overlay-pane svg {\n\t-moz-user-select: none;\n\t}\n\n.leaflet-pane         { z-index: 400; }\n\n.leaflet-tile-pane    { z-index: 200; }\n.leaflet-overlay-pane { z-index: 400; }\n.leaflet-shadow-pane  { z-index: 500; }\n.leaflet-marker-pane  { z-index: 600; }\n.leaflet-tooltip-pane   { z-index: 650; }\n.leaflet-popup-pane   { z-index: 700; }\n\n.leaflet-map-pane canvas { z-index: 100; }\n.leaflet-map-pane svg    { z-index: 200; }\n\n.leaflet-vml-shape {\n\twidth: 1px;\n\theight: 1px;\n\t}\n.lvml {\n\tbehavior: url(#default#VML);\n\tdisplay: inline-block;\n\tposition: absolute;\n\t}\n\n\n/* control positioning */\n\n.leaflet-control {\n\tposition: relative;\n\tz-index: 800;\n\tpointer-events: visiblePainted; /* IE 9-10 doesn't have auto */\n\tpointer-events: auto;\n\t}\n.leaflet-top,\n.leaflet-bottom {\n\tposition: absolute;\n\tz-index: 1000;\n\tpointer-events: none;\n\t}\n.leaflet-top {\n\ttop: 0;\n\t}\n.leaflet-right {\n\tright: 0;\n\t}\n.leaflet-bottom {\n\tbottom: 0;\n\t}\n.leaflet-left {\n\tleft: 0;\n\t}\n.leaflet-control {\n\tfloat: left;\n\tclear: both;\n\t}\n.leaflet-right .leaflet-control {\n\tfloat: right;\n\t}\n.leaflet-top .leaflet-control {\n\tmargin-top: 10px;\n\t}\n.leaflet-bottom .leaflet-control {\n\tmargin-bottom: 10px;\n\t}\n.leaflet-left .leaflet-control {\n\tmargin-left: 10px;\n\t}\n.leaflet-right .leaflet-control {\n\tmargin-right: 10px;\n\t}\n\n\n/* zoom and fade animations */\n\n.leaflet-fade-anim .leaflet-popup {\n\topacity: 0;\n\t-webkit-transition: opacity 0.2s linear;\n\t   -moz-transition: opacity 0.2s linear;\n\t        transition: opacity 0.2s linear;\n\t}\n.leaflet-fade-anim .leaflet-map-pane .leaflet-popup {\n\topacity: 1;\n\t}\n.leaflet-zoom-animated {\n\t-webkit-transform-origin: 0 0;\n\t    -ms-transform-origin: 0 0;\n\t        transform-origin: 0 0;\n\t}\nsvg.leaflet-zoom-animated {\n\twill-change: transform;\n}\n\n.leaflet-zoom-anim .leaflet-zoom-animated {\n\t-webkit-transition: -webkit-transform 0.25s cubic-bezier(0,0,0.25,1);\n\t   -moz-transition:    -moz-transform 0.25s cubic-bezier(0,0,0.25,1);\n\t        transition:         transform 0.25s cubic-bezier(0,0,0.25,1);\n\t}\n.leaflet-zoom-anim .leaflet-tile,\n.leaflet-pan-anim .leaflet-tile {\n\t-webkit-transition: none;\n\t   -moz-transition: none;\n\t        transition: none;\n\t}\n\n.leaflet-zoom-anim .leaflet-zoom-hide {\n\tvisibility: hidden;\n\t}\n\n\n/* cursors */\n\n.leaflet-interactive {\n\tcursor: pointer;\n\t}\n.leaflet-grab {\n\tcursor: -webkit-grab;\n\tcursor:    -moz-grab;\n\tcursor:         grab;\n\t}\n.leaflet-crosshair,\n.leaflet-crosshair .leaflet-interactive {\n\tcursor: crosshair;\n\t}\n.leaflet-popup-pane,\n.leaflet-control {\n\tcursor: auto;\n\t}\n.leaflet-dragging .leaflet-grab,\n.leaflet-dragging .leaflet-grab .leaflet-interactive,\n.leaflet-dragging .leaflet-marker-draggable {\n\tcursor: move;\n\tcursor: -webkit-grabbing;\n\tcursor:    -moz-grabbing;\n\tcursor:         grabbing;\n\t}\n\n/* marker & overlays interactivity */\n.leaflet-marker-icon,\n.leaflet-marker-shadow,\n.leaflet-image-layer,\n.leaflet-pane > svg path,\n.leaflet-tile-container {\n\tpointer-events: none;\n\t}\n\n.leaflet-marker-icon.leaflet-interactive,\n.leaflet-image-layer.leaflet-interactive,\n.leaflet-pane > svg path.leaflet-interactive,\nsvg.leaflet-image-layer.leaflet-interactive path {\n\tpointer-events: visiblePainted; /* IE 9-10 doesn't have auto */\n\tpointer-events: auto;\n\t}\n\n/* visual tweaks */\n\n.leaflet-container {\n\tbackground: #ddd;\n\toutline-offset: 1px;\n\t}\n.leaflet-container a {\n\tcolor: #0078A8;\n\t}\n.leaflet-zoom-box {\n\tborder: 2px dotted #38f;\n\tbackground: rgba(255,255,255,0.5);\n\t}\n\n\n/* general typography */\n.leaflet-container {\n\tfont-family: \"Helvetica Neue\", Arial, Helvetica, sans-serif;\n\tfont-size: 12px;\n\tfont-size: 0.75rem;\n\tline-height: 1.5;\n\t}\n\n\n/* general toolbar styles */\n\n.leaflet-bar {\n\tbox-shadow: 0 1px 5px rgba(0,0,0,0.65);\n\tborder-radius: 4px;\n\t}\n.leaflet-bar a {\n\tbackground-color: #fff;\n\tborder-bottom: 1px solid #ccc;\n\twidth: 26px;\n\theight: 26px;\n\tline-height: 26px;\n\tdisplay: block;\n\ttext-align: center;\n\ttext-decoration: none;\n\tcolor: black;\n\t}\n.leaflet-bar a,\n.leaflet-control-layers-toggle {\n\tbackground-position: 50% 50%;\n\tbackground-repeat: no-repeat;\n\tdisplay: block;\n\t}\n.leaflet-bar a:hover,\n.leaflet-bar a:focus {\n\tbackground-color: #f4f4f4;\n\t}\n.leaflet-bar a:first-child {\n\tborder-top-left-radius: 4px;\n\tborder-top-right-radius: 4px;\n\t}\n.leaflet-bar a:last-child {\n\tborder-bottom-left-radius: 4px;\n\tborder-bottom-right-radius: 4px;\n\tborder-bottom: none;\n\t}\n.leaflet-bar a.leaflet-disabled {\n\tcursor: default;\n\tbackground-color: #f4f4f4;\n\tcolor: #bbb;\n\t}\n\n.leaflet-touch .leaflet-bar a {\n\twidth: 30px;\n\theight: 30px;\n\tline-height: 30px;\n\t}\n.leaflet-touch .leaflet-bar a:first-child {\n\tborder-top-left-radius: 2px;\n\tborder-top-right-radius: 2px;\n\t}\n.leaflet-touch .leaflet-bar a:last-child {\n\tborder-bottom-left-radius: 2px;\n\tborder-bottom-right-radius: 2px;\n\t}\n\n/* zoom control */\n\n.leaflet-control-zoom-in,\n.leaflet-control-zoom-out {\n\tfont: bold 18px 'Lucida Console', Monaco, monospace;\n\ttext-indent: 1px;\n\t}\n\n.leaflet-touch .leaflet-control-zoom-in, .leaflet-touch .leaflet-control-zoom-out  {\n\tfont-size: 22px;\n\t}\n\n\n/* layers control */\n\n.leaflet-control-layers {\n\tbox-shadow: 0 1px 5px rgba(0,0,0,0.4);\n\tbackground: #fff;\n\tborder-radius: 5px;\n\t}\n.leaflet-control-layers-toggle {\n\tbackground-image: url(images/layers.png);\n\twidth: 36px;\n\theight: 36px;\n\t}\n.leaflet-retina .leaflet-control-layers-toggle {\n\tbackground-image: url(images/layers-2x.png);\n\tbackground-size: 26px 26px;\n\t}\n.leaflet-touch .leaflet-control-layers-toggle {\n\twidth: 44px;\n\theight: 44px;\n\t}\n.leaflet-control-layers .leaflet-control-layers-list,\n.leaflet-control-layers-expanded .leaflet-control-layers-toggle {\n\tdisplay: none;\n\t}\n.leaflet-control-layers-expanded .leaflet-control-layers-list {\n\tdisplay: block;\n\tposition: relative;\n\t}\n.leaflet-control-layers-expanded {\n\tpadding: 6px 10px 6px 6px;\n\tcolor: #333;\n\tbackground: #fff;\n\t}\n.leaflet-control-layers-scrollbar {\n\toverflow-y: scroll;\n\toverflow-x: hidden;\n\tpadding-right: 5px;\n\t}\n.leaflet-control-layers-selector {\n\tmargin-top: 2px;\n\tposition: relative;\n\ttop: 1px;\n\t}\n.leaflet-control-layers label {\n\tdisplay: block;\n\tfont-size: 13px;\n\tfont-size: 1.08333em;\n\t}\n.leaflet-control-layers-separator {\n\theight: 0;\n\tborder-top: 1px solid #ddd;\n\tmargin: 5px -10px 5px -6px;\n\t}\n\n/* Default icon URLs */\n.leaflet-default-icon-path { /* used only in path-guessing heuristic, see L.Icon.Default */\n\tbackground-image: url(images/marker-icon.png);\n\t}\n\n\n/* attribution and scale controls */\n\n.leaflet-container .leaflet-control-attribution {\n\tbackground: #fff;\n\tbackground: rgba(255, 255, 255, 0.8);\n\tmargin: 0;\n\t}\n.leaflet-control-attribution,\n.leaflet-control-scale-line {\n\tpadding: 0 5px;\n\tcolor: #333;\n\tline-height: 1.4;\n\t}\n.leaflet-control-attribution a {\n\ttext-decoration: none;\n\t}\n.leaflet-control-attribution a:hover,\n.leaflet-control-attribution a:focus {\n\ttext-decoration: underline;\n\t}\n.leaflet-attribution-flag {\n\tdisplay: inline !important;\n\tvertical-align: baseline !important;\n\twidth: 1em;\n\theight: 0.6669em;\n\t}\n.leaflet-left .leaflet-control-scale {\n\tmargin-left: 5px;\n\t}\n.leaflet-bottom .leaflet-control-scale {\n\tmargin-bottom: 5px;\n\t}\n.leaflet-control-scale-line {\n\tborder: 2px solid #777;\n\tborder-top: none;\n\tline-height: 1.1;\n\tpadding: 2px 5px 1px;\n\twhite-space: nowrap;\n\t-moz-box-sizing: border-box;\n\t     box-sizing: border-box;\n\tbackground: rgba(255, 255, 255, 0.8);\n\ttext-shadow: 1px 1px #fff;\n\t}\n.leaflet-control-scale-line:not(:first-child) {\n\tborder-top: 2px solid #777;\n\tborder-bottom: none;\n\tmargin-top: -2px;\n\t}\n.leaflet-control-scale-line:not(:first-child):not(:last-child) {\n\tborder-bottom: 2px solid #777;\n\t}\n\n.leaflet-touch .leaflet-control-attribution,\n.leaflet-touch .leaflet-control-layers,\n.leaflet-touch .leaflet-bar {\n\tbox-shadow: none;\n\t}\n.leaflet-touch .leaflet-control-layers,\n.leaflet-touch .leaflet-bar {\n\tborder: 2px solid rgba(0,0,0,0.2);\n\tbackground-clip: padding-box;\n\t}\n\n\n/* popup */\n\n.leaflet-popup {\n\tposition: absolute;\n\ttext-align: center;\n\tmargin-bottom: 20px;\n\t}\n.leaflet-popup-content-wrapper {\n\tpadding: 1px;\n\ttext-align: left;\n\tborder-radius: 12px;\n\t}\n.leaflet-popup-content {\n\tmargin: 13px 24px 13px 20px;\n\tline-height: 1.3;\n\tfont-size: 13px;\n\tfont-size: 1.08333em;\n\tmin-height: 1px;\n\t}\n.leaflet-popup-content p {\n\tmargin: 17px 0;\n\tmargin: 1.3em 0;\n\t}\n.leaflet-popup-tip-container {\n\twidth: 40px;\n\theight: 20px;\n\tposition: absolute;\n\tleft: 50%;\n\tmargin-top: -1px;\n\tmargin-left: -20px;\n\toverflow: hidden;\n\tpointer-events: none;\n\t}\n.leaflet-popup-tip {\n\twidth: 17px;\n\theight: 17px;\n\tpadding: 1px;\n\n\tmargin: -10px auto 0;\n\tpointer-events: auto;\n\n\t-webkit-transform: rotate(45deg);\n\t   -moz-transform: rotate(45deg);\n\t    -ms-transform: rotate(45deg);\n\t        transform: rotate(45deg);\n\t}\n.leaflet-popup-content-wrapper,\n.leaflet-popup-tip {\n\tbackground: white;\n\tcolor: #333;\n\tbox-shadow: 0 3px 14px rgba(0,0,0,0.4);\n\t}\n.leaflet-container a.leaflet-popup-close-button {\n\tposition: absolute;\n\ttop: 0;\n\tright: 0;\n\tborder: none;\n\ttext-align: center;\n\twidth: 24px;\n\theight: 24px;\n\tfont: 16px/24px Tahoma, Verdana, sans-serif;\n\tcolor: #757575;\n\ttext-decoration: none;\n\tbackground: transparent;\n\t}\n.leaflet-container a.leaflet-popup-close-button:hover,\n.leaflet-container a.leaflet-popup-close-button:focus {\n\tcolor: #585858;\n\t}\n.leaflet-popup-scrolled {\n\toverflow: auto;\n\t}\n\n.leaflet-oldie .leaflet-popup-content-wrapper {\n\t-ms-zoom: 1;\n\t}\n.leaflet-oldie .leaflet-popup-tip {\n\twidth: 24px;\n\tmargin: 0 auto;\n\n\t-ms-filter: \"progid:DXImageTransform.Microsoft.Matrix(M11=0.70710678, M12=0.70710678, M21=-0.70710678, M22=0.70710678)\";\n\tfilter: progid:DXImageTransform.Microsoft.Matrix(M11=0.70710678, M12=0.70710678, M21=-0.70710678, M22=0.70710678);\n\t}\n\n.leaflet-oldie .leaflet-control-zoom,\n.leaflet-oldie .leaflet-control-layers,\n.leaflet-oldie .leaflet-popup-content-wrapper,\n.leaflet-oldie .leaflet-popup-tip {\n\tborder: 1px solid #999;\n\t}\n\n\n/* div icon */\n\n.leaflet-div-icon {\n\tbackground: #fff;\n\tborder: 1px solid #666;\n\t}\n\n\n/* Tooltip */\n/* Base styles for the element that has a tooltip */\n.leaflet-tooltip {\n\tposition: absolute;\n\tpadding: 6px;\n\tbackground-color: #fff;\n\tborder: 1px solid #fff;\n\tborder-radius: 3px;\n\tcolor: #222;\n\twhite-space: nowrap;\n\t-webkit-user-select: none;\n\t-moz-user-select: none;\n\t-ms-user-select: none;\n\tuser-select: none;\n\tpointer-events: none;\n\tbox-shadow: 0 1px 3px rgba(0,0,0,0.4);\n\t}\n.leaflet-tooltip.leaflet-interactive {\n\tcursor: pointer;\n\tpointer-events: auto;\n\t}\n.leaflet-tooltip-top:before,\n.leaflet-tooltip-bottom:before,\n.leaflet-tooltip-left:before,\n.leaflet-tooltip-right:before {\n\tposition: absolute;\n\tpointer-events: none;\n\tborder: 6px solid transparent;\n\tbackground: transparent;\n\tcontent: \"\";\n\t}\n\n/* Directions */\n\n.leaflet-tooltip-bottom {\n\tmargin-top: 6px;\n}\n.leaflet-tooltip-top {\n\tmargin-top: -6px;\n}\n.leaflet-tooltip-bottom:before,\n.leaflet-tooltip-top:before {\n\tleft: 50%;\n\tmargin-left: -6px;\n\t}\n.leaflet-tooltip-top:before {\n\tbottom: 0;\n\tmargin-bottom: -12px;\n\tborder-top-color: #fff;\n\t}\n.leaflet-tooltip-bottom:before {\n\ttop: 0;\n\tmargin-top: -12px;\n\tmargin-left: -6px;\n\tborder-bottom-color: #fff;\n\t}\n.leaflet-tooltip-left {\n\tmargin-left: -6px;\n}\n.leaflet-tooltip-right {\n\tmargin-left: 6px;\n}\n.leaflet-tooltip-left:before,\n.leaflet-tooltip-right:before {\n\ttop: 50%;\n\tmargin-top: -6px;\n\t}\n.leaflet-tooltip-left:before {\n\tright: 0;\n\tmargin-right: -12px;\n\tborder-left-color: #fff;\n\t}\n.leaflet-tooltip-right:before {\n\tleft: 0;\n\tmargin-left: -12px;\n\tborder-right-color: #fff;\n\t}\n\n/* Printing */\n\n@media print {\n\t/* Prevent printers from removing background-images of controls. */\n\t.leaflet-control {\n\t\t-webkit-print-color-adjust: exact;\n\t\tprint-color-adjust: exact;\n\t\t}\n\t}\n\n";
  export const TILE_TEMPLATE = "tileUrl";
  export const DEFAULT_TILE_TEMPLATE = "https://services.arcgisonline.com/arcgis/rest/services/Ocean/World_Ocean_Base/MapServer/tile/{z}/{y}/{x}";
  export const TILE_ATTRIBUTION = "tileAttribution";
  export const MAX_ZOOM = "maxZoom";
  export const DEFAULT_MAX_ZOOM = 17;
  export const CENTER_LAT = "centerLat";
  export const DEFAULT_CENTER_LAT = 35;
  export const CENTER_LON = "centerLon";
  export const DEFAULT_CENTER_LON = -81;
  export const ZOOM_LEVEL = "zoomLevel";
  export const DEFAULT_ZOOM_LEVEL = 1;
  export const MAG_SCALE = "magScale";
  export const DEFAULT_MAG_SCALE = 5;
  export const FIT_BOUNDS = "fitBounds";
  export const QUAKE_MARKER_STYLE_EL = "quakeMarkerStyle";
  export const STATION_MARKER_STYLE_EL = "staMarkerStyle";
  export const STATION_CODE_SEP = "_";
  export class QuakeStationMap extends SeisPlotElement {
      quakeList: Array<Quake>;
      stationList: Array<Station>;
      geoRegionList: Array<LatLonBox | LatLonRadius>;
      map: L.Map | null;
      classToColor: Map<string, string>;
      stationClassMap: Map<string, Array<string>>;
      quakeClassMap: Map<string, Array<string>>;
      constructor(seisData?: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      addQuake(quake: Quake | Array<Quake>, classname?: string): void;
      /**
       * Adds a css class for the quake icon for additional styling,
       * either via addStyle() for general or via colorClass() for just
       * simply coloring.
       *
       * @param  quake  the quake
       * @param  classname  css class name
       */
      quakeAddClass(quake: Quake, classname: string): void;
      /**
       * Removes a css class from the earthquake circle.
       *
       * @param  quake  quake to remove
       * @param  classname   class to remove
       */
      quakeRemoveClass(quake: Quake, classname: string): void;
      /**
       * Removes a css class from all earthquake circles.
       *
       * @param  classname   class to remove
       */
      quakeRemoveAllClass(classname: string): void;
      addStation(station: Station | Array<Station>, classname?: string): void;
      /**
       * Adds a css class for the station icon for additional styling,
       * either via addStyle() for general or via colorClass() for just
       * simply coloring.
       *
       * @param  station  the station
       * @param  classname  css class name
       */
      stationAddClass(station: Station, classname: string): void;
      /**
       * Removes a css class from the station triangle
       *
       * @param  station   the station
       * @param  classname  css class name
       */
      stationRemoveClass(station: Station, classname: string): void;
      /**
       * Set a color in css for the classname. This is a simple alternative
       * to full styling via addStyle().
       *
       * @param  classname  css class name
       * @param  color      color, like red
       */
      colorClass(classname: string, color: string): void;
      removeColorClass(classname: string): void;
      get fitBounds(): boolean;
      set fitBounds(val: number);
      get centerLat(): number;
      set centerLat(val: number);
      get centerLon(): number;
      set centerLon(val: number);
      get zoomLevel(): number;
      set zoomLevel(val: number);
      get magScale(): number;
      set magScale(val: number);
      draw(): void;
      updateQuakeMarkerStyle(): void;
      updateStationMarkerStyle(): void;
      drawGeoRegions(map: L.Map): Array<[number, number]>;
      createStationMarkerColorStyle(): string;
      createQuakeMarkerColorStyle(): string;
      attributeChangedCallback(_name: string, _oldValue: string, _newValue: string): void;
      static get observedAttributes(): string[];
  }
  export function cssClassForQuake(q: Quake): string;

}
declare module 'seisplotjs/src/miniseed' {
  import { DateTime } from "luxon";
  import { SeismogramSegment } from "seisplotjs/src/seismogramsegment";
  import { Seismogram } from "seisplotjs/src/seismogram";
  import { EncodedDataSegment } from "seisplotjs/src/seedcodec";
  export const MINISEED_MIME = "application/vnd.fdsn.mseed";
  export const R_TYPECODE: number;
  export const D_TYPECODE: number;
  export const Q_TYPECODE: number;
  export const M_TYPECODE: number;
  /**
   * parse arrayBuffer into an array of DataRecords.
   *
   * @param arrayBuffer bytes to parse
   * @returns arry of data records
   */
  export function parseDataRecords(arrayBuffer: ArrayBuffer): Array<DataRecord>;
  /**
   * parse a single DataRecord starting at the beginning of the DataView.
   * Currently only some blockettes are parsed, 100, 1000, 1001, others are separated,
   * but left as just a DataView.
   *
   * @param dataView bytes as DataView
   * @returns data record
   */
  export function parseSingleDataRecord(dataView: DataView): DataRecord;
  /**
   * parse the DataHeader from a single DataRecord starting at the beginning of the DataView.
   *
   * @param dataView bytes as DataView
   * @returns data record header
   */
  export function parseSingleDataRecordHeader(dataView: DataView): DataHeader;
  /**
   * parses a Blockette within the DataView.
   *
   * @param  dataView containing the data
   * @param  offset offset into the DataView to start
   * @param  length size in bytes of the Blockette
   * @param headerByteSwap true if byte swapping is needed
   * @returns Blockette instance
   */
  export function parseBlockette(dataView: DataView, offset: number, length: number, headerByteSwap: boolean): Blockette;
  /**
   * Represents a SEED Data Record, with header, blockettes and data.
   */
  export class DataRecord {
      header: DataHeader;
      data: DataView;
      constructor(header: DataHeader, data: DataView);
      /**
       * Decompresses the data , if the compression type is known.
       *
       * @returns decompressed data
       */
      decompress(): Int32Array | Float32Array | Float64Array;
      asEncodedDataSegment(): EncodedDataSegment;
      /**
       * Concatenates the net, station, loc and channel codes,
       * separated by the given seperator, or periods if not given.
       *
       * @param sep optional separater, defaults to .
       * @returns string of codes
       */
      codes(sep?: string): string;
  }
  /**
   * Represents the header part of the DataRecord, including all the actual
   *  fixed header plus fields pulled from a blockette 1000 if present.
   */
  export class DataHeader {
      seq: string;
      typeCode: number;
      continuationCode: number;
      staCode: string;
      locCode: string;
      chanCode: string;
      netCode: string;
      startBTime: BTime;
      numSamples: number;
      encoding: number;
      littleEndian: boolean;
      sampRateFac: number;
      sampRateMul: number;
      sampleRate: number;
      activityFlags: number;
      ioClockFlags: number;
      dataQualityFlags: number;
      numBlockettes: number;
      timeCorrection: number;
      dataOffset: number;
      blocketteOffset: number;
      recordSize: number;
      blocketteList: Array<Blockette>;
      startTime: DateTime;
      endTime: DateTime;
      constructor();
      toString(): string;
      /**
       * Calculates the sample rate in hertz from the sampRateFac and sampRateMul
       * parameters. This.sampleRate value is set to this value at construction.
       *
       * @returns sample rate
       */
      calcSampleRate(): number;
      /**
       * Calculates the time of the i-th sample in the record, zero based,
       *  so timeOfSample(0) is the start and timeOfSample(this.numSamples-1) is end.
       *
       * @param i sample index
       * @returns time at i-th sample as DateTime
       */
      timeOfSample(i: number): DateTime;
  }
  export class Blockette {
      type: number;
      body: DataView;
      constructor(type: number, body: DataView);
  }
  export class Blockette1000 extends Blockette {
      encoding: number;
      dataRecordLengthByte: number;
      wordOrder: number;
      constructor(type: number, body: DataView, encoding: number, dataRecordLengthByte: number, wordOrder: number);
  }
  export class Blockette1001 extends Blockette {
      timeQual: number;
      microsecond: number;
      frameCount: number;
      constructor(type: number, body: DataView, timeQual: number, microsecond: number, frameCount: number);
  }
  export class Blockette100 extends Blockette {
      sampleRate: number;
      flags: number;
      constructor(type: number, body: DataView, sampleRate: number, flags: number);
  }
  export function parseBTime(dataView: DataView, offset: number, byteSwap?: boolean): BTime;
  export class BTime {
      year: number;
      jday: number;
      hour: number;
      min: number;
      sec: number;
      tenthMilli: number;
      microsecond: number;
      length: number;
      constructor(year: number, jday: number, hour: number, min: number, sec: number, tenthMilli: number);
      toString(): string;
      /**
       * Converts this BTime to a luxon utc DateTime. Note DateTime's precision
       * is limited to milliseconds and leap seconds are not supported,
       * ie 60 seconds returns DateTime.invalid.
       *
       * @returns         BTime as a DateTime
       */
      toDateTime(): DateTime;
  }
  /**
   * Sanity checks on a BTime to see if a record might be in the wrong byte order
   * and so need to be byte swapped before parsing. Checks year betwee 1960 and 2055.
   *
   * @param   bTime  time
   * @returns        true is byte order appears to be wrong, false if it seems ok
   */
  export function checkByteSwap(bTime: BTime): boolean;
  /**
   * Determines if two DataRecords are contiguous, ie if the second starts
   * after the end of the first and the start time of the second is within
   * 1.5 times the sample period of the end of the first.
   *
   * @param dr1 first data record
   * @param dr2 seconds data record
   * @returns true if contiguous
   */
  export function areContiguous(dr1: DataRecord, dr2: DataRecord): boolean;
  /**
   * Concatentates a sequence of DataRecords into a single seismogram object.
   * Assumes that they are all contiguous and in order. Header values from the first
   * DataRecord are used.
   *
   * @param contig array of data records
   * @returns SeismogramSegment instance
   */
  export function createSeismogramSegment(contig: Array<DataRecord> | DataRecord): SeismogramSegment;
  /**
   * Merges data records into a Seismogram object, each of
   * which consists of SeismogramSegment objects
   * containing the data as EncodedDataSegment objects. DataRecords are
   * sorted by startTime.
   * This assumes all data records are from the same channel, byChannel
   * can be used first if multiple channels may be present. It also
   * assumes that new seismograms should not be created for gaps, for example
   * when getting data from the same channel for multiple earthquakes.
   *
   * @param drList array of data records
   * @returns Seismogram instance
   */
  export function merge(drList: Array<DataRecord>): Seismogram;
  /**
   * merges contiguous DataRecords into SeismogramSegments.
   *
   * @param drList array of data records
   * @returns array of SeismogramSegments for contiguous data
   */
  export function mergeSegments(drList: Array<DataRecord>): Array<SeismogramSegment>;
  /**
   * Splits a list of data records by channel code, returning a Map
   * with each NSLC string mapped to an array of data records.
   *
   * @param drList array of data records
   * @returns map of arrays of data records keyed by channel
   */
  export function byChannel(drList: Array<DataRecord>): Map<string, Array<DataRecord>>;
  /**
   * splits the DataRecords by channel and creates a single
   * SeismogramSegment for each contiguous window from each channel.
   *
   * @param   drList DataRecords array
   * @returns         Array of SeismogramSegment
   */
  export function seismogramSegmentPerChannel(drList: Array<DataRecord>): Array<SeismogramSegment>;
  /**
   * splits the DataRecords by channel and creates a single
   * Seismogram for each channel.
   *
   * @param   drList DataRecords array
   * @returns         Array of Seismogram
   */
  export function seismogramPerChannel(drList: Array<DataRecord>): Array<Seismogram>;

}
declare module 'seisplotjs/src/ms3ehtypes' {
  /**
   * This file was automatically generated by json-schema-to-typescript.
   * DO NOT MODIFY IT BY HAND. Instead, modify the source JSONSchema file,
   * and run json-schema-to-typescript to regenerate this file.
   */
  /**
   * Standard metadata about a seismogram.
   */
  export interface MS3ExtraHeader {
      bag?: BagExtraHeader;
      [k: string]: unknown;
  }
  /**
   * Common seismic values that are useful to embed in header
   */
  export interface BagExtraHeader {
      y?: Timeseries;
      ch?: Channel;
      ev?: Event;
      path?: Path;
      mark?: Marker[];
      [k: string]: unknown;
  }
  /**
   * timeseries units and processing state
   */
  export interface Timeseries {
      /**
       * si units of the timeseries amplitude, ex: count, m/s, m/s2
       */
      si: string;
      /**
       * basic processing type. Raw is unprocessed, gain has scalar gain/units, corrected transfer of frequency response, processed further userlevel processing
       */
      proc?: "raw" | "gain" | "corrected" | "synth" | "processed";
      [k: string]: unknown;
  }
  /**
   * recording station/channel
   */
  export interface Channel {
      /**
       * latitude in degrees
       */
      la: number;
      /**
       * longitude in degrees
       */
      lo: number;
      /**
       * elevation in meters
       */
      el?: number;
      /**
       * depth below surface in meters
       */
      dp?: number;
      /**
       * channel azimuth from north in degrees
       */
      az?: number;
      /**
       * channel dip from horizontal in degrees, up is -90
       */
      dip?: number;
      [k: string]: unknown;
  }
  /**
   * source earthquake
   */
  export interface Event {
      /**
       * public identifier for earthquake
       */
      id?: string;
      or?: Origin;
      mag?: Magnitude;
      mt?: MomentTensor;
      [k: string]: unknown;
  }
  /**
   * origin, location and time
   */
  export interface Origin {
      /**
       * origin time as ISO8601
       */
      tm: string;
      /**
       * latitude in degrees
       */
      la: number;
      /**
       * longitude in degrees
       */
      lo: number;
      /**
       * depth in kilometers
       */
      dp: number;
      [k: string]: unknown;
  }
  /**
   * magnitude
   */
  export interface Magnitude {
      /**
       * magnitude value
       */
      v: number;
      /**
       * magnitude type
       */
      t?: string;
      [k: string]: unknown;
  }
  /**
   * moment tensor
   */
  export interface MomentTensor {
      /**
       * scalar moment, M0
       */
      moment: number;
      /**
       * 3x3 symmetric tensor, 6 values, often scaled by the scalar moment
       */
      tensor?: unknown[];
      [k: string]: unknown;
  }
  /**
   * path between source and receiver
   */
  export interface Path {
      /**
       * great circle arc distance in degrees, for uses when only distance is needed
       */
      gcarc?: number;
      /**
       * great circle azimuth degrees from event to station, for uses when only distance is needed
       */
      az?: number;
      /**
       * great circle back azimuth in degrees back from station to event, for uses when only distance is needed
       */
      baz?: number;
      [k: string]: unknown;
  }
  export interface Marker {
      /**
       * marker time as ISO8601
       */
      tm: string;
      /**
       * name of the marker, like P
       */
      n: string;
      /**
       * type of marker, usual 'pk' for measurement/pick on data or 'md' for predicted from model
       */
      mtype?: string;
      amp?: number;
      desc?: string;
      [k: string]: unknown;
  }

}
declare module 'seisplotjs/src/mseed3' {
  import { EncodedDataSegment } from "seisplotjs/src/seedcodec";
  import { SeismogramSegment } from "seisplotjs/src/seismogramsegment";
  import { Seismogram, SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { DataRecord } from "seisplotjs/src/miniseed";
  import { DateTime } from "luxon";
  export type json_object = Record<string, unknown>;
  export const MINISEED_THREE_MIME = "application/vnd.fdsn.mseed3";
  /** const for unknown data version, 0 */
  export const UNKNOWN_DATA_VERSION = 0;
  /** const for offset to crc in record, 28 */
  export const CRC_OFFSET = 28;
  /** const for size of fixed header part of record, 40 */
  export const FIXED_HEADER_SIZE = 40;
  /** const for fdsn prefix for extra headers, FDSN */
  export const FDSN_PREFIX = "FDSN";
  /** const for little endian, true */
  export const LITTLE_ENDIAN = true;
  /** const for big endian, false */
  export const BIG_ENDIAN = false;
  export function toMSeed3(seis: Seismogram, extraHeaders?: Record<string, unknown>): Array<MSeed3Record>;
  /**
   * parse arrayBuffer into an array of MSeed3Records.
   *
   * @param arrayBuffer bytes to extract miniseed3 records from
   * @returns array of all miniseed3 records contained in the buffer
   */
  export function parseMSeed3Records(arrayBuffer: ArrayBuffer): Array<MSeed3Record>;
  /**
   * Represents a MSEED3 Data Record, with header, extras and data.
   *
   * @param header miniseed3 fixed record header
   * @param extraHeaders json compatible object with extra headers
   * @param rawData waveform data, in correct compression for value in header
   */
  export class MSeed3Record {
      header: MSeed3Header;
      extraHeaders: Record<string, unknown>;
      rawData: DataView;
      constructor(header: MSeed3Header, extraHeaders: Record<string, unknown>, rawData: DataView);
      /**
       * Parses an miniseed3 data record from a DataView.
       *
       * @param   dataView bytes to parse
       * @returns parsed record
       */
      static parseSingleDataRecord(dataView: DataView): MSeed3Record;
      /**
       * Calculates the byte size of the miniseed3 record to hold this data.
       * This should be called if the size is needed after modification
       * of the extraHeaders.
       *
       * @returns size in bytes
       */
      calcSize(): number;
      /**
       * Gets the byte size of the miniseed3 record to hold this data.
       * Note that unless calcSize() has been called, this may not
       * take into account modifications to the extra headers.
       *
       * @returns size in bytes
       */
      getSize(): number;
      /**
       * Decompresses the data , if the compression
       *  type is known
       *
       * @returns decompressed data as a typed array, usually Int32Array or Float32Array
       */
      decompress(): Int32Array | Float32Array | Float64Array;
      /**
       * Wraps data in an EncodedDataSegment for future decompression.
       *
       * @returns waveform data
       */
      asEncodedDataSegment(): EncodedDataSegment;
      /**
       * Just the header.identifier, included as codes() for compatiblility
       * with parsed miniseed2 data records.
       *
       * @returns string identifier
       */
      codes(): string;
      /**
       * Saves miniseed3 record into a DataView, recalculating crc.
       *
       * @param   dataView DataView to save into, must be large enough to hold the record.
       * @returns the number of bytes written to the DataView, can be used as offset
       * for writting the next record.
       */
      save(dataView: DataView): number;
      /**
       * Calculates crc by saving to a DataView, which sets the crc header to zero
       * and then calculates it based on the rest of the record.
       *
       * @returns         crc pulled from saved miniseed3 record
       */
      calcCrc(): number;
      toString(): string;
  }
  /**
   * Fixed header of an MSeed3 data record.
   */
  export class MSeed3Header {
      recordIndicator: string;
      formatVersion: number;
      flags: number;
      nanosecond: number;
      year: number;
      dayOfYear: number;
      hour: number;
      minute: number;
      second: number;
      encoding: number;
      sampleRateOrPeriod: number;
      numSamples: number;
      crc: number;
      publicationVersion: number;
      identifierLength: number;
      extraHeadersLength: number;
      identifier: string;
      extraHeaders: json_object;
      dataLength: number;
      constructor();
      /**
       * Parses an miniseed3 fixed header from a DataView.
       *
       * @param   dataView bytes to parse
       * @returns parsed header object
       */
      static createFromDataView(dataView: DataView): MSeed3Header;
      get start(): DateTime<boolean>;
      get end(): DateTime<boolean>;
      get sampleRate(): number;
      get samplePeriod(): number;
      /**
       * Calculates size of the fixed header including the variable
       * length identifier, but without the extra headers.
       *
       * @returns size in bytes of fixed header
       */
      getSize(): number;
      encodingName(): string;
      /**
       * Text representation of the miniseed3 header. This is modeled after
       * the output of mseed3-text from the mseed3-utils package from IRIS.
       *
       * @returns textual repersentation
       */
      toString(): string;
      /**
       * Start time in the format output by mseed3-utils from IRIS. Format is
       * yyyy,ooo,HH:mm:ss.SSSSSS
       *
       * @returns start time
       */
      startFieldsInUtilFormat(): string;
      /**
       * Converts start time header fields to ISO8601 time string. This will include
       * factional seconds to nanosecond precision.
       *
       * @param trimMicroNano trim to microsecond precision if nanos are 000
       * @returns iso start time
       */
      getStartFieldsAsISO(trimMicroNano?: boolean): string;
      /**
       * sets start time headers.
       *
       * @param starttime start as DateTime
       */
      setStart(starttime: DateTime): void;
      /**
       * Calculates time of the ith sample.
       *
       * @param   i sample number
       * @returns the time
       */
      timeOfSample(i: number): DateTime;
      /**
       * Writes to the given dataview.
       *
       * @param   dataView write buffer
       * @param   offset   offset within the buffer
       * @param   zeroCrc  optionally zero out the crc field in order to recalculate
       * @returns          new offset after this record
       */
      save(dataView: DataView, offset?: number, zeroCrc?: boolean): number;
      /**
       * Converts header start time to DateTime
       *
       * @returns         start time as DateTime
       */
      startAsDateTime(): DateTime;
  }
  /**
   * Parses extra headers as json.
   *
   * @param   dataView json bytes as DataView
   * @returns           json object
   */
  export function parseExtraHeaders(dataView: DataView): Record<string, unknown>;
  /**
   * Creates a string version of a number with zero prefix padding. For example
   * padZeros(5, 3) is 005.
   *
   * @param   val number to stringify
   * @param   len total length of string
   * @returns      zero padded string
   */
  export function padZeros(val: number, len: number): string;
  /**
   * creates a string from utf-8 bytes in a DataView.
   *
   * @param   dataView data bytes
   * @param   offset   offset to first byte to use
   * @param   length   number of bytes to convert
   * @returns           string resulting from utf-8 conversion
   */
  export function makeString(dataView: DataView, offset: number, length: number): string;
  /**
   * Checks if two miniseed3 records are (nearly) contiguous.
   *
   * @param   dr1 first record
   * @param   dr2 second record
   * @param   sampRatio tolerence expressed as ratio of sample period, default 1.5
   * @returns      true if contiguous
   */
  export function areContiguous(dr1: MSeed3Record, dr2: MSeed3Record, sampRatio?: number): boolean;
  /**
   * Concatentates a sequence of MSeed3 Records into a single seismogram object.
   * Assumes that they are all contiguous (no gaps or overlaps) and in order.
   * Header values from the first MSeed3 Record are used.
   *
   * @param contig array of miniseed3 records
   * @returns seismogram segment for the records
   */
  export function createSeismogramSegment(contig: Array<MSeed3Record>): SeismogramSegment;
  /**
   * Merges miniseed3 records into a Seismogram object, each of
   * which consists of SeismogramSegment objects
   * containing the data as EncodedDataSegment objects. DataRecords are
   * sorted by startTime.
   * This assumes all data records are from the same channel, byChannel
   * can be used first if multiple channels may be present. Gaps may be present.
   *
   * @param drList list of miniseed3 records to convert
   * @returns the seismogram
   */
  export function merge(drList: Array<MSeed3Record>): Seismogram;
  /**
   * merges contiguous MSeed3Record into SeismogramSegments.
   *
   * @param drList array of data records
   * @returns array of SeismogramSegments for contiguous data
   */
  export function mergeSegments(drList: Array<MSeed3Record>): Array<SeismogramSegment>;
  /**
   * splits a list of data records by channel identifier, returning an object
   * with each NSLC mapped to an array of data records.
   *
   * @param drList array of miniseed3 records
   * @returns map of channel id to array of miniseed3 records, possibly not contiguous
   */
  export function byChannel(drList: Array<MSeed3Record>): Map<string, Array<MSeed3Record>>;
  /**
   * splits the records by channel and creates a single
   * SeismogramSegment for each contiguous window from each channel.
   *
   * @param   drList MSeed3Records array
   * @returns         Array of SeismogramSegment
   */
  export function seismogramSegmentPerChannel(drList: Array<MSeed3Record>): Array<SeismogramSegment>;
  /**
   * splits the MSeed3Records by channel and creates a single
   * Seismogram for each channel.
   *
   * @param   drList MSeed3Records array
   * @returns         Map of code to Seismogram
   */
  export function seismogramPerChannel(drList: Array<MSeed3Record>): Array<Seismogram>;
  /**
   * splits the MSeed3Records by channel and creates a single
   * SeismogramDisplayData for each channel. BAG extra headers
   * are extracted and Quake and Markers are created.
   *
   * @param   drList MSeed3Records array
   * @returns         Map of code to Seismogram
   */
  export function sddPerChannel(drList: Array<MSeed3Record>): Array<SeismogramDisplayData>;
  /**
   * Convert array of Miniseed2 DataRecords into an array of MSeed3Records.
   *
   * @param   mseed2 array of DataRecords
   * @returns         array of MSeed3Records
   */
  export function convertMS2toMSeed3(mseed2: Array<DataRecord>): Array<MSeed3Record>;
  /**
   * Converts a single miniseed2 DataRecord into a single MSeed3Record.
   *
   * @param   ms2record Miniseed2 DataRecord to convert
   * @returns            MSeed3Record
   */
  export function convertMS2Record(ms2record: DataRecord): MSeed3Record;
  /**
   * Copy from https://github.com/ashi009/node-fast-crc32c/blob/master/impls/js_crc32c.js
   * and modify to use ArrayBuffer. Rename calculateCRC32C
   *
   * This code is a manual javascript translation of c code generated by
   * pycrc 0.7.1 (https://www.tty1.net/pycrc/). Command line used:
   * './pycrc.py --model=crc-32c --generate c --algorithm=table-driven'
   *
   * @param buf input data
   * @param initial starting value, from earlier data
   * @returns calculated crc32c value
   */
  export function calculateCRC32C(buf: ArrayBuffer | Uint8Array, initial?: number): number;
  /**
   * Convert crc as a number into a hex string.
   *
   * @param   crc crc as a number
   * @returns      hex representation
   */
  export function crcToHexString(crc: number): string;

}
declare module 'seisplotjs/src/mseed3eh' {
  import { FDSNSourceId } from "seisplotjs/src/fdsnsourceid";
  import { Quake } from "seisplotjs/src/quakeml";
  import { Channel } from "seisplotjs/src/stationxml";
  import { MS3ExtraHeader, Marker as EHMarker, Event as EHEvent, Origin as EHOrigin, Magnitude as EHMagnitude, Channel as EHChannel, Timeseries as EHTimeseries, BagExtraHeader as EHBag } from "seisplotjs/src/ms3ehtypes";
  import { MarkerType } from "seisplotjs/src/seismographmarker";
  export const STD_EH = "bag";
  export function ehToQuake(exHead: MS3ExtraHeader): Quake | null;
  export function quakeToEH(quake: Quake): EHEvent;
  export function channelToEH(channel: Channel): EHChannel;
  export function ehToChannel(exHead: MS3ExtraHeader, sid: FDSNSourceId): Channel | null;
  export function markerTypeFromEH(mtype: string): string;
  export function ehToMarkers(exHead: MS3ExtraHeader): Array<MarkerType>;
  export function extractBagEH(jsonEH: Record<string, unknown>): EHBag | null;
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v Bag station JSON object, usually from MSeed3 extra headers
   * @returns   true if matches expected structure
   */
  export function isValidBagChannelJsonEHType(v: unknown): v is EHChannel;
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v Bag event JSON object, usually from MSeed3 extra headers
   * @returns   true if matches expected structure
   */
  export function isValidBagEventJsonEHType(v: unknown): v is EHEvent;
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v Bag origin JSON object, usually from MSeed3 extra headers
   * @returns   true if matches expected structure
   */
  export function isValidBagOriginJsonEHType(v: unknown): v is EHOrigin;
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v Bag magnitude JSON object, usually from MSeed3 extra headers
   * @returns   true if matches expected structure
   */
  export function isValidBagMagJsonEHType(v: unknown): v is EHMagnitude;
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v Bag magnitude JSON object, usually from MSeed3 extra headers
   * @returns   true if matches expected structure
   */
  export function isValidBagPathJsonEHType(v: unknown): v is EHMagnitude;
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v Bag magnitude JSON object, usually from MSeed3 extra headers
   * @returns   true if matches expected structure
   */
  export function isValidBagMarkJsonEHType(v: unknown): v is EHMarker;
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v Bag magnitude JSON object, usually from MSeed3 extra headers
   * @returns   true if matches expected structure
   */
  export function isValidBagTimeseriesJsonEHType(v: unknown): v is EHTimeseries;
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v Bag JSON object, usually from MSeed3 extra headers
   * @returns   true if matches expected structure
   */
  export function isValidBagJsonEHType(v: unknown): v is EHBag;

}
declare module 'seisplotjs/src/mseedarchive' {
  import { DateTime, Duration } from "luxon";
  import * as miniseed from "seisplotjs/src/miniseed";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { Channel } from "seisplotjs/src/stationxml";
  export const Allowed_Flags: string[];
  /**
   * A web based connection to an archive of miniseed files
   * arranged based on a pattern using n, s, l, c, Y, j, H
   * for network, station, locid, channel, year, day of year
   * and hour. This is a subset of the options available within
   * the IRIS Ringserver MSeedArchive option, on which this
   * is based. Retrieved seismograms are cut from the larger
   * miniseed files retrieved via http(s), and so there is
   * wasted bandwidth to the server. On the other hand this
   * requires no extra software on the server side beyond
   * a directory structure with suitably small miniseed files.
   * Generally we find channel-hour is a reasonable size for
   * most seismic channels. The URL to needed files is
   * constructed by concatenating the rootUrl with the pattern
   * using a time range large enough to get all overlaps
   * based on the smallest sample rate per channel band code
   * and record size, which defaults to 512.
   *
   */
  export class MSeedArchive {
      _rootUrl: string;
      _pattern: string;
      _recordSize: number;
      _timeoutSec: number;
      constructor(rootUrl: string, pattern: string);
      get rootUrl(): string;
      get pattern(): string;
      get recordSize(): number;
      /**
       * checks pattern for allowed flags as not all that are supported
       * by ringserver are supported here. Must only include:
       * * n network code, white space removed
       * * s station code, white space removed
       * * l  location code, white space removed
       * * c  channel code, white space removed
       * * Y  year, 4 digits
       * * j  day of year, 3 digits zero padded
       * * H  hour, 2 digits zero padded
       *
       * @param p mseed archive pattern string
       * @returns true if all flags are allowed
       */
      checkPattern(p: string): boolean;
      /**
       * Loads seismograms from the remote miniseed archive via
       * http(s). Files downloaded include all that might overlap
       * the given time window based on record size,
       * the minimum sample rate
       * for the channel band code and the given time window.
       *
       * @param   channelTimeList request channels and time windows
       * @returns Promise to the same SeismogramDisplayData array, but with seismograms populated
       */
      loadSeismograms(channelTimeList: Array<SeismogramDisplayData>): Promise<Array<SeismogramDisplayData>>;
      /**
       * Loads miniseed records based on channel and time window.
       *
       * @param   channel   channel to request
       * @param   startTime start time
       * @param   endTime   end time
       * @returns Promise to array of miniseed records
       */
      loadDataForChannel(channel: Channel, startTime: DateTime, endTime: DateTime): Promise<Array<miniseed.DataRecord>>;
      /**
       * Loads miniseed records based on string channel codes.
       *
       * @param   net        network code
       * @param   sta        station code
       * @param   loc        location code
       * @param   chan       channel code
       * @param   startTime  start time
       * @param   endTime    end time
       * @param   sampleRate known sample rate for this channel
       * @returns             Promise to array of miniseed records
       */
      loadData(net: string, sta: string, loc: string, chan: string, startTime: DateTime, endTime: DateTime, sampleRate?: number): Promise<Array<miniseed.DataRecord>>;
      /**
       * Replaces codes from channel in base pattern.
       *
       * @param   net  string to replace '%n'
       * @param   sta  string to replace '%s'
       * @param   loc  string to replace '%l'
       * @param   chan string to replace '%c'
       * @returns       new string with channel replacements made
       */
      fillBasePattern(net: string, sta: string, loc: string, chan: string): string;
      /**
       * Replaces time entries ( %Y, %j, %H ) in pattern.
       *
       * @param   basePattern pattern to replace in
       * @param   t           DateTime in time
       * @returns              string with time replaces
       */
      fillTimePattern(basePattern: string, t: DateTime): string;
  }
  export function loadDataRecords(urlList: Array<string>, fetchInit?: RequestInit, timeoutSec?: number): Promise<Array<miniseed.DataRecord>>;
  /**
   * Gives the maximum sample rate for the channel, based on the
   * band code, first char, of the channel code.
   *
   * @param chan channel code like BHZ, only the first letter is used
   * @returns mimumum sample rate this could be
   */
  export function maxSampleRate(chan: string): number;
  /**
   * Gives the minimum sample rate for the channel, based on the
   * band code, first char, of the channel code.
   *
   * @param chan channel code like BHZ, only the first letter is used
   * @returns mimumum sample rate this could be
   */
  export function minSampleRate(chan: string): number;
  /**
   * Calculates the maximum time coverage for a single miniseed record
   * given the record size (usually 512 or 4096) and the sample rate (Hertz).
   * This assumes 40 bytes of header and maximum compression of 2 samples
   * per byte (4 bit per sample) which is the best Steim2.
   *
   * @param recordSize record size (usually 512 or 4096)
   * @param sampleRate sample rate of record
   * @returns maximum interval of time that a full record could cover when
   * compression is at its most efficient
   */
  export function maxTimeForRecord(recordSize: number, sampleRate: number): Duration;

}
declare module 'seisplotjs/src/oregondsputil' {
  import OregonDSPTop from "oregondsp";
  export const OregonDSP: typeof OregonDSPTop.com.oregondsp.signalProcessing;
  export const CDFT: typeof OregonDSPTop.com.oregondsp.signalProcessing.fft.CDFT;
  export const RDFT: typeof OregonDSPTop.com.oregondsp.signalProcessing.fft.RDFT;
  export const CenteredDifferentiator: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.CenteredDifferentiator;
  export const CenteredHilbertTransform: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.CenteredHilbertTransform;
  export const EquirippleBandpass: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.EquirippleBandpass;
  export const EquirippleFIRFilter: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.EquirippleFIRFilter;
  export const EquirippleHalfBand: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.EquirippleHalfBand;
  export const EquirippleHighpass: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.EquirippleHighpass;
  export const EquirippleLowpass: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.EquirippleLowpass;
  export const FIRTypeI: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.FIRTypeI;
  export const FIRTypeII: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.FIRTypeII;
  export const FIRTypeIII: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.FIRTypeIII;
  export const StaggeredDifferentiator: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.StaggeredDifferentiator;
  export const StaggeredHilbertTranform: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.equiripple.StaggeredHilbertTranform;
  export const ComplexAnalyticSignal: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.ComplexAnalyticSignal;
  export const Interpolator: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.Interpolator;
  export const OverlapAdd: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.fir.OverlapAdd;
  export const Allpass: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.Allpass;
  export const AnalogPrototype: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.AnalogPrototype;
  export const Butterworth: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.Butterworth;
  export const ChebyshevI: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.ChebyshevI;
  export const ChebyshevII: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.ChebyshevII;
  export const Complex: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.Complex;
  export const IIRFilter: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.IIRFilter;
  export const PassbandType: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.PassbandType;
  export const ThiranAllpass: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.ThiranAllpass;
  export const LOWPASS: OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.PassbandType & {
      readonly name: "LOWPASS";
      readonly ordinal: 0;
  };
  export const BANDPASS: OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.PassbandType & {
      readonly name: "BANDPASS";
      readonly ordinal: 1;
  };
  export const HIGHPASS: OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.PassbandType & {
      readonly name: "HIGHPASS";
      readonly ordinal: 2;
  };
  export const LagrangePolynomial: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.LagrangePolynomial;
  export const Polynomial: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.Polynomial;
  export const Rational: typeof OregonDSPTop.com.oregondsp.signalProcessing.filter.Rational;
  export const HammingWindow: typeof OregonDSPTop.com.oregondsp.signalProcessing.HammingWindow;
  export const HanningWindow: typeof OregonDSPTop.com.oregondsp.signalProcessing.HanningWindow;
  export const Sequence: typeof OregonDSPTop.com.oregondsp.signalProcessing.Sequence;
  export const Window: typeof OregonDSPTop.com.oregondsp.signalProcessing.Window;
  export function complexFromPolar(amp: number, phase: number): OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.Complex;
  export function createComplex(real: number, imag: number): OregonDSPTop.com.oregondsp.signalProcessing.filter.iir.Complex;

}
declare module 'seisplotjs/src/organizeddisplay' {
  import { AmplitudeScalable, TimeScalable } from "seisplotjs/src/scale";
  import { SeisPlotElement } from "seisplotjs/src/spelement";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  export const ORG_DISP_ITEM = "sp-organized-display-item";
  export const ORG_DISPLAY = "sp-organized-display";
  export const ORG_TYPE = "orgtype";
  export const PLOT_TYPE = "plottype";
  export const SEISMOGRAPH = "seismograph";
  export const SPECTRA = "amp_spectra";
  export const PARTICLE_MOTION = "particlemotion";
  export const MAP = "map";
  export const INFO = "info";
  export const QUAKE_TABLE = "quake_table";
  export const STATION_TABLE = "station_table";
  export class OrganizedDisplayItem extends SeisPlotElement {
      extras: Map<string, unknown>;
      constructor(seisData?: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      get plottype(): string;
      set plottype(val: string);
      static get observedAttributes(): string[];
      attributeChangedCallback(_name: string, _oldValue: unknown, _newValue: unknown): void;
      setExtra(key: string, value: unknown): void;
      hasExtra(key: string): boolean;
      getExtra(key: string): unknown;
      getContainedPlotElements(): Array<SeisPlotElement>;
      draw(): void;
  }
  export const WITH_INFO = "info";
  export const DEFAULT_WITH_INFO = "false";
  export const WITH_MAP = "map";
  export const DEFAULT_WITH_MAP = "false";
  export const WITH_TOOLS = "tools";
  export const DEFAULT_WITH_TOOLS = "true";
  export const SORT_BY = "sort";
  export const OVERLAY_BY = "overlay";
  export const OVERLAY_NONE = "none";
  export const OVERLAY_INDIVIDUAL = "individual";
  export const OVERLAY_VECTOR = "vector";
  export const OVERLAY_COMPONENT = "component";
  export const OVERLAY_STATION = "station";
  export const OVERLAY_ALL = "all";
  export const OVERLAY_FUNCTION = "function";
  export const TOOLS_HTML = "\n<details>\n  <summary>Tools</summary>\n  <form>\n    <fieldset class=\"plottype\">\n      <legend>Plot</legend>\n      <span>\n        <input type=\"checkbox\" name=\"with_map\" id=\"with_map\">\n        <label for=\"with_map\">map</label>\n      </span>\n      <span>\n        <input type=\"checkbox\" name=\"with_info\" id=\"with_info\">\n        <label for=\"with_info\">info</label>\n      </span>\n    </fieldset>\n    <fieldset class=\"overlay\">\n    <legend>Overlay Type</legend>\n    <span>\n      <input type=\"radio\" name=\"overlay\" id=\"overlay_individual\" value=\"individual\" checked>\n      <label for=\"overlay_individual\">individual</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"overlay\" id=\"overlay_vector\" value=\"vector\">\n      <label for=\"overlay_vector\">vector</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"overlay\" id=\"overlay_component\" value=\"component\">\n      <label for=\"overlay_component\">component</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"overlay\" id=\"overlay_station\" value=\"station\">\n      <label for=\"overlay_station\">station</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"overlay\" id=\"overlay_all\" value=\"all\">\n      <label for=\"overlay_all\">all</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"overlay\" id=\"overlay_none\" value=\"none\">\n      <label for=\"overlay_none\">none</label>\n    </span>\n  </fieldset>\n  <fieldset class=\"sort\">\n    <legend>Sort Type</legend>\n    <span>\n      <input type=\"radio\" name=\"sort\" id=\"sort_distance\" value=\"distance\">\n      <label for=\"sort_distance\">distance</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"sort\" id=\"sort_azimuth\" value=\"azimuth\">\n      <label for=\"sort_azimuth\">azimuth</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"sort\" id=\"sort_backazimuth\" value=\"backazimuth\">\n      <label for=\"sort_backazimuth\">back azimuth</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"sort\" id=\"sort_alphabetical\" value=\"alphabetical\">\n      <label for=\"sort_alphabetical\">alphabetical</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"sort\" id=\"sort_starttime\" value=\"starttime\">\n      <label for=\"sort_starttime\">starttime</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"sort\" id=\"sort_origin\" value=\"origin\">\n      <label for=\"sort_origin\">origin</label>\n    </span>\n    <span>\n      <input type=\"radio\" name=\"sort\" id=\"sort_none\" value=\"none\" checked>\n      <label for=\"sort_none\">none</label>\n    </span>\n  </fieldset>\n  </form>\n</details>\n";
  export class OrganizedDisplayTools extends SeisPlotElement {
      _organizedDisplay: OrganizedDisplay | null;
      constructor(seisData?: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      get organizedDisplay(): OrganizedDisplay | null;
      set organizedDisplay(orgdisp: OrganizedDisplay | null);
      initCheckboxes(orgdisp: OrganizedDisplay | null): void;
      draw(): void;
      wireComponents(): void;
  }
  export const ORG_DISP_TOOLS_ELEMENT = "sp-orgdisp-tools";
  export class OrganizedDisplay extends SeisPlotElement {
      constructor(seisData?: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      static get observedAttributes(): string[];
      getDisplayItems(): Array<OrganizedDisplayItem>;
      get orgtype(): string;
      set orgtype(val: string);
      get tools(): string;
      set tools(val: string);
      get map(): string;
      set map(val: string);
      get info(): string;
      set info(val: string);
      get overlayby(): string;
      set overlayby(val: string);
      get sortby(): string;
      set sortby(val: string);
      draw(): Promise<[TimeScalable[], AmplitudeScalable[]]> | undefined;
      drawTools(sortedData: Array<SeismogramDisplayData>): void;
      drawMap(sortedData: Array<SeismogramDisplayData>): void;
      drawInfo(sortedData: Array<SeismogramDisplayData>): void;
      attributeChangedCallback(name: string, oldValue: string, newValue: string): void;
  }
  export function getFromQueryParams(qParams: Record<string, unknown>, name: string, defaultValue?: string): string;
  export function individualDisplay(sddList: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig): Array<OrganizedDisplayItem>;
  export function mapAndIndividualDisplay(sddList: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig): Array<OrganizedDisplayItem>;
  export function overlayBySDDFunction(sddList: Array<SeismogramDisplayData>, key: string, sddFun: (arg0: SeismogramDisplayData) => string | number | null, seisConfig?: SeismographConfig): Array<OrganizedDisplayItem>;
  export function overlayByComponent(sddList: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig): Array<OrganizedDisplayItem>;
  export function overlayByStation(sddList: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig): Array<OrganizedDisplayItem>;
  export function overlayAll(sddList: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig): Array<OrganizedDisplayItem>;
  export function sortByKey(organized: Array<OrganizedDisplayItem>, key: string): Array<OrganizedDisplayItem>;
  /**
   * Groups seismic data into subarrays where members of each subarray are
   * from the same network/station, have the same band and gain/instrument code
   * and overlap in time. Note, in most cases the subarrays will have
   * 1, 2 or 3 elements, but this is not checked nor guaranteed.
   *
   * @param   sddList list of SeismogramDisplayData to split
   * @returns          array of array of data, organized by component of motion
   */
  export function groupComponentOfMotion(sddList: Array<SeismogramDisplayData>): Array<Array<SeismogramDisplayData>>;
  export function createAttribute(organized: Array<OrganizedDisplayItem>, key: string, valueFun: (arg0: OrganizedDisplayItem) => string | number | null): Array<OrganizedDisplayItem>;
  export function createPlots(organized: Array<OrganizedDisplayItem>, divElement: HTMLElement): void;

}
declare module 'seisplotjs/src/particlemotion' {
  import "d3-transition";
  import { Interval } from "luxon";
  import { SeisPlotElement } from "seisplotjs/src/spelement";
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { SeismogramSegment } from "seisplotjs/src/seismogramsegment";
  import type { HandlebarsInput } from "seisplotjs/src/axisutil";
  import type { Axis } from "d3-axis";
  import type { ScaleLinear, NumberValue as d3NumberValue } from "d3-scale";
  import type { Selection } from "d3-selection";
  export const DEFAULT_TITLE = "<tspan>{{#each seisDataList}}{{onlyChangesChannel ../seisDataList @index}} {{else}}No Data{{/each}}</tspan>";
  export const DEFAULT_XLABEL = "{{#each seisXData}}<tspan>{{this.channelCode}}</tspan> {{else}}No Data{{/each}}";
  export const DEFAULT_YLABEL = "{{#each seisYData}}<tspan>{{this.channelCode}}</tspan> {{else}}No Data{{/each}}";
  export const PARTICLE_MOTION_ELEMENT = "sp-particle-motion";
  export const particleMotion_css = "\n\n:host {\n  display: block;\n  min-height: 200px;\n  height: 100%;\n}\n\ndiv.wrapper {\n  min-height: 100px;\n  height: 100%;\n  width: 100%;\n}\n\nsvg {\n  height: 100%;\n  width: 100%;\n  min-height: 125px;\n  min-width: 125px;\n  z-index: 100;\n}\n\nsvg text.title {\n  font-size: larger;\n  font-weight: bold;\n  fill: black;\n  color: black;\n}\n\nsvg path.seispath {\n    stroke: skyblue;\n    fill: none;\n    stroke-width: 1px;\n}\n";
  export function createParticleMotionConfig(timeRange?: Interval | null, defaultSeisConfig?: SeismographConfig): SeismographConfig;
  /**
   * Particle motion plot.
   *
   * @param xSeisData x axis seismogram
   * @param ySeisData y axis seismogram
   * @param seismographConfig config, not all parameters are used in
   * particle motion plots. Can be null for defaults.
   */
  export class ParticleMotion extends SeisPlotElement {
      plotId: number;
      _xSeisData: Array<SeismogramDisplayData>;
      _ySeisData: Array<SeismogramDisplayData>;
      width: number;
      height: number;
      outerWidth: number;
      outerHeight: number;
      xScale: ScaleLinear<number, number, never>;
      xScaleRmean: ScaleLinear<number, number, never>;
      xAxis: Axis<d3NumberValue>;
      yScale: ScaleLinear<number, number, never>;
      yScaleRmean: ScaleLinear<number, number, never>;
      yAxis: Axis<d3NumberValue>;
      g: Selection<SVGGElement, unknown, null, undefined>;
      static _lastID: number;
      constructor(xSeisData?: Array<SeismogramDisplayData>, ySeisData?: Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      get xSeisData(): Array<SeismogramDisplayData>;
      set xSeisData(xsdd: Array<SeismogramDisplayData> | SeismogramDisplayData);
      get ySeisData(): Array<SeismogramDisplayData>;
      set ySeisData(ysdd: Array<SeismogramDisplayData> | SeismogramDisplayData);
      draw(): void;
      checkResize(): boolean;
      drawParticleMotion(): void;
      drawParticleMotionForSegment(lineG: Selection<SVGGElement, unknown, null, undefined>, segA: SeismogramSegment, segB: SeismogramSegment): void;
      drawAxis(): void;
      rescaleAxis(): void;
      calcScaleDomain(): void;
      calcWidthHeight(nOuterWidth: number, nOuterHeight: number): void;
      createHandlebarsInput(): HandlebarsInput;
  }

}
declare module 'seisplotjs/src/quakeml' {
  import { Station, Channel } from "seisplotjs/src/stationxml";
  import { DateTime } from "luxon";
  export const QML_NS = "http://quakeml.org/xmlns/quakeml/1.2";
  export const BED_NS = "http://quakeml.org/xmlns/bed/1.2";
  export const IRIS_NS = "http://service.iris.edu/fdsnws/event/1/";
  export const ANSS_NS = "http://anss.org/xmlns/event/0.1";
  export const ANSS_CATALOG_NS = "http://anss.org/xmlns/catalog/0.1";
  export const USGS_HOST = "earthquake.usgs.gov";
  export const UNKNOWN_MAG_TYPE = "unknown";
  export const UNKNOWN_PUBLIC_ID = "unknownId";
  export const FAKE_ORIGIN_TIME: DateTime<true> | DateTime<false>;
  export const FAKE_EMPTY_XML = "<?xml version=\"1.0\"?><q:quakeml xmlns=\"http://quakeml.org/xmlns/bed/1.2\" xmlns:q=\"http://quakeml.org/xmlns/quakeml/1.2\"><eventParameters publicID=\"quakeml:fake/empty\"></eventParameters></q:quakeml>";
  export const QUAKE_CLICK_EVENT = "quakeclick";
  /**
   * Utility function to create CustomEvent for clicking on a Quake, for example
   * in a map or table.
   *
   * @param  q          Quake clicked on
   * @param  mouseclick original mouse click Event
   * @returns           CustomEvent populated with quake field in detail.
   */
  export function createQuakeClickEvent(q: Quake, mouseclick: Event): CustomEvent;
  class BaseElement {
      publicId: string;
      comments: Comment[];
      creationInfo?: CreationInfo;
      protected populate(qml: Element): void;
  }
  /**
   * Represent a QuakeML EventParameters.
   */
  export class EventParameters extends BaseElement {
      eventList: Quake[];
      description?: string;
      /**
       * Parses a QuakeML event parameters xml element into an EventParameters object.
       *
       * @param eventParametersQML the event parameters xml Element
       * @param host optional source of the xml, helpful for parsing the eventid
       * @returns EventParameters instance
       */
      static createFromXml(eventParametersQML: Element, host?: string): EventParameters;
  }
  /**
   * Represent a QuakeML Event. Renamed to Quake as Event conflicts with
   * other uses in javascript.
   */
  export class Quake extends BaseElement {
      eventId: string | undefined;
      descriptionList: EventDescription[];
      amplitudeList: Array<Amplitude>;
      stationMagnitudeList: Array<StationMagnitude>;
      magnitudeList: Array<Magnitude>;
      originList: Array<Origin>;
      pickList: Array<Pick>;
      focalMechanismList: Array<FocalMechanism>;
      preferredOrigin?: Origin;
      preferredMagnitude?: Magnitude;
      preferredFocalMechanism?: FocalMechanism;
      type?: string;
      typeCertainty?: string;
      /**
       * Parses a QuakeML event xml element into a Quake object. Pass in
       * host=seisplotjs.fdsnevent.USGS_HOST for xml from the USGS service
       * in order to parse the eventid, otherwise this can be left out
       *
       * @param qml the event xml Element
       * @param host optional source of the xml, helpful for parsing the eventid
       * @returns QuakeML Quake(Event) object
       */
      static createFromXml(qml: Element, host?: string): Quake;
      /**
       * Extracts the EventId from a QuakeML element, guessing from one of several
       * incompatible (grumble grumble) formats.
       *
       * @param   qml Quake(Event) to extract from
       * @param   host optional source of the xml to help determine the event id style
       * @returns     Extracted Id, or "unknownEventId" if we can't figure it out
       */
      static extractEventId(qml: Element, host?: string): string;
      hasPreferredOrigin(): boolean;
      hasOrigin(): boolean;
      get origin(): Origin;
      hasPreferredMagnitude(): boolean;
      hasMagnitude(): boolean;
      get magnitude(): Magnitude;
      get time(): DateTime;
      get latitude(): number;
      get longitude(): number;
      get depth(): number;
      get depthKm(): number;
      get description(): string;
      get arrivals(): Array<Arrival>;
      get picks(): Array<Pick>;
      toString(): string;
  }
  /**
    Represents a QuakeML EventDescription.
   */
  export class EventDescription {
      text: string;
      type?: string;
      constructor(text: string);
      /**
       * Parses a QuakeML description xml element into a EventDescription object.
       *
       * @param descriptionQML the description xml Element
       * @returns EventDescription instance
       */
      static createFromXml(descriptionQML: Element): EventDescription;
      toString(): string;
  }
  /** Represents a QuakeML Amplitude. */
  export class Amplitude extends BaseElement {
      genericAmplitude: RealQuantity;
      type?: string;
      category?: string;
      unit?: string;
      methodID?: string;
      period?: RealQuantity;
      snr?: number;
      timeWindow?: TimeWindow;
      pick?: Pick;
      waveformID?: WaveformID;
      filterID?: string;
      scalingTime?: TimeQuantity;
      magnitudeHint?: string;
      evaluationMode?: string;
      evaluationStatus?: string;
      constructor(genericAmplitude: RealQuantity);
      /**
       * Parses a QuakeML amplitude xml element into an Amplitude object.
       *
       * @param amplitudeQML the amplitude xml Element
       * @param allPicks picks already extracted from the xml for linking arrivals with picks
       * @returns Amplitude instance
       */
      static createFromXml(amplitudeQML: Element, allPicks: Pick[]): Amplitude;
  }
  /** Represents a QuakeML StationMagnitude. */
  export class StationMagnitude extends BaseElement {
      origin: Origin;
      mag: RealQuantity;
      type?: string;
      amplitude?: Amplitude;
      methodID?: string;
      waveformID?: WaveformID;
      constructor(origin: Origin, mag: RealQuantity);
      /**
       * Parses a QuakeML station magnitude xml element into a StationMagnitude object.
       *
       * @param stationMagnitudeQML the station magnitude xml Element
       * @param allOrigins origins already extracted from the xml for linking station magnitudes with origins
       * @param allAmplitudes amplitudes already extracted from the xml for linking station magnitudes with amplitudes
       * @returns StationMagnitude instance
       */
      static createFromXml(stationMagnitudeQML: Element, allOrigins: Origin[], allAmplitudes: Amplitude[]): StationMagnitude;
  }
  /** Represents a QuakeML TimeWindow. */
  export class TimeWindow {
      begin: number;
      end: number;
      reference: DateTime;
      constructor(begin: number, end: number, reference: DateTime);
      /**
       * Parses a QuakeML time window xml element into a TimeWindow object.
       *
       * @param timeWindowQML the time window xml Element
       * @returns TimeWindow instance
       */
      static createFromXml(timeWindowQML: Element): TimeWindow;
  }
  /** Represents a QuakeML Origin. */
  export class Origin extends BaseElement {
      compositeTimes: Array<CompositeTime>;
      originUncertainty?: OriginUncertainty;
      arrivalList: Array<Arrival>;
      timeQuantity: TimeQuantity;
      latitudeQuantity: RealQuantity;
      longitudeQuantity: RealQuantity;
      depthQuantity?: RealQuantity;
      depthType?: string;
      timeFixed?: boolean;
      epicenterFixed?: boolean;
      referenceSystemID?: string;
      methodID?: string;
      earthModelID?: string;
      quality?: OriginQuality;
      type?: string;
      region?: string;
      evaluationMode?: string;
      evaluationStatus?: string;
      constructor(time: TimeQuantity | DateTime, latitude: RealQuantity | number, longitude: RealQuantity | number);
      /**
       * Parses a QuakeML origin xml element into a Origin object.
       *
       * @param qml the origin xml Element
       * @param allPicks picks already extracted from the xml for linking arrivals with picks
       * @returns Origin instance
       */
      static createFromXml(qml: Element, allPicks: Array<Pick>): Origin;
      toString(): string;
      get time(): DateTime;
      set time(t: TimeQuantity | DateTime);
      get latitude(): number;
      set latitude(lat: RealQuantity | number);
      get longitude(): number;
      set longitude(lon: RealQuantity | number);
      get depthKm(): number;
      get depth(): number;
      set depth(depth: RealQuantity | number);
      get arrivals(): Array<Arrival>;
  }
  /** Represents a QuakeML CompositeTime. */
  export class CompositeTime {
      year?: IntegerQuantity;
      month?: IntegerQuantity;
      day?: IntegerQuantity;
      hour?: IntegerQuantity;
      minute?: IntegerQuantity;
      second?: RealQuantity;
      /**
       * Parses a QuakeML composite time xml element into an CompositeTime object.
       *
       * @param qml the composite time xml Element
       * @returns CompositeTime instance
       */
      static createFromXml(qml: Element): CompositeTime;
  }
  /** Represents a QuakeML OriginUncertainty. */
  export class OriginUncertainty {
      horizontalUncertainty?: number;
      minHorizontalUncertainty?: number;
      maxHorizontalUncertainty?: number;
      azimuthMaxHorizontalUncertainty?: number;
      confidenceEllipsoid?: ConfidenceEllipsoid;
      preferredDescription?: string;
      confidenceLevel?: number;
      /**
       * Parses a QuakeML origin uncertainty xml element into an OriginUncertainty object.
       *
       * @param qml the origin uncertainty xml Element
       * @returns OriginUncertainty instance
       */
      static createFromXml(qml: Element): OriginUncertainty;
  }
  /** Represents a QuakeML ConfidenceEllipsoid. */
  export class ConfidenceEllipsoid {
      semiMajorAxisLength: number;
      semiMinorAxisLength: number;
      semiIntermediateAxisLength: number;
      majorAxisPlunge: number;
      majorAxisAzimuth: number;
      majorAxisRotation: number;
      constructor(semiMajorAxisLength: number, semiMinorAxisLength: number, semiIntermediateAxisLength: number, majorAxisPlunge: number, majorAxisAzimuth: number, majorAxisRotation: number);
      /**
       * Parses a QuakeML confidence ellipsoid xml element into an ConfidenceEllipsoid object.
       *
       * @param qml the confidence ellipsoid xml Element
       * @returns ConfidenceEllipsoid instance
       */
      static createFromXml(qml: Element): ConfidenceEllipsoid;
  }
  /** Represents a QuakeML OriginQuality. */
  export class OriginQuality {
      associatedPhaseCount?: number;
      usedPhaseCount?: number;
      associatedStationCount?: number;
      usedStationCount?: number;
      depthPhaseCount?: number;
      standardError?: number;
      azimuthalGap?: number;
      secondaryAzimuthalGap?: number;
      groundTruthLevel?: string;
      maximumDistance?: number;
      minimumDistance?: number;
      medianDistance?: number;
      /**
       * Parses a QuakeML origin quality xml element into an OriginQuality object.
       *
       * @param qml the origin quality xml Element
       * @returns OriginQuality instance
       */
      static createFromXml(qml: Element): OriginQuality;
  }
  /**
    Represents a QuakeML Magnitude.
   */
  export class Magnitude extends BaseElement {
      stationMagnitudeContributions: StationMagnitudeContribution[];
      magQuantity: RealQuantity;
      type?: string;
      origin?: Origin;
      methodID?: string;
      stationCount?: number;
      azimuthalGap?: number;
      evaluationMode?: string;
      evaluationStatus?: string;
      constructor(mag: RealQuantity | number, type?: string);
      /**
       * Parses a QuakeML magnitude xml element into a Magnitude object.
       *
       * @param qml the magnitude xml Element
       * @param allOrigins origins already extracted from the xml for linking magnitudes with origins
       * @param allStationMagnitudes station magnitudes already extracted from the xml
       * @returns Magnitude instance
       */
      static createFromXml(qml: Element, allOrigins: Origin[], allStationMagnitudes: StationMagnitude[]): Magnitude;
      toString(): string;
      get mag(): number;
      set mag(value: RealQuantity | number);
  }
  /**
    Represents a QuakeML StationMagnitudeContribution.
   */
  export class StationMagnitudeContribution {
      stationMagnitude: StationMagnitude;
      residual?: number;
      weight?: number;
      constructor(stationMagnitude: StationMagnitude);
      /**
       * Parses a QuakeML station magnitude contribution xml element into a StationMagnitudeContribution object.
       *
       * @param qml the station magnitude contribution xml Element
       * @param allStationMagnitudes station magnitudes already extracted from the xml for linking station magnitudes with station magnitude contributions
       * @returns StationMagnitudeContribution instance
       */
      static createFromXml(qml: Element, allStationMagnitudes: Array<StationMagnitude>): StationMagnitudeContribution;
  }
  /**
    Represents a QuakeML Arrival, a combination of a Pick with a phase name.
   */
  export class Arrival extends BaseElement {
      phase: string;
      pick: Pick;
      timeCorrection?: number;
      azimuth?: number;
      distance?: number;
      takeoffAngle?: RealQuantity;
      timeResidual?: number;
      horizontalSlownessResidual?: number;
      backazimuthResidual?: number;
      timeWeight?: number;
      horizontalSlownessWeight?: number;
      backazimuthWeight?: number;
      earthModelID?: string;
      constructor(phase: string, pick: Pick);
      /**
       * Parses a QuakeML arrival xml element into a Arrival object.
       *
       * @param arrivalQML the arrival xml Element
       * @param allPicks picks already extracted from the xml for linking arrivals with picks
       * @returns Arrival instance
       */
      static createFromXml(arrivalQML: Element, allPicks: Array<Pick>): Arrival;
  }
  /**
    Represents a QuakeML Pick.
   */
  export class Pick extends BaseElement {
      timeQuantity: TimeQuantity;
      waveformID: WaveformID;
      filterID?: string;
      methodID?: string;
      horizontalSlowness?: RealQuantity;
      backazimuth?: RealQuantity;
      slownessMethodID?: string;
      onset?: string;
      phaseHint?: string;
      polarity?: string;
      evaluationMode?: string;
      evaluationStatus?: string;
      constructor(time: TimeQuantity | DateTime, waveformID: WaveformID);
      get time(): DateTime;
      set time(t: Quantity<DateTime> | DateTime);
      /**
       * Parses a QuakeML pick xml element into a Pick object.
       *
       * @param pickQML the pick xml Element
       * @returns Pick instance
       */
      static createFromXml(pickQML: Element): Pick;
      get networkCode(): string;
      get stationCode(): string;
      get locationCode(): string;
      get channelCode(): string;
      isAtStation(station: Station): boolean;
      isOnChannel(channel: Channel): boolean;
      toString(): string;
  }
  /**
    Represents a QuakeML Focal Mechanism.
   */
  export class FocalMechanism extends BaseElement {
      waveformIDList: WaveformID[];
      momentTensorList: MomentTensor[];
      triggeringOrigin?: Origin;
      nodalPlanes?: NodalPlanes;
      principalAxes?: PrincipalAxes;
      azimuthalGap?: number;
      stationPolarityCount?: number;
      misfit?: number;
      stationDistributionRatio?: number;
      methodID?: string;
      evaluationMode?: string;
      evaluationStatus?: string;
      /**
       * Parses a QuakeML focal mechanism xml element into a FocalMechanism object.
       *
       * @param focalMechQML the focal mechanism xml Element
       * @param allOrigins origins already extracted from the xml for linking focal mechanisms with origins
       * @param allMagnitudes magnitudes already extracted from the xml for linking moment tensors with magnitudes
       * @returns FocalMechanism instance
       */
      static createFromXml(focalMechQML: Element, allOrigins: Origin[], allMagnitudes: Magnitude[]): FocalMechanism;
  }
  /**
    Represents a QuakeML NodalPlanes.
   */
  export class NodalPlanes {
      nodalPlane1?: NodalPlane;
      nodalPlane2?: NodalPlane;
      preferredPlane?: number;
      /**
       * Parses a QuakeML nodal planes xml element into a NodalPlanes object.
       *
       * @param nodalPlanesQML the nodal planes xml Element
       * @returns NodalPlanes instance
       */
      static createFromXml(nodalPlanesQML: Element): NodalPlanes;
  }
  /**
    Represents a QuakeML NodalPlane.
   */
  export class NodalPlane {
      strike: RealQuantity;
      dip: RealQuantity;
      rake: RealQuantity;
      constructor(strike: RealQuantity, dip: RealQuantity, rake: RealQuantity);
      /**
       * Parses a QuakeML nodal plane xml element into a NodalPlane object.
       *
       * @param nodalPlaneQML the nodal plane xml Element
       * @returns NodalPlane instance
       */
      static createFromXml(nodalPlaneQML: Element): NodalPlane;
  }
  /**
    Represents a QuakeML PrincipalAxes.
   */
  export class PrincipalAxes {
      tAxis: Axis;
      pAxis: Axis;
      nAxis?: Axis;
      constructor(tAxis: Axis, pAxis: Axis);
      /**
       * Parses a QuakeML princpalAxes element into a PrincipalAxes object.
       *
       * @param princpalAxesQML the princpalAxes xml Element
       * @returns PrincipalAxes instance
       */
      static createFromXml(princpalAxesQML: Element): PrincipalAxes;
  }
  /**
    Represents a QuakeML Axis.
   */
  export class Axis {
      azimuth: RealQuantity;
      plunge: RealQuantity;
      length: RealQuantity;
      constructor(azimuth: RealQuantity, plunge: RealQuantity, length: RealQuantity);
      /**
       * Parses a QuakeML axis xml element into a Axis object.
       *
       * @param axisQML the axis xml Element
       * @returns Axis instance
       */
      static createFromXml(axisQML: Element): Axis;
  }
  /**
    Represents a QuakeML MomentTensor.
   */
  export class MomentTensor extends BaseElement {
      dataUsedList: DataUsed[];
      derivedOrigin: Origin;
      momentMagnitude?: Magnitude;
      scalarMoment?: RealQuantity;
      tensor?: Tensor;
      variance?: number;
      varianceReduction?: number;
      doubleCouple?: number;
      clvd?: number;
      iso?: number;
      greensFunctionID?: string;
      filterID?: string;
      sourceTimeFunction?: SourceTimeFunction;
      methodID?: string;
      category?: string;
      inversionType?: string;
      constructor(derivedOrigin: Origin);
      /**
       * Parses a QuakeML momentTensor xml element into a MomentTensor object.
       *
       * @param momentTensorQML the momentTensor xml Element
       * @param allOrigins origins already extracted from the xml for linking moment tensors with origins
       * @param allMagnitudes magnitudes already extracted from the xml for linking moment tensors with magnitudes
       * @returns MomentTensor instance
       */
      static createFromXml(momentTensorQML: Element, allOrigins: Origin[], allMagnitudes: Magnitude[]): MomentTensor;
  }
  /**
    Represents a QuakeML Tensor.
   */
  export class Tensor {
      Mrr: RealQuantity;
      Mtt: RealQuantity;
      Mpp: RealQuantity;
      Mrt: RealQuantity;
      Mrp: RealQuantity;
      Mtp: RealQuantity;
      constructor(Mrr: RealQuantity, Mtt: RealQuantity, Mpp: RealQuantity, Mrt: RealQuantity, Mrp: RealQuantity, Mtp: RealQuantity);
      /**
       * Parses a QuakeML tensor xml element into a Tensor object.
       *
       * @param tensorQML the tensor xml Element
       * @returns Tensor instance
       */
      static createFromXml(tensorQML: Element): Tensor;
  }
  /**
    Represents a QuakeML SourceTimeFunction.
   */
  export class SourceTimeFunction {
      type: string;
      duration: number;
      riseTime?: number;
      decayTime?: number;
      constructor(type: string, duration: number);
      /**
       * Parses a QuakeML sourceTimeFunction xml element into a SourceTimeFunction object.
       *
       * @param sourceTimeFunctionQML the sourceTimeFunction xml Element
       * @returns SourceTimeFunction instance
       */
      static createFromXml(sourceTimeFunctionQML: Element): SourceTimeFunction;
  }
  /**
    Represents a QuakeML DataUsed.
   */
  export class DataUsed {
      waveType: string;
      stationCount?: number;
      componentCount?: number;
      shortestPeriod?: number;
      longestPeriod?: number;
      constructor(waveType: string);
      /**
       * Parses a QuakeML dataUsed xml element into a DataUsed object.
       *
       * @param dataUsedQML the dataUsed xml Element
       * @returns SourceTimeFunction instance
       */
      static createFromXml(dataUsedQML: Element): DataUsed;
  }
  /**
    Represents a QuakeML WaveformID.
   */
  export class WaveformID {
      networkCode: string;
      stationCode: string;
      channelCode?: string;
      locationCode?: string;
      constructor(networkCode: string, stationCode: string);
      /**
       * Parses a QuakeML waveform ID xml element into a WaveformID object.
       *
       * @param waveformQML the waveform ID xml Element
       * @returns WaveformID instance
       */
      static createFromXml(waveformQML: Element): WaveformID;
      toString(): string;
  }
  export class Quantity<T> {
      value: T;
      uncertainty?: number;
      lowerUncertainty?: number;
      upperUncertainty?: number;
      confidenceLevel?: number;
      constructor(value: T);
      /**
       * Parses a QuakeML quantity xml element into a Quantity object.
       *
       * @param quantityQML the quantity xml Element
       * @param grab a callback to obtain the value
       * @param grabUncertainty a callback to obtain the uncertainties
       * @returns Quantity instance
       */
      static _createFromXml<T>(quantityQML: Element, grab: (xml: Element | null | void, tagName: string) => T | undefined, grabUncertainty: (xml: Element | null | void, tagName: string) => number | undefined): Quantity<T>;
      /**
       * Parses a QuakeML real quantity xml element into a RealQuantity object.
       *
       * @param realQuantityQML the real quantity xml Element
       * @returns RealQuantity instance
       */
      static createRealQuantityFromXml(realQuantityQML: Element): RealQuantity;
      /**
       * Parses a QuakeML integer quantity xml element into a RealQuantity object.
       *
       * @param integerQuantityQML the integer quantity xml Element
       * @returns IntegerQuantity instance
       */
      static createIntegerQuantityFromXml(integerQuantityQML: Element): IntegerQuantity;
      /**
       * Parses a QuakeML time quantity xml element into a TimeQuantity object.
       *
       * @param timeQuantityQML the time quantity xml Element
       * @returns TimeQuantity instance
       */
      static createTimeQuantityFromXml(timeQuantityQML: Element): TimeQuantity;
  }
  export type IntegerQuantity = Quantity<number>;
  export type RealQuantity = Quantity<number>;
  export type TimeQuantity = Quantity<DateTime>;
  /**
    Represents a QuakeML comment.
   */
  export class Comment {
      text: string;
      creationInfo?: CreationInfo;
      constructor(text: string);
      /**
       * Parses a QuakeML comment xml element into a Comment object.
       *
       * @param commentQML the comment xml Element
       * @returns Comment instance
       */
      static createFromXml(commentQML: Element): Comment;
  }
  export class CreationInfo {
      agencyID?: string;
      agencyURI?: string;
      author?: string;
      authorURI?: string;
      creationTime?: DateTime;
      version?: string;
      /**
       * Parses a QuakeML creation info xml element into a CreationInfo object.
       *
       * @param creationInfoQML the creation info xml Element
       * @returns CreationInfo instance
       */
      static createFromXml(creationInfoQML: Element): CreationInfo;
  }
  /**
   * Parses a QuakeML xml document into seisplotjs objects
   *
   *  @param rawXml the xml Document to parse
   *  @param host optional source of the xml, helpful for parsing the eventid
   *  @returns EventParameters object
   */
  export function parseQuakeML(rawXml: Document, host?: string): EventParameters;
  export function createQuakeFromValues(publicId: string, time: DateTime, latitude: number, longitude: number, depth_meter: number): Quake;
  /**
   * Fetches and parses QuakeML from a URL. This can be used in instances where
   * a static quakeML file is available on a web site instead of via a web
   * service with query paramters.
   * @param  url          the url to download from
   * @param  timeoutSec   timeout seconds in case of failed connection
   * @param nodata        nodata http error code
   * @returns             Promise to parsed quakeML as an EventParameters object
   */
  export function fetchQuakeML(url: string | URL, timeoutSec?: number, nodata?: number): Promise<EventParameters>;
  export const parseUtil: {
      _grabFirstEl: (xml: Element | null | void, tagName: string) => Element | undefined;
      _grabFirstElNS: (xml: Element | null | void, namespace: string, tagName: string) => Element | undefined;
      _grabFirstElText: (xml: Element | null | void, tagName: string) => string | undefined;
      _grabFirstElFloat: (xml: Element | null | void, tagName: string) => number | undefined;
      _grabFirstElInt: (xml: Element | null | void, tagName: string) => number | undefined;
      _grabAttribute: (xml: Element | null | void, tagName: string) => string | undefined;
      _requireAttribute: (xml: Element | null | void, tagName: string) => string;
      _grabAttributeNS: (xml: Element | null | void, namespace: string, tagName: string) => string | undefined;
  };
  export {};

}
declare module 'seisplotjs/src/ringserverweb' {
  import { DateTime, Duration } from "luxon";
  import { NslcId } from "seisplotjs/src/fdsnsourceid";
  export const SEEDLINK_PATH = "/seedlink";
  export const DATALINK_PATH = "/datalink";
  export type RingserverVersion = {
      ringserverVersion: string;
      serverId: string;
  };
  export type StreamsResult = {
      accessTime: DateTime;
      streams: Array<StreamStat>;
  };
  export const IRIS_HOST = "rtserve.iris.washington.edu";
  /**
   * Web connection to a Ringserver.
   *
   *
   * @param host optional host to connect to, defaults to IRIS. This maybe a full URL.
   * @param port optional host to connect to, defaults to 80
   */
  export class RingserverConnection {
      /** @private */
      _host: string;
      /** @private */
      _port: number;
      /** @private */
      _prefix: string;
      /** @private */
      _timeoutSec: number;
      constructor(host?: string, port?: number);
      /**
       * Gets/Sets the remote host to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      host(value?: string): RingserverConnection;
      getHost(): string;
      /**
       * Gets/Sets the remote port to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      port(value?: number): number | RingserverConnection;
      getPort(): number | undefined;
      /**
       * Get/Set the timeout in seconds for the request. Default is 30.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      timeout(value?: number): RingserverConnection;
      getTimeout(): number | undefined;
      /**
       * Pulls id result from ringserver /id parsed into an object with
       * 'ringserverVersion' and 'serverId' fields.
       *
       * @returns Result as a Promise.
       */
      pullId(): Promise<RingserverVersion>;
      /**
       *  Use numeric level (1-6) to pull just IDs from ringserver.
       *  In a default ringserver,
       *  level=1 would return all networks like
       *  CO
       *  and level=2 would return all stations like
       *  CO_JSC
       *  If level is falsy/missing, level=6 is used.
       *  The optional matchPattern is a regular expression, so for example
       *  '.+_JSC_00_HH.' would get all HH? channels from any station name JSC.
       *
       * @param level 1-6
       * @param matchPattern regular expression to match
       * @returns Result as a Promise.
       */
      pullStreamIds(level: number, matchPattern: string): Promise<Array<string>>;
      /**
       * Pull streams, including start and end times, from the ringserver.
       * The optional matchPattern is a regular expression, so for example
       * '.+_JSC_00_HH.' would get all HH? channels from any station name JSC.
       * Result returned is an Promise.
       *
       * @param matchPattern regular expression to match
       * @returns promise to object with 'accessTime' as a DateTime
       * and 'streams' as an array of StreamStat objects.
       */
      pullStreams(matchPattern: string): Promise<StreamsResult>;
      /**
       * Utility method to pull raw result from ringserver url.
       * Result returned is an Promise.
       *
       * @param url the url
       * @returns promise to string result
       */
      pullRaw(url: string): Promise<string>;
      getDataLinkURL(): string;
      getSeedLinkURL(): string;
      /**
       * Forms base url from protocol, host and port.
       *
       * @returns the string url
       */
      formBaseURL(): string;
      /**
       * Forms the ringserver id url.
       *
       * @returns the id url
       */
      formIdURL(): string;
      /**
       * Forms the ringserver streams url using the query parameters.
       *
       * @param queryParams optional string of query parameters
       * @returns the streams url
       */
      formStreamsURL(queryParams?: string): string;
      /**
       * Forms the ringserver stream ids url using the query parameters.
       *
       * @param queryParams optional string of query parameters
       * @returns the stream ids url
       */
      formStreamIdsURL(queryParams: string): string;
  }
  /**
   * Extract one StreamStat per station from an array of channel level
   * StreamStats. The start and end times are the min and max times for all
   * the channels within the station. Can be used to get most time of most
   * recent packet from the stations to give an idea of current latency.
   *
   * @param   streams array of channel level StreamStats
   * @returns array of station level StreamStats
   */
  export function stationsFromStreams(streams: Array<StreamStat>): Array<StreamStat>;
  export class NslcWithType {
      type: string;
      nslc: NslcId;
      constructor(type: string, nslc: NslcId);
  }
  /**
   * Split type, networkCode, stationCode, locationCode and channelCode
   * from a ringserver id formatted like net_sta_loc_chan/type
   *
   * @param   id id string to split
   * @returns  object with the split fields
   */
  export function nslcSplit(id: string): NslcWithType;
  /**
   * Object to hold start and end times for a key, usually channel or station.
   *
   * @param key id, usually station or channel
   * @param start start time
   * @param end end time
   */
  export class StreamStat {
      key: string;
      startRaw: string;
      endRaw: string;
      start: DateTime;
      end: DateTime;
      constructor(key: string, start: string, end: string);
      /**
       * Calculates latency time difference between last packet and current time.
       *
       * @param accessTime time latency is calculated relative to
       * @returns latency
       */
      calcLatency(accessTime?: DateTime): Duration;
  }

}
declare module 'seisplotjs/src/sacpolezero' {
  import { Complex } from "seisplotjs/src/oregondsputil";
  /**
   * SAC style response file. This contains poles and zeros to represent the
   * analog stage, plus a constant for the overall gain. See
   * seisplotjs.transfer.convertToSacPoleZero() for conversion from a
   * StationXML response to SacPoleZero.
   */
  export class SacPoleZero {
      /**
       * Complex poles
       */
      poles: Array<InstanceType<typeof Complex>>;
      /**
       * Complex zeros
       */
      zeros: Array<InstanceType<typeof Complex>>;
      /**
       * Scalar overall gain
       */
      constant: number;
      /** number of zeros added to convert to displacement, for debugging */
      gamma: number | undefined;
      /** hertz/radian factor, for debugging */
      mulFactor: number;
      /** sensitivity accounting for gamma, for debugging */
      sd: number;
      /** normalization factor for poles and zeros accounting for gamma, for debugging */
      A0: number;
      constructor(poles: Array<InstanceType<typeof Complex>>, zeros: Array<InstanceType<typeof Complex>>, constant: number);
      toString(): string;
      evalPoleZeroInverse(freq: number): InstanceType<typeof Complex>;
      trimZeros(gamma: number): void;
      toText(): string;
      /**
       * Caclulates the frequency response from the given poles and zeros.
       *
       * @param freqs frequencies to compute
       * @returns  frequency response
       */
      calcForDisplay(freqs: Array<number>): Array<InstanceType<typeof Complex>>;
      /**
       * Parses a string in sac polezero format into a SacPoleZero.
       *
       * @param data string to parse
       * @returns SacPoleZero instance
       */
      static parse(data: string): SacPoleZero;
  }
  export function geomspace(start: number, stop: number, num: number): Array<number>;
  export function logspace(start: number, stop: number, num: number): Array<number>;
  export function linspace(start: number, stop: number, num: number): Array<number>;

}
declare module 'seisplotjs/src/scale' {
  import { Duration } from "luxon";
  /** enum for amplitude modes, RAW, ZERO, MINMAX, MEAN */
  export enum AMPLITUDE_MODE {
      Raw = "raw",// raw values, no centering
      Zero = "zero",// same as Raw, but also includes zero
      MinMax = "minmax",// centered on midpoint of min-max
      Mean = "mean"
  }
  export class MinMaxable {
      min: number;
      max: number;
      constructor(min: number, max: number);
      get middle(): number;
      get halfWidth(): number;
      get fullWidth(): number;
      union(omm?: MinMaxable): MinMaxable;
      expandPercentage(percent: number): MinMaxable;
      /**
       * This as a d3 style 2 element array.
       *
       * @returns length 2 array of min then max
       */
      asArray(): [number, number];
      toString(): string;
      /**
       * Create MinMaxable from a d3 style two element array.
       *
       * @param  minmax  array of min then max
       * @returns       new MinMaxable
       */
      static fromArray(minmax: number[]): MinMaxable;
      static fromMiddleHalfWidth(mid: number, halfWidth: number): MinMaxable;
  }
  export class AmplitudeScalable {
      minMax: MinMaxable;
      constructor(minMax?: MinMaxable);
      notifyAmplitudeChange(_middle: number, _halfWidth: number): void;
      get middle(): number;
      get halfWidth(): number;
      get fullWidth(): number;
      get min(): number;
      get max(): number;
      toString(): string;
  }
  export class TimeScalable {
      alignmentTimeOffset: Duration;
      duration: Duration;
      constructor(alignmentTimeOffset: Duration, duration: Duration);
      notifyTimeRangeChange(_alignmentTimeOffset: Duration, _duration: Duration): void;
  }
  /**
   * Links amplitude scales across multiple seismographs, respecting doRmean.
   *
   * @param graphList optional list of AmplitudeScalable to link
   */
  export class LinkedAmplitudeScale {
      /**
       * @private
       */
      _graphSet: Set<AmplitudeScalable>;
      _halfWidth: number;
      _recalcTimeoutID: ReturnType<typeof setTimeout> | null;
      _scaleId: number;
      constructor(graphList?: Array<AmplitudeScalable>);
      get halfWidth(): number;
      set halfWidth(val: number);
      /**
       * Links new Seismograph with this amplitude scale.
       *
       * @param   graphList Array of AmplitudeScalable to link
       */
      linkAll(graphList: Array<AmplitudeScalable | {
          amp_scalable: AmplitudeScalable;
      }>): void;
      /**
       * Link new Seismograph with this amplitude scale.
       *
       * @param   graph AmplitudeScalable to link
       */
      link(graph: AmplitudeScalable | {
          amp_scalable: AmplitudeScalable;
      }): void;
      /**
       * Unlink Seismograph with this amplitude scale.
       *
       * @param   graph AmplitudeScalable to unlink
       */
      unlink(graph: AmplitudeScalable): void;
      /**
       * Recalculate the best amplitude scale for all Seismographs. Causes a redraw.
       *
       * @returns array of promise of best amp scales
       */
      recalculate(): Promise<Array<AmplitudeScalable>>;
      _internalNotifyAll(): Promise<Array<AmplitudeScalable>>;
      notifyAll(): Promise<Array<AmplitudeScalable>>;
      get graphList(): AmplitudeScalable[];
  }
  export class IndividualAmplitudeScale extends LinkedAmplitudeScale {
      constructor(graphList?: Array<AmplitudeScalable>);
      recalculate(): Promise<Array<AmplitudeScalable>>;
      notifyAll(): Promise<Array<AmplitudeScalable>>;
  }
  export class FixedHalfWidthAmplitudeScale extends LinkedAmplitudeScale {
      constructor(halfWidth: number, graphList?: Array<AmplitudeScalable>);
      recalculate(): Promise<Array<AmplitudeScalable>>;
      notifyAll(): Promise<Array<AmplitudeScalable>>;
  }
  /**
   * Links time scales across multiple seismographs.
   *
   * @param graphList optional list of TimeScalables to link
   */
  export class LinkedTimeScale {
      /**
       * @private
       */
      _graphSet: Set<TimeScalable>;
      _originalDuration: Duration;
      _originalOffset: Duration;
      _zoomedDuration: null | Duration;
      _zoomedOffset: null | Duration;
      _scaleId: number;
      constructor(graphList?: Array<TimeScalable>, originalDuration?: Duration, originalOffset?: Duration, scaleId?: number);
      /**
       * Link new TimeScalable with this time scale.
       *
       * @param   graph TimeScalable to link
       */
      link(graph: TimeScalable | {
          time_scalable: TimeScalable;
      }): void;
      /**
       * Links TimeScalable with this time scale. Each
       * object in the array should either be a TimeScalable
       * or have a time_scalable field that is a TimeScalable.
       *
       * @param   graphList Array of TimeScalable to link
       */
      linkAll(graphList: Array<TimeScalable | {
          time_scalable: TimeScalable;
      }>): void;
      /**
       * Unlink TimeScalable with this amplitude scale.
       *
       * @param   graph TimeScalable to unlink
       */
      unlink(graph: TimeScalable): void;
      zoom(startOffset: Duration, duration: Duration): void;
      unzoom(): void;
      get offset(): Duration;
      set offset(offset: Duration);
      get duration(): Duration;
      set duration(duration: Duration);
      get origOffset(): Duration;
      get origDuration(): Duration;
      /**
       * Recalculate the best time scale for all Seismographs. Causes a redraw.
       * @returns promise to array of all linked items
       */
      recalculate(): Promise<Array<TimeScalable>>;
      notifyAll(): Promise<Array<TimeScalable>>;
      get graphList(): TimeScalable[];
  }
  /**
   * Linked Time Scale that only modifies the alignment via the offset. The
   * duration of the linked TimeScalable is reused.
   * @param graphList            [description]
   * @param originalDuration     [description]
   * @param originalOffset  [description]
   * @param scaleId              [description]
   */
  export class AlignmentLinkedTimeScale extends LinkedTimeScale {
      constructor(graphList?: Array<TimeScalable>, originalDuration?: Duration, originalOffset?: Duration, scaleId?: number);
      /**
       * Does no calculation, just causes a redraw.
       * @returns promise to all linked items
       */
      recalculate(): Promise<Array<TimeScalable>>;
      notifyAll(): Promise<Array<TimeScalable>>;
  }

}
declare module 'seisplotjs/src/seedcodec' {
  /** ascii */
  export const ASCII = 0;
  /** 16 bit integer, or java short */
  export const SHORT = 1;
  /** 24 bit integer */
  export const INT24 = 2;
  /** 32 bit integer, or java int */
  export const INTEGER = 3;
  /** ieee float */
  export const FLOAT = 4;
  /** ieee double*/
  export const DOUBLE = 5;
  /** Steim1 compression */
  export const STEIM1 = 10;
  /** Steim2 compression */
  export const STEIM2 = 11;
  /** CDSN 16 bit gain ranged */
  export const CDSN = 16;
  /** (A)SRO */
  export const SRO = 30;
  /** DWWSSN 16 bit */
  export const DWWSSN = 32;
  export class CodecException extends Error {
      constructor(message: string);
  }
  export class UnsupportedCompressionType extends Error {
      constructor(message: string);
  }
  export function isFloatCompression(compressionType: number): boolean;
  /**
   * A holder for compressed data independent of the file format.
   */
  export class EncodedDataSegment {
      compressionType: number;
      dataView: DataView;
      numSamples: number;
      littleEndian: boolean;
      constructor(compressionType: number, dataView: DataView, numSamples: number, littleEndian: boolean);
      isFloatCompression(): boolean;
      decode(): Int32Array | Float32Array | Float64Array;
  }
  /**
   *  Decompress the samples from the provided DataView and
   *  return an array of the decompressed values.
   *  Only 16 bit short, 32 bit int, 32 bit float and 64 bit double
   *  along with Steim1 and Steim2 are supported.
   *
   *  @param compressionType compression format as defined in SEED blockette 1000
   *  @param dataView input DataView to be decoded
   *  @param numSamples the number of samples that can be decoded from array
   *  <b>b</b>
   *  @param littleEndian if true, dataView is little-endian (intel byte order) <b>b</b>.
   *  @returns array of length <b>numSamples</b>.
   *  @throws CodecException fail to decompress.
   *  @throws UnsupportedCompressionType unsupported compression type
   */
  export function decompress(compressionType: number, dataView: DataView, numSamples: number, littleEndian: boolean): Int32Array | Float32Array | Float64Array;
  /**
   *  Decode the indicated number of samples from the provided byte array and
   *  return an integer array of the decompressed values.  Being differencing
   *  compression, there may be an offset carried over from a previous data
   *  record.  This offset value can be placed in <b>bias</b>, otherwise leave
   *  the value as 0.
   *
   *  @param dataView input DataView to be decoded
   *  @param numSamples the number of samples that can be decoded from array
   *  <b>b</b>
   *  @param littleEndian if true, dataView is little-endian (intel byte order) <b>b</b>.
   *  @param bias the first difference value will be computed from this value.
   *  If set to 0, the method will attempt to use the X(0) constant instead.
   *  @returns int array of length <b>numSamples</b>.
   *  @throws CodecException - encoded data length is not multiple of 64
   *  bytes.
   */
  export function decodeSteim1(dataView: DataView, numSamples: number, littleEndian: boolean, bias: number): Int32Array;
  /**
   *  Decode the indicated number of samples from the provided byte array and
   *  return an integer array of the decompressed values.  Being differencing
   *  compression, there may be an offset carried over from a previous data
   *  record.  This offset value can be placed in <b>bias</b>, otherwise leave
   *  the value as 0.
   *
   *  @param dataView input byte array to be decoded
   *  @param numSamples the number of samples that can be decoded from array
   *  @param swapBytes if true, swap reverse the endian-ness of the elements of
   *  dataview
   *  @param bias the first difference value will be computed from this value.
   *  If set to 0, the method will attempt to use the X(0) constant instead.
   *  @returns int array of length <b>numSamples</b>.
   *  @throws SteimException - encoded data length is not multiple of 64
   *  bytes.
   */
  export function decodeSteim2(dataView: DataView, numSamples: number, swapBytes: boolean, bias: number): Int32Array;

}
declare module 'seisplotjs/src/seedlink' {
  import { DataRecord } from "seisplotjs/src/miniseed";
  import { DateTime } from "luxon";
  export const SEEDLINK_PROTOCOL = "SeedLink3.1";
  export type SequencedDataRecord = {
      rawsequence: string;
      sequence: number;
      miniseed: DataRecord;
  };
  /**
   * A seedlink websocket connection to the given url.
   * The connection is not made until the connect() method is called.
   * Note this cannot connect directly to a native TCP socket, instead it
   * sends the seedlink protocol over a websocket. Currently only the IRIS
   * ringserver, https://github.com/iris-edu/ringserver,
   * supports websockets, but it may be possible to use thrid party
   * tools to proxy the websocket to a TCP seedlink socket.
   *
   * The seedlink (ver 3) protocol does not have an official spec document, but
   * some details are here:
   * https://www.seiscomp.de/doc/apps/seedlink.html
   *
   * @param url websocket URL to connect to
   * @param requestConfig an array of seedlink commands
   * like:<pre><code>
   *   [ 'STATION JSC CO',
   *     'SELECT 00BHZ.D' ]
   *     </pre></code>
   * @param receiveMiniseedFn the callback function that
   * will be invoked for each seedlink packet received
   * which contains 'sequence', a sequence number
   * and 'miniseed', a single miniseed record.
   */
  export class SeedlinkConnection {
      url: string;
      requestConfig: Array<string>;
      receiveMiniseedFn: (packet: SequencedDataRecord) => void;
      errorHandler: (error: Error) => void;
      closeFn: null | ((close: CloseEvent) => void);
      webSocket: null | WebSocket;
      command: string;
      constructor(url: string, requestConfig: Array<string>, receiveMiniseedFn: (packet: SequencedDataRecord) => void, errorHandler: (error: Error) => void);
      setTimeCommand(startTime: DateTime): void;
      setOnError(errorHandler: (error: Error) => void): void;
      setOnClose(closeFn: (close: CloseEvent) => void): void;
      connect(): void;
      close(): void;
      handle(event: MessageEvent): void;
      handleMiniseed(data: ArrayBuffer): void;
      isConnected(): boolean;
      /**
       * Sends initial HELLO to server and waits for response.
       *
       * @returns            Promise that resolves to the response from the server.
       */
      sendHello(): Promise<[string, string]>;
      /**
       * Sends an array of commands, each as a Promise waiting for the 'OK' response
       * before sending the next.
       *
       * @param   cmd array of commands to send
       * @returns      Promise that resolves to the 'OK' returned by the last
       *   command if successful, or rejects on the first failure.
       */
      sendCmdArray(cmd: Array<string>): Promise<string>;
      /**
       * creates a Promise that sends a command and waits resolved with the result.
       *
       * @param   mycmd command string to send.
       * @returns        Promise that resolves to the reply from the server.
       */
      createCmdPromise(mycmd: string): Promise<string>;
      /**
       * handle errors that arise
       *
       * @private
       * @param   error the error
       */
      handleError(error: Error): void;
  }

}
declare module 'seisplotjs/src/seedlink4' {
  import * as miniseed from "seisplotjs/src/miniseed";
  import * as mseed3 from "seisplotjs/src/mseed3";
  import { DataRecord } from "seisplotjs/src/miniseed";
  import { MSeed3Record } from "seisplotjs/src/mseed3";
  import { DateTime } from "luxon";
  export const SEEDLINK4_PROTOCOL = "SLPROTO4.0";
  export const MINISEED_2_FORMAT = "2";
  export const MINISEED_3_FORMAT = "3";
  export const SE_PACKET_SIGNATURE = "SE";
  export const END_COMMAND = "END";
  export const ENDFETCH_COMMAND = "ENDFETCH";
  export const AUTH_COMMAND = "AUTH";
  export const BYE_COMMAND = "BYE";
  export const DATA_COMMAND = "DATA";
  export const HELLO_COMMAND = "HELLO";
  export const INFO_COMMAND = "INFO";
  export const SELECT_COMMAND = "SELECT";
  export const SLPROTO_COMMAND = "SLPROTO";
  export const STATION_COMMAND = "STATION";
  export const USERAGENT_COMMAND = "USERAGENT";
  export class SEPacket {
      dataFormat: string;
      dataSubformat: string;
      payloadLength: number;
      sequence: bigint;
      stationId: string;
      _miniseed: DataRecord | null;
      _mseed3: MSeed3Record | null;
      _json: Record<string, unknown> | null;
      _rawPayload: DataView | null;
      constructor(dataFormat: string, dataSubformat: string, payloadLength: number, sequence: bigint, stationId: string);
      static parse(data: ArrayBuffer): SEPacket;
      /**
       * is this packet a miniseed packet
       *
       * @returns          true if it is miniseed
       */
      isMiniseed(): boolean;
      /**
       * Parsed payload as a miniseed data record, if the streamid
       * ends with '/MSEED', null otherwise.
       *
       * @returns miniseed DataRecord or null
       */
      asMiniseed(): miniseed.DataRecord | null;
      /**
       * is this packet a miniseed3 packet
       *
       * @returns          true if it is miniseed3
       */
      isMiniseed3(): boolean;
      /**
       * Parsed payload as a miniseed3 data record, if the data format is 3, null otherwise.
       *
       * @returns miniseed3 DataRecord or null
       */
      asMiniseed3(): mseed3.MSeed3Record | null;
  }
  /**
   * A seedlink websocket connection to the given url.
   * The connection is not made until the connect() method is called.
   * Note this cannot connect directly to a native TCP socket, instead it
   * sends the seedlink protocol over a websocket. Currently only the IRIS
   * ringserver, https://github.com/iris-edu/ringserver,
   * supports websockets, but it may be possible to use thrid party
   * tools to proxy the websocket to a TCP seedlink socket.
   *
   * The spec is available via the FDSN, https://www.fdsn.org/publications/
   *
   * Note as of 2023, this is largely untested as there are now servers
   * available to test against.
   *
   * @param url websocket URL to connect to
   * @param requestConfig an array of seedlink commands
   * like:<pre><code>
   *   [ 'STATION CO_JSC',
   *     'SELECT 00_B_H_Z' ]
   *     </pre></code>
   * @param receivePacketFn the callback function that
   * will be invoked for each seedlink packet received.
   * @param errorHandler callback function for errors
   * @param closeFn callback function for closing connection
   * @param webSocket optional web socket connection
   * @param endCommand handshake ending command, either END or ENDFETCH
   * @param agent agent identifier
   * @param agentVersion agent version
   */
  export class SeedlinkConnection {
      url: string;
      requestConfig: Array<string>;
      receivePacketFn: (packet: SEPacket) => void;
      errorHandler: (error: Error) => void;
      closeFn: null | ((close: CloseEvent) => void);
      webSocket: null | WebSocket;
      endCommand: string;
      agent: string;
      agentVersion: string;
      constructor(url: string, requestConfig: Array<string>, receivePacketFn: (packet: SEPacket) => void, errorHandler: (error: Error) => void);
      setAgent(agent: string): void;
      createDataTimeCommand(startTime: DateTime, endTime: DateTime | undefined): string;
      setOnError(errorHandler: (error: Error) => void): void;
      setOnClose(closeFn: (close: CloseEvent) => void): void;
      connect(): void;
      interactiveConnect(): Promise<SeedlinkConnection>;
      checkProto(lines: Array<string>): boolean;
      close(): void;
      handle(event: MessageEvent): void;
      handleSEPacket(event: MessageEvent): void;
      isConnected(): boolean;
      /**
       * Sends initial HELLO to server and waits for response.
       *
       * @returns            Promise that resolves to the response from the server.
       */
      sendHello(): Promise<Array<string>>;
      /**
       * Sends an array of commands, each as a Promise waiting for the 'OK' response
       * before sending the next.
       *
       * @param   cmd array of commands to send
       * @returns      Promise that resolves to the 'OK' returned by the last
       *   command if successful, or rejects on the first failure.
       */
      sendCmdArray(cmd: Array<string>): Promise<string>;
      /**
       * creates a Promise that sends a command and waits resolved with the result.
       *
       * @param   mycmd command string to send.
       * @returns        Promise that resolves to the reply from the server.
       */
      createCmdPromise(mycmd: string): Promise<string>;
      /**
       * handle errors that arise
       *
       * @private
       * @param   error the error
       */
      handleError(error: Error): void;
  }

}
declare module 'seisplotjs/src/seismogram' {
  import { DateTime, Duration, Interval } from "luxon";
  import { FDSNSourceId, NslcId } from "seisplotjs/src/fdsnsourceid";
  import * as seedcodec from "seisplotjs/src/seedcodec";
  import { DistAzOutput } from "seisplotjs/src/distaz";
  import { Network, Station, Channel, InstrumentSensitivity } from "seisplotjs/src/stationxml";
  import { Quake } from "seisplotjs/src/quakeml";
  import { AMPLITUDE_MODE, MinMaxable } from "seisplotjs/src/scale";
  import { SeismogramSegment } from "seisplotjs/src/seismogramsegment";
  import type { TraveltimeJsonType, TraveltimeArrivalType } from "seisplotjs/src/traveltime";
  export const COUNT_UNIT = "count";
  export type HighLowType = {
      xScaleDomain: Array<Date>;
      xScaleRange: Array<number>;
      secondsPerPixel: number;
      samplesPerPixel: number;
      highlowArray: Array<number>;
  };
  import type { MarkerType } from "seisplotjs/src/seismographmarker";
  /**
   * Represents time window for a single channel that may
   * contain gaps or overlaps, but is otherwise more or less
   * continuous, or at least adjacent data from the channel.
   * Each segment within
   * the Seismogram will have the same units, channel identifiers
   * and sample rate, but cover different times.
   */
  export class Seismogram {
      _segmentArray: Array<SeismogramSegment>;
      _interval: Interval;
      _y: null | Int32Array | Float32Array | Float64Array;
      constructor(segmentArray: SeismogramSegment | Array<SeismogramSegment>);
      checkAllSimilar(): void;
      checkSimilar(f: SeismogramSegment, s: SeismogramSegment): void;
      findStartEnd(): Interval;
      findMinMax(minMaxAccumulator?: MinMaxable): MinMaxable;
      /**
       * calculates the mean of a seismogrma.
       *
       * @returns       mean value
       */
      mean(): number;
      get start(): DateTime;
      get startTime(): DateTime;
      get end(): DateTime;
      get endTime(): DateTime;
      get timeRange(): Interval;
      get networkCode(): string | null;
      get stationCode(): string | null;
      get locationCode(): string | null;
      get channelCode(): string | null;
      /**
       * return FDSN source id as a string.
       *
       * @returns FDSN source id
       */
      get sourceId(): FDSNSourceId;
      set sourceId(sid: FDSNSourceId);
      get sampleRate(): number;
      get samplePeriod(): number;
      get yUnit(): string | null;
      isYUnitCount(): boolean;
      get numPoints(): number;
      hasCodes(): boolean;
      /**
       * return network, station, location and channels codes as one string.
       * Uses this.channel if it exists, this.seismogram if not.
       *
       * @returns net.sta.loc.chan
       */
      get nslc(): string;
      get nslcId(): NslcId;
      codes(): string;
      get segments(): Array<SeismogramSegment>;
      append(seismogram: SeismogramSegment | Seismogram): void;
      /**
       * Cut the seismogram. Creates a new seismogram with all datapoints
       * contained in the time window.
       *
       * @param  timeRange start and end of cut
       * @returns            new seismogram
       */
      cut(timeRange: Interval): null | Seismogram;
      /**
       * Creates a new Seismogram composed of all seismogram segments that overlap the
       * given time window. If none do, this returns null. This is a faster but coarser
       * version of cut as it only removes whole segments that do not overlap the
       * time window. For most seismograms that consist of a single contiguous
       * data segment, this will do nothing.
       *
       * @param timeRange time range to trim to
       * @returns new seismogram if data in the window, null otherwise
       * @see cut
       */
      trim(timeRange: Interval): null | Seismogram;
      break(duration: Duration): void;
      isContiguous(): boolean;
      /**
       * Merges all segments into a single array of the same type as the first
       * segment. No checking is done for gaps or overlaps, this is a simple
       * congatination. Be careful!
       *
       * @returns contatenated data
       */
      merge(): Int32Array | Float32Array | Float64Array;
      /**
       * Gets the timeseries as an typed array if it is contiguous.
       *
       * @throws {NonContiguousData} if data is not contiguous.
       * @returns  timeseries as array of number
       */
      get y(): Int32Array | Float32Array | Float64Array;
      set y(val: Int32Array | Float32Array | Float64Array);
      clone(): Seismogram;
      cloneWithNewData(newY: Int32Array | Float32Array | Float64Array): Seismogram;
      /**
       * factory method to create a single segment Seismogram from either encoded data
       *  or a TypedArray, along with sample rate and start time.
       *
       * @param yArray array of encoded data or typed array
       * @param sampleRate sample rate, samples per second of the data
       * @param startTime time of first sample
       * @param sourceId optional source id
       * @returns seismogram initialized with the data
       */
      static fromContiguousData(yArray: Array<seedcodec.EncodedDataSegment> | Int32Array | Float32Array | Float64Array, sampleRate: number, startTime: DateTime, sourceId?: FDSNSourceId): Seismogram;
  }
  export class NonContiguousData extends Error {
      constructor(message?: string);
  }
  export function ensureIsSeismogram(seisSeismogram: Seismogram | SeismogramSegment): Seismogram;
  export class SeismogramDisplayData {
      /** @private */
      _seismogram: Seismogram | null;
      _id: string | null;
      _sourceId: FDSNSourceId | null;
      label: string | null;
      markerList: Array<MarkerType>;
      traveltimeList: Array<TraveltimeArrivalType>;
      channel: Channel | null;
      _instrumentSensitivity: InstrumentSensitivity | null;
      quakeList: Array<Quake>;
      quakeReferenceList: Array<string>;
      timeRange: Interval;
      alignmentTime: DateTime;
      doShow: boolean;
      _statsCache: SeismogramDisplayStats | null;
      constructor(timeRange: Interval);
      static fromSeismogram(seismogram: Seismogram): SeismogramDisplayData;
      /**
       * Create a Seismogram from the segment, then call fromSeismogram to create
       * the SeismogramDisplayData;
       * @param  seisSegment segment of contiguous data
       * @returns             new SeismogramDisplayData
       */
      static fromSeismogramSegment(seisSegment: SeismogramSegment): SeismogramDisplayData;
      /**
       * Useful for creating fake data from an array, sample rate and start time
       *
       * @param yArray fake data
       * @param sampleRate samples per second
       * @param startTime  start of data, time of first point
       * @param sourceId  optional source id
       * @returns seismogramdisplaydata
       */
      static fromContiguousData(yArray: Array<seedcodec.EncodedDataSegment> | Int32Array | Float32Array | Float64Array, sampleRate: number, startTime: DateTime, sourceId?: FDSNSourceId): SeismogramDisplayData;
      static fromChannelAndTimeWindow(channel: Channel, timeRange: Interval): SeismogramDisplayData;
      static fromChannelAndTimes(channel: Channel, startTime: DateTime, endTime: DateTime): SeismogramDisplayData;
      static fromSourceIdAndTimes(sourceId: FDSNSourceId, startTime: DateTime, endTime: DateTime): SeismogramDisplayData;
      static fromCodesAndTimes(networkCode: string, stationCode: string, locationCode: string, channelCode: string, startTime: DateTime, endTime: DateTime): SeismogramDisplayData;
      addQuake(quake: Quake | Array<Quake>): void;
      /**
       * Adds a public id for a Quake to the seismogram. For use in case where
       * the quake is not yet available, but wish to retain the connection.
       * @param  publicId  id of the earthquake assocated with this seismogram
       */
      addQuakeId(publicId: string): void;
      addMarker(marker: MarkerType): void;
      addMarkers(markers: MarkerType | Array<MarkerType>): void;
      getMarkers(): Array<MarkerType>;
      addTravelTimes(ttimes: TraveltimeJsonType | TraveltimeArrivalType | Array<TraveltimeArrivalType>): void;
      hasQuake(): boolean;
      get quake(): Quake | null;
      hasSeismogram(): this is {
          _seismogram: Seismogram;
      };
      append(seismogram: SeismogramSegment | Seismogram): void;
      hasChannel(): this is {
          channel: Channel;
      };
      hasSensitivity(): this is {
          _instrumentSensitivity: InstrumentSensitivity;
      };
      /**
       * Allows id-ing a seismogram. Optional.
       *
       * @returns         string id
       */
      get id(): string | null;
      /**
       * Allows iding a seismogram. Optional.
       *
       * @param   value string id
       */
      set id(value: string | null);
      /**
       * return network code as a string.
       * Uses this.channel if it exists, this.seismogram if not.
       *
       * @returns network code
       */
      get networkCode(): string;
      /**
       * return station code as a string.
       * Uses this.channel if it exists, this.seismogram if not.
       *
       * @returns station code
       */
      get stationCode(): string;
      /**
       * return location code a a string.
       * Uses this.channel if it exists, this.seismogram if not.
       *
       * @returns location code
       */
      get locationCode(): string;
      /**
       * return channels code as a string.
       * Uses this.channel if it exists, this.seismogram if not.
       *
       * @returns channel code
       */
      get channelCode(): string;
      /**
       * return FDSN source id as a string.
       * Uses this.channel if it exists, this.seismogram if not.
       *
       * @returns FDSN source id
       */
      get sourceId(): FDSNSourceId;
      /**
       * return network, station, location and channels codes as one string.
       * Uses this.channel if it exists, this.seismogram if not
       *
       * @returns net.sta.loc.chan
       */
      get nslc(): string;
      get nslcId(): NslcId;
      /**
       * return network, station, location and channels codes as one string.
       * Uses this.channel if it exists, this.seismogram if not.
       *
       * @param sep separator, defaults to '.'
       * @returns nslc codes separated by sep
       */
      codes(sep?: string): string;
      get startTime(): DateTime;
      get start(): DateTime;
      get endTime(): DateTime;
      get end(): DateTime;
      get numPoints(): number;
      associateChannel(nets: Array<Network>): void;
      alignStartTime(): void;
      alignOriginTime(): void;
      alignPhaseTime(phaseRegExp: RegExp | string): void;
      /**
       * Create a time window relative to the alignmentTime if set, or the start time if not.
       * Negative durations are allowed.
       * @param alignmentOffset offset duration from the alignment time
       * @param duration duration from the offset for the window
       * @returns time window as an Interval
       */
      relativeTimeWindow(alignmentOffset: Duration, duration: Duration): Interval;
      get sensitivity(): InstrumentSensitivity | null;
      set sensitivity(value: InstrumentSensitivity | null);
      get min(): number;
      get max(): number;
      get mean(): number;
      get middle(): number;
      get seismogram(): Seismogram | null;
      set seismogram(value: Seismogram | null);
      get segments(): Array<SeismogramSegment>;
      calcStats(): SeismogramDisplayStats;
      /**
       * Calculates distance and azimuth for each event in quakeList.
       *
       * @returns Array of DistAzOutput, empty array if no quakes.
       */
      get distazList(): Array<DistAzOutput>;
      /**
       * Calculates distance and azimuth for the first event in quakeList. This is
       * a convienence method as usually there will only be one quake.
       *
       * @returns DistAzOutput, null if no quakes.
       */
      get distaz(): null | DistAzOutput;
      clone(): SeismogramDisplayData;
      cloneWithNewSeismogram(seis: Seismogram | null): SeismogramDisplayData;
      /**
       * Cut the seismogram. Creates a new seismogramDisplayData with the cut
       * seismogram and the timeRange set to the new time window.
       *
       * @param  timeRange start and end of cut
       * @returns           new seismogramDisplayData
       */
      cut(timeRange: Interval): null | SeismogramDisplayData;
      /**
       * Coarse trim the seismogram. Creates a new seismogramDisplayData with the
       * trimmed seismogram and the timeRange set to the new time window.
       * If timeRange is not given, the current time range of the
       * SeismogramDisplayData is used, effectively trimming data to the current
       * window.
       *
       * @param  timeRange start and end of cut
       * @returns           new seismogramDisplayData
       */
      trim(timeRange?: Interval): null | SeismogramDisplayData;
      /**
       * Coarse trim the seismogram in place. The seismogram is
       * trimmed to the given time window.
       * If timeRange is not given, the current time range of the
       * SeismogramDisplayData is used, effectively trimming data to the current
       * window.
       *
       * @param  timeRange start and end of cut
       */
      trimInPlace(timeRange?: Interval): void;
      toString(): string;
  }
  export class SeismogramDisplayStats {
      min: number;
      max: number;
      mean: number;
      trendSlope: number;
      constructor();
      get middle(): number;
  }
  export function findStartEnd(sddList: Array<SeismogramDisplayData>): Interval;
  export function findMaxDuration(sddList: Array<SeismogramDisplayData>): Duration;
  /**
   * Finds max duration of from one of starttime of sdd, origin time
   * of earthquake, or alignmentTime.
   *
   * @param  type  one of start, origin or align
   * @param  sddList list of seis data
   * @returns        max duration
   */
  export function findMaxDurationOfType(type: string, sddList: Array<SeismogramDisplayData>): Duration;
  /**
   * Finds the min and max amplitude over the seismogram list, considering gain
   * and how to center the seismograms, either Raw, MinMax or Mean.
   *
   * @param  sddList  list of seismogramdisplaydata
   * @param  doGain  should gain be used
   * @param  amplitudeMode centering style
   * @returns    min max
   */
  export function findMinMax(sddList: Array<SeismogramDisplayData>, doGain?: boolean, amplitudeMode?: AMPLITUDE_MODE): MinMaxable;
  export function findMinMaxOverTimeRange(sddList: Array<SeismogramDisplayData>, timeRange?: Interval | null, doGain?: boolean, amplitudeMode?: AMPLITUDE_MODE): MinMaxable;
  export function findMinMaxOverRelativeTimeRange(sddList: Array<SeismogramDisplayData>, alignmentOffset: Duration, duration: Duration, doGain?: boolean, amplitudeMode?: AMPLITUDE_MODE): MinMaxable;
  export function calcMinMax(sdd: SeismogramDisplayData, timeRange?: Interval | null, doGain?: boolean, amplitudeMode?: AMPLITUDE_MODE): MinMaxable | null;
  export function findStartEndOfSeismograms(data: Array<Seismogram>, accumulator?: Interval): Interval;
  export function findMinMaxOfSeismograms(data: Array<Seismogram>, minMaxAccumulator?: MinMaxable): MinMaxable;
  export function findMinMaxOfSDD(data: Array<SeismogramDisplayData>, minMaxAccumulator?: MinMaxable): MinMaxable;
  export function uniqueStations(seisData: Array<SeismogramDisplayData>): Array<Station>;
  export function uniqueChannels(seisData: Array<SeismogramDisplayData>): Array<Channel>;
  export function uniqueQuakes(seisData: Array<SeismogramDisplayData>): Array<Quake>;

}
declare module 'seisplotjs/src/seismogramloader' {
  import { Dataset } from "seisplotjs/src/dataset";
  import { Duration } from "luxon";
  import { DataSelectQuery } from "seisplotjs/src/fdsndataselect";
  import { EventQuery } from "seisplotjs/src/fdsnevent";
  import { StationQuery } from "seisplotjs/src/fdsnstation";
  import { Quake } from "seisplotjs/src/quakeml";
  import { Network } from "seisplotjs/src/stationxml";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  /**
   * Loads seismograms based on queries to Station and Event web services.
   * Uses the traveltime web service to create times for data query. Default
   * is P-30 sec to S+60 sec.
   *
   * @param stationQuery     query to find channels
   * @param eventQuery       query to find events
   * @param dataselectQuery  optional additional parameters for seismogram query
   */
  export class SeismogramLoader {
      stationQuery: StationQuery | Promise<Array<Network>>;
      withFedCatalog: boolean;
      withResponse: boolean;
      markOrigin: boolean;
      eventQuery: EventQuery | Promise<Array<Quake>>;
      dataselectQuery: null | DataSelectQuery;
      _startPhaseList: Array<string>;
      _endPhaseList: Array<string>;
      _markedPhaseList: Array<string>;
      _startOffset: Duration;
      _endOffset: Duration;
      constructor(stationQuery: StationQuery | Array<Network>, eventQuery: EventQuery | Array<Quake>, dataselectQuery?: DataSelectQuery);
      get startPhaseList(): Array<string>;
      set startPhaseList(val: Array<string> | string);
      get startOffset(): Duration;
      set startOffset(val: Duration);
      /**
       * Sets the startOffset Duration to be val seconds.
       *
       * @param  val  number of seconds, negative for before, positive for after
       * @returns     this
       */
      startOffsetSeconds(val: number): SeismogramLoader;
      get endPhaseList(): Array<string>;
      set endPhaseList(val: Array<string> | string);
      get endOffset(): Duration;
      set endOffset(val: Duration);
      /**
       * Sets the endOffset Duration to be val seconds.
       *
       * @param  val  number of seconds, negative for before, positive for after
       * @returns     this
       */
      endOffsetSeconds(val: number): SeismogramLoader;
      /**
       * Additional phase arrival travel times to be marked, but do not effect
       * the request time window.
       *
       * @returns array of phase names.
       */
      get markedPhaseList(): Array<string>;
      set markedPhaseList(val: Array<string> | string);
      loadSeismograms(): Promise<Array<SeismogramDisplayData>>;
      /**
       * Loads a Dataset based on the input station and event queries.
       *
       * The raw traveltimes are included in the extras of the dataset with
       * key "traveltimes", which is a Map with the quake as the key.
       *
       * @returns a Dataset
       */
      load(): Promise<Dataset>;
  }

}
declare module 'seisplotjs/src/seismogramsegment' {
  import { DateTime, Interval } from "luxon";
  import { MinMaxable } from "seisplotjs/src/scale";
  import * as seedcodec from "seisplotjs/src/seedcodec";
  import { FDSNSourceId, NslcId } from "seisplotjs/src/fdsnsourceid";
  export const COUNT_UNIT = "count";
  export type HighLowType = {
      xScaleDomain: Array<Date>;
      xScaleRange: Array<number>;
      secondsPerPixel: number;
      samplesPerPixel: number;
      highlowArray: Array<number>;
  };
  /**
   * A contiguous segment of a Seismogram.
   *
   * @param  yArray array of Y sample values, ie the timeseries
   * @param  sampleRate sample rate of the seismogram, hertz
   * @param  startTime start time of seismogrm as a luxon DateTime in utc or a string that can be parsed
   */
  export class SeismogramSegment {
      /** Array of y values */
      _y: null | Int32Array | Float32Array | Float64Array;
      _compressed: null | Array<seedcodec.EncodedDataSegment>;
      /**
       * the sample rate in hertz
       *
       * @private
       */
      _sampleRate: number;
      /** @private */
      _startTime: DateTime;
      _endTime_cache: null | DateTime;
      _endTime_cache_numPoints: number;
      _sourceId: FDSNSourceId;
      yUnit: string;
      _highlow: HighLowType | undefined;
      constructor(yArray: Array<seedcodec.EncodedDataSegment> | Array<number> | Int32Array | Float32Array | Float64Array, sampleRate: number, startTime: DateTime, sourceId?: FDSNSourceId);
      /**
       * Y data of the seismogram. Decompresses data if needed.
       *
       * @returns y data as typed array
       */
      get y(): Int32Array | Float32Array | Float64Array;
      set y(value: Int32Array | Float32Array | Float64Array);
      get start(): DateTime;
      set start(value: DateTime | string);
      get startTime(): DateTime;
      set startTime(value: DateTime | string);
      get end(): DateTime;
      get endTime(): DateTime;
      get timeRange(): Interval;
      get sampleRate(): number;
      set sampleRate(value: number);
      get samplePeriod(): number;
      get numPoints(): number;
      get networkCode(): string;
      get stationCode(): string | null;
      get locationCode(): string | null;
      get channelCode(): string | null;
      /**
       * Checks if the data is encoded
       *
       * @returns true if encoded, false otherwise
       */
      isEncoded(): boolean;
      /**
       * Gets encoded data, if it is.
       *
       * @returns array of encoded data segments
       * @throws Error if data is not encoded
       */
      getEncoded(): Array<seedcodec.EncodedDataSegment>;
      yAtIndex(i: number): number;
      /**
       * Finds the min and max values of a SeismogramSegment, with an optional
       * accumulator for use with gappy data.
       *
       * @param minMaxAccumulator optional initialized accumulator as an array
       * of two numbers, min and max
       * @returns min, max as arry of length two
       */
      findMinMax(minMaxAccumulator?: MinMaxable): MinMaxable;
      /**
       * Time of the i-th sample, indexed from zero.
       * If i is negative, counting from end, so
       * timeOfSample(-1) is time of last data point;
       *
       * @param  i               sample index
       * @returns   time
       */
      timeOfSample(i: number): DateTime;
      indexOfTime(t: DateTime): number;
      hasCodes(): boolean;
      /**
       * return network, station, location and channels codes as one string.
       * Uses this.channel if it exists, this.seismogram if not.
       *
       * @returns nslc codes separated by '.'
       */
      get nslc(): string;
      get nslcId(): NslcId;
      /**
       * return network, station, location and channels codes as one string
       *
       * @param sep separator, defaults to '.'
       * @returns nslc codes separated by sep
       */
      codes(sep?: string): string;
      seisId(): string;
      /**
       * return FDSN source id.
       *
       * @returns FDSN source id
       */
      get sourceId(): FDSNSourceId;
      set sourceId(sid: FDSNSourceId);
      clone(): SeismogramSegment;
      cloneWithNewData(clonedData: Array<seedcodec.EncodedDataSegment> | Int32Array | Float32Array | Float64Array, clonedStartTime?: DateTime): SeismogramSegment;
      cut(timeRange: Interval): SeismogramSegment | null;
      _invalidate_endTime_cache(): void;
  }

}
declare module 'seisplotjs/src/seismograph' {
  import { DateTime, Duration, Interval } from "luxon";
  import "d3-transition";
  import { AmplitudeScalable, TimeScalable, MinMaxable } from "seisplotjs/src/scale";
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  import { createMarkersForTravelTimes, createMarkerForOriginTime, createFullMarkersForQuakeAtStation, createFullMarkersForQuakeAtChannel, createMarkerForQuakePicks, createMarkerForPicks } from "seisplotjs/src/seismographmarker";
  export { createMarkersForTravelTimes, createMarkerForOriginTime, createFullMarkersForQuakeAtStation, createFullMarkersForQuakeAtChannel, createMarkerForQuakePicks, createMarkerForPicks, };
  import type { MarkerType } from "seisplotjs/src/seismographmarker";
  import type { HandlebarsInput } from "seisplotjs/src/axisutil";
  import type { Axis } from "d3-axis";
  import type { ScaleLinear, NumberValue as d3NumberValue } from "d3-scale";
  import type { Selection } from "d3-selection";
  import { SeismogramDisplayData, Seismogram } from "seisplotjs/src/seismogram";
  import { SeisPlotElement } from "seisplotjs/src/spelement";
  import * as axisutil from "seisplotjs/src/axisutil";
  export type BBoxType = {
      height: number;
      width: number;
  };
  export type MarkerHolderType = {
      marker: MarkerType;
      sdd: SeismogramDisplayData;
      xscale: axisutil.LuxonTimeScale;
      bbox?: BBoxType;
  };
  export const SEISMOGRAPH_ELEMENT = "sp-seismograph";
  export const seismograph_css = "\n\n:host {\n  display: block;\n  min-height: 50px;\n  height: 100%;\n}\n\ndiv.wrapper {\n  min-height: 50px;\n  height: 100%;\n}\n\n.marker .markerpath {\n  fill: none;\n  stroke: black;\n  stroke-width: 1px;\n}\n\n.marker polygon {\n  fill: rgba(150,220,150,.4);\n}\n\n.marker.predicted polygon {\n  fill: rgba(220,220,220,.4);\n}\n\n.marker.pick polygon {\n  fill: rgba(255,100,100,.4);\n}\n\npath.seispath {\n  stroke: skyblue;\n  fill: none;\n  stroke-width: 1px;\n}\n\npath.orientZ {\n  stroke: seagreen;\n}\n\npath.orientN {\n  stroke: cornflowerblue;\n}\n\npath.orientE {\n  stroke: orange;\n}\n\npath.alignment {\n  stroke-dasharray: 8;\n  stroke-width: 2px;\n}\n\nsvg.seismograph {\n  height: 100%;\n  width: 100%;\n  min-height: 25px;\n  min-width: 25px;\n}\n\nsvg.seismograph g.ySublabel text {\n  font-size: smaller;\n}\n\nsvg.seismograph g.xSublabel text {\n  font-size: smaller;\n}\n\nsvg.seismograph text.title {\n  font-size: larger;\n  font-weight: bold;\n  fill: black;\n  color: black;\n}\n\nsvg.realtimePlot g.allseismograms path.seispath {\n  stroke: skyblue;\n}\n\n/* links in svg */\nsvg.seismograph text a {\n  fill: #0000EE;\n  text-decoration: underline;\n}\n\n";
  export const COLOR_CSS_ID = "seismographcolors";
  export class Seismograph extends SeisPlotElement {
      /** @private */
      static _lastID: number;
      plotId: number;
      beforeFirstDraw: boolean;
      /** @private */
      _debugAlignmentSeisData: Array<SeismogramDisplayData>;
      width: number;
      height: number;
      outerWidth: number;
      outerHeight: number;
      svg: Selection<SVGSVGElement, unknown, null, undefined>;
      canvasHolder: null | Selection<SVGForeignObjectElement, unknown, null, undefined>;
      canvas: null | Selection<HTMLCanvasElement, unknown, null, undefined>;
      g: Selection<SVGGElement, unknown, null, undefined>;
      throttleRescale: ReturnType<typeof setTimeout> | null;
      throttleRedraw: ReturnType<typeof requestAnimationFrame> | null;
      time_scalable: SeismographTimeScalable;
      amp_scalable: SeismographAmplitudeScalable;
      _resizeObserver: ResizeObserver;
      minmax_sample_pixels: number;
      constructor(seisData?: SeismogramDisplayData | Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      get seisData(): Array<SeismogramDisplayData>;
      set seisData(seisData: Array<SeismogramDisplayData>);
      get seismographConfig(): SeismographConfig;
      set seismographConfig(seismographConfig: SeismographConfig);
      connectedCallback(): void;
      disconnectedCallback(): void;
      attributeChangedCallback(_name: string, _oldValue: string, _newValue: string): void;
      checkResize(): boolean;
      enableZoom(): void;
      draw(): void;
      printSizes(): void;
      calcDetailForEvent(evt: MouseEvent, _type?: string): SeisMouseEventType;
      isVisible(): boolean;
      drawSeismograms(): void;
      calcScaleAndZoom(): void;
      ampScaleForSeisDisplayData(sdd: SeismogramDisplayData): ScaleLinear<number, number, never>;
      displayTimeRangeForSeisDisplayData(sdd: SeismogramDisplayData): Interval;
      timeScaleForSeisDisplayData(sdd?: SeismogramDisplayData | Interval): axisutil.LuxonTimeScale;
      /**
       * Draws the top, bottom, (time) axis and the left and right (amplitude) axis if configured.
       */
      drawAxis(): void;
      /**
       * Creates amp scale, set range based on height.
       * @private
       * @returns amp scale with range set
       */
      __initAmpScale(): ScaleLinear<number, number, never>;
      ampScaleForAxis(): ScaleLinear<number, number, never>;
      timeScaleForAxis(): ScaleLinear<number, number, never> | axisutil.LuxonTimeScale;
      /**
       * Draws the left and right (amplitude) axis if configured.
       *
       */
      drawTopBottomAxis(): void;
      /**
       * Draws the left and right (amplitude) axis if configured.
       */
      drawLeftRightAxis(): void;
      createLeftRightAxis(): Array<Axis<d3NumberValue> | null>;
      rescaleYAxis(): void;
      createHandlebarsInput(): HandlebarsInput;
      drawAxisLabels(): void;
      resetZoom(): void;
      zoomed(e: any): void;
      redrawWithXScale(): void;
      drawMarkers(): void;
      calcWidthHeight(nOuterWidth: number, nOuterHeight: number): void;
      drawTitle(): void;
      drawXLabel(): void;
      drawXSublabel(): void;
      drawYLabel(): void;
      drawYSublabel(): void;
      /**
       * Update the duration if not already set. This only matters for
       * linedTimeScale currently.
       */
      calcTimeScaleDomain(): void;
      /**
       * Calculate the amplitude range over the current time range, depending
       * on amplitude style.
       *
       * @returns min max over the time range
       */
      calcAmpScaleDomain(): MinMaxable;
      recheckAmpScaleDomain(): void;
      redoDisplayYScale(): void;
      createUnitsLabel(): string;
      getSeismogramData(): Array<SeismogramDisplayData>;
      /**
       * can append single seismogram segment or an array of segments.
       *
       * @param sddList array or single SeismogramDisplayData or Seismogram
       * @private
       */
      _internalAppend(sddList: Array<SeismogramDisplayData> | SeismogramDisplayData | Array<Seismogram> | Seismogram): void;
      /**
       * appends the seismogram(s) or SeismogramDisplayData as separate time series.
       *
       * @param seismogram data to append
       */
      appendSeisData(seismogram: Array<Seismogram> | Array<SeismogramDisplayData> | Seismogram | SeismogramDisplayData): void;
      /**
       * Notification to the element that something about the current seismogram
       * data has changed. This could be that the actual waveform data has been updated
       * or that auxillary data like quake or channel has been added. This should
       * trigger a redraw.
       */
      seisDataUpdated(): void;
      /**
       * Finds the SeismogramDisplayData within the display containing the given
       * Seismogram.
       *
       * @param   seis seismogram to search for
       * @returns       SeismogramDisplayData if found or null if not
       */
      getDisplayDataForSeismogram(seis: Seismogram): SeismogramDisplayData | null;
      /**
       * Removes a seismogram from the display.
       *
       * @param   seisData seis data to remove
       */
      removeSeisData(seisData: SeismogramDisplayData): void;
      /**
       * Removes seismograms that do not overlap the window.
       *
       * @param   timeRange overlap data to keep
       */
      trim(timeRange: Interval): void;
  }
  export class SeismographAmplitudeScalable extends AmplitudeScalable {
      graph: Seismograph;
      drawHalfWidth: number;
      drawMiddle: number;
      constructor(graph: Seismograph);
      notifyAmplitudeChange(middle: number, halfWidth: number): void;
  }
  export const ZERO_DURATION: Duration<true>;
  export class SeismographTimeScalable extends TimeScalable {
      graph: Seismograph;
      drawAlignmentTimeOffset: Duration;
      drawDuration: Duration;
      constructor(graph: Seismograph, alignmentTimeOffset: Duration, duration: Duration);
      notifyTimeRangeChange(offset: Duration, duration: Duration): void;
  }
  /**
   * Creates a wrapper for d3 formatter for numbers for axis that keeps typescript happy.
   *
   * @param  formatter simple formatter
   * @returns           function that converts input types
   */
  export function createNumberFormatWrapper(formatter: (value: number) => string): (nValue: d3NumberValue) => string;
  /**
   * Creates a wrapper for d3 formatter for Dates for axis that keeps typescript happy.
   *
   * @param  formatter simple formatter
   * @returns           function that converts input types
   */
  export function createDateFormatWrapper(formatter: (value: Date) => string): (nValue: Date | d3NumberValue, index: number) => string;
  export type SeisMouseEventType = {
      mouseevent: MouseEvent;
      time: DateTime | null;
      relative_time: Duration | null;
      amplitude: number;
      seismograph: Seismograph;
  };

}
declare module 'seisplotjs/src/seismographconfig' {
  import { LinkedAmplitudeScale, LinkedTimeScale, AMPLITUDE_MODE } from "seisplotjs/src/scale";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { Interval } from "luxon";
  import type { AxisDomain } from "d3-axis";
  export type MarginType = {
      top: number;
      right: number;
      bottom: number;
      left: number;
      toString?: () => string;
  };
  export const DEFAULT_TITLE = "{{#each seisDataList}}<tspan>{{onlyChangesChannel ../seisDataList @index}}</tspan> {{else}}No Data{{/each}}";
  export class SeismographConfigCache {
      titleHandlebarsCompiled: null | ((arg0: object, arg1: object) => string);
      xLabelHandlebarsCompiled: null | ((arg0: object, arg1: object) => string);
      xSublabelHandlebarsCompiled: null | ((arg0: object, arg1: object) => string);
      yLabelHandlebarsCompiled: null | ((arg0: object, arg1: object) => string);
      yLabelRightHandlebarsCompiled: null | ((arg0: object, arg1: object) => string);
      ySublabelHandlebarsCompiled: null | ((arg0: object, arg1: object) => string);
      constructor();
  }
  /**
   * Configuration object for Seismograph display.
   *
   */
  export class SeismographConfig {
      /** @private */
      static _lastID: number;
      configId: number;
      /** @private */
      __cache__: SeismographConfigCache;
      timeFormat: (date: Date) => string;
      relativeTimeFormat: (value: number) => string;
      amplitudeFormat: (value: number) => string;
      showTitle: boolean;
      /** @private */
      _title: Array<string>;
      /** @private */
      isXAxis: boolean;
      isXAxisTop: boolean;
      /** @private */
      _xLabel: string;
      xLabelOrientation: string;
      /** @private */
      _xSublabel: string;
      xSublabelIsUnits: boolean;
      /**
       * Should grid lines be drawn for each tick on the x axis.
       */
      xGridLines: boolean;
      isYAxis: boolean;
      isYAxisRight: boolean;
      isYAxisNice: boolean;
      /** @private */
      _yLabel: string;
      /** @private */
      _yLabelRight: string;
      yLabelOrientation: string;
      /** @private */
      _ySublabel: string;
      ySublabelTrans: number;
      ySublabelIsUnits: boolean;
      yGridLines: boolean;
      doMarkers: boolean;
      markerTextOffset: number;
      markerTextAngle: number;
      markerFlagpoleBase: string;
      minHeight: number;
      maxHeight: null | number;
      minWidth: number;
      maxWidth: null | number;
      margin: MarginType;
      segmentDrawCompressedCutoff: number;
      maxZoomPixelPerSample: number;
      connectSegments: boolean;
      lineColors: Array<string>;
      lineWidth: number;
      gridLineColor: string;
      wheelZoom: boolean;
      amplitudeMode: AMPLITUDE_MODE;
      doGain: boolean;
      windowAmp: boolean;
      /** @private */
      _fixedAmplitudeScale: null | Array<number>;
      /** @private */
      _fixedTimeScale: null | Interval;
      /** @private */
      _linkedAmplitudeScale: null | LinkedAmplitudeScale;
      /** @private */
      _linkedTimeScale: null | LinkedTimeScale;
      isRelativeTime: boolean;
      constructor();
      static fromJSON(json: SeismographConfigJsonType): SeismographConfig;
      asJSON(): SeismographConfigJsonType;
      get fixedAmplitudeScale(): null | Array<number>;
      set fixedAmplitudeScale(ts: null | Array<number>);
      get linkedAmplitudeScale(): null | LinkedAmplitudeScale;
      set linkedAmplitudeScale(ts: null | LinkedAmplitudeScale);
      /**
       * Enable linked amplitude scales across seismographs.
       *
       */
      enableLinkedAmplitude(): void;
      /**
       * Set Raw amplitude mode, plot absolute and
       * goes from minimun to maximum of data
       */
      amplitudeRaw(): void;
      /**
       * Set MinMax amplitude mode, plot is relative and
       * centered on (minimun + maximum)/2
       */
      amplitudeMinMax(): void;
      /**
       * Set Mean amplitude mode, plot is relative and
       * centered on mean of data
       */
      amplitudeMean(): void;
      /**
       * Set WithZero amplitude mode, plot is absolute and
       * centered on mean of data like Raw, but also includes zero
       * even if all data is positive. Useful when showing
       * data compared to zero is helpful.
       */
      amplitudeWithZero(): void;
      /**
       * True if the amplitude is "centered".
       *
       * Both MinMax and Mean center the amplitude, Raw and Zero do not.
       *
       * @returns if centered
       */
      isCenteredAmp(): boolean;
      get fixedTimeScale(): null | Interval;
      set fixedTimeScale(ts: null | Interval);
      get linkedTimeScale(): null | LinkedTimeScale;
      set linkedTimeScale(ts: null | LinkedTimeScale);
      /**
       * gets the current title
       *
       * @returns        title as an array of strings
       */
      get title(): Array<string>;
      /**
       * Sets the title as simple string or array of strings or a
       * handlebars template. If an array
       * then each item will be in a separate tspan for easier formatting.
       *
       * @param value string or array of strings to be the title
       */
      set title(value: null | string | Array<string>);
      handlebarsTitle(context: object, runtimeOptions: object): string;
      /**
       * gets the current x label
       *
       * @returns        x label
       */
      get xLabel(): string;
      /**
       * Sets the xLabel as simple string or a
       * handlebars template.
       *
       * @param value string  to be the x label
       */
      set xLabel(value: null | string);
      /**
       * gets the current x sublabel
       *
       * @returns        x sublabel
       */
      get xSublabel(): string;
      /**
       * Sets the xSublabel as simple string or a
       * handlebars template.
       *
       * @param value string  to be the x sublabel
       */
      set xSublabel(value: null | string);
      handlebarsXLabel(context: object, runtimeOptions: object): string;
      handlebarsXSublabel(context: object, runtimeOptions: object): string;
      /**
       * gets the current y label
       *
       * @returns        yLabel
       */
      get yLabel(): string;
      /**
       * Sets the ylabel as simple string or a
       * handlebars template.
       *
       * @param value string to be the y label
       */
      set yLabel(value: null | string);
      /**
       * gets the current y sublabel
       *
       * @returns        ySublabel
       */
      get ySublabel(): string;
      /**
       * Sets the y sublabel as simple string or a
       * handlebars template.
       *
       * @param value string to be the y sublabel
       */
      set ySublabel(value: null | string);
      handlebarsYLabel(context: object, runtimeOptions: object): string;
      handlebarsYSublabel(context: object, runtimeOptions: object): string;
      /**
       * gets the current title
       *
       * @returns        yLabel
       */
      get yLabelRight(): string;
      /**
       * Sets the ylabel as simple string or a
       * handlebars template.
       *
       * @param value string to be the y label
       */
      set yLabelRight(value: null | string);
      handlebarsYLabelRight(context: object, runtimeOptions: object): string;
      /**
       * Fake data to use to test alignment of seismograph axis and between canvas
       *  and svg drawing.
       *
       * @param   timeRange start and end of fake data
       * @param   min        min amplitude for fake data, default is -100
       * @param   max        max amplitude for fake data, default is 100
       * @returns             fake data
       */
      createAlignmentData(timeRange: Interval, min?: number, max?: number): SeismogramDisplayData;
      getColorForIndex(i: number): string;
      createCSSForLineColors(svgClass?: string): string;
      clone(): SeismographConfig;
      toString(): string;
  }
  export type SeismographConfigJsonType = {
      configId: number;
      showTitle: boolean;
      title: Array<string>;
      isXAxis: boolean;
      isXAxisTop: boolean;
      xLabel: string;
      xLabelOrientation: string;
      xSublabel: string;
      xSublabelIsUnits: boolean;
      isYAxis: boolean;
      isYAxisRight: boolean;
      isYAxisNice: boolean;
      yLabel: string;
      yLabelRight: string;
      yLabelOrientation: string;
      ySublabel: string;
      ySublabelTrans: number;
      ySublabelIsUnits: boolean;
      doMarkers: boolean;
      markerTextOffset: number;
      markerTextAngle: number;
      markerFlagpoleBase: string;
      minHeight: number;
      maxHeight: null | number;
      minWidth: number;
      maxWidth: null | number;
      margin: MarginType;
      segmentDrawCompressedCutoff: number;
      maxZoomPixelPerSample: number;
      connectSegments: boolean;
      lineColors: Array<string>;
      lineWidth: number;
      wheelZoom: boolean;
      amplitudeMode: AMPLITUDE_MODE;
      doGain: boolean;
      windowAmp: boolean;
      fixedAmplitudeScale: null | Array<number>;
      fixedTimeScale: null | Interval;
      isLinkedAmplitudeScale: boolean;
      isLinkedTimeScale: boolean;
      isRelativeTime: boolean;
  };
  export function numberFormatWrapper(formater: (arg0: number) => string): (domainValue: AxisDomain) => string;
  export const formatCount: (arg0: number) => string;
  export const formatExp: (arg0: number) => string;
  export const formatCountOrAmp: (v: number) => string;
  export const formatMillisecond: (arg0: Date) => string;
  export const formatSecond: (arg0: Date) => string;
  export const formatMinute: (arg0: Date) => string;
  export const formatHour: (arg0: Date) => string;
  export const formatDay: (arg0: Date) => string;
  export const formatMonth: (arg0: Date) => string;
  export const formatYear: (arg0: Date) => string;
  export const multiFormatHour: (date: Date) => string;

}
declare module 'seisplotjs/src/seismographconfigeditor' {
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  import type { Selection } from "d3-selection";
  export function createEditor(div: Selection<HTMLElement, unknown, null, undefined>, config: SeismographConfig, onChange: () => void): void;
  export type SEL_DIV_SPAN = Selection<HTMLDivElement, unknown, null, undefined> | Selection<HTMLSpanElement, unknown, null, undefined>;
  export const configEditor_css = "\ninput[type=\"text\"].smallconfigtext {\n  width: 7em;\n}\n\ninput[type=\"number\"].smallconfigtext {\n  width: 7em;\n}\n\ninput[type=\"text\"].bigconfigtext {\n  width: 27em;\n}\n";

}
declare module 'seisplotjs/src/seismographmarker' {
  import type { TraveltimeJsonType } from "seisplotjs/src/traveltime";
  import { Quake, Origin } from "seisplotjs/src/quakeml";
  import { Station, Channel } from "seisplotjs/src/stationxml";
  import { DateTime } from "luxon";
  export type MarkerType = {
      name: string;
      time: DateTime;
      markertype: string;
      description: string;
      link?: string;
  };
  export function isValidMarker(v: unknown): v is MarkerType;
  /**
   * Creates Markers for all of the arrivals in ttime.arrivals, relative
   * to the given Quake.
   *
   * @param   quake quake the travel times are relative to
   * @param   ttime travel times json object as returned from the
   * IRIS traveltime web service, or the json output of TauP
   * @returns        array of Markers suitable for adding to a seismograph
   */
  export function createMarkersForTravelTimes(quake: Quake, ttime: TraveltimeJsonType): Array<MarkerType>;
  /**
   * Creates a Marker for the origin time in ttime.arrivals, for the given Quake.
   *
   * @param   quake quake the travel times are relative to
   * @returns        Marker suitable for adding to a seismograph
   */
  export function createMarkerForOriginTime(quake: Quake): MarkerType;
  export function createFullMarkersForQuakeAtStation(quake: Quake, station: Station): Array<MarkerType>;
  export function createFullMarkersForQuakeAtChannel(quake: Quake, channel: Channel): Array<MarkerType>;
  /**
   * Creates a Marker for the picked arrival times in quake.pickList, for the given Quake.
   *
   * @param quake quake the travel times are relative to
   * @param channel channel picks made on
   * @returns        Marker suitable for adding to a seismograph
   */
  export function createMarkerForQuakePicks(quake: Quake, channel: Channel): Array<MarkerType>;
  /**
   * Creates a Marker for the picked arrival times in quake.arrivals, for the given Quake.
   *
   * @param origin quake the travel times are relative to
   * @param channel channel picks made on
   * @returns        Marker suitable for adding to a seismograph
   */
  export function createMarkerForPicks(origin: Origin, channel: Channel): Array<MarkerType>;

}
declare module 'seisplotjs/src/seismographutil' {
  import { LuxonTimeScale } from "seisplotjs/src/axisutil";
  import { SeismogramSegment } from "seisplotjs/src/seismogramsegment";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import type { ScaleLinear } from "d3-scale";
  export const DEFAULT_MAX_SAMPLE_PER_PIXEL = 3;
  export const DEFAULT_GRID_LINE_COLOR = "gainsboro";
  export function clearCanvas(canvas: HTMLCanvasElement): void;
  export function drawXScaleGridLines(canvas: HTMLCanvasElement, xScale: ScaleLinear<number, number, never> | LuxonTimeScale, colorname?: string, lineWidth?: number): void;
  export function drawYScaleGridLines(canvas: HTMLCanvasElement, yScale: ScaleLinear<number, number, never>, colorname?: string, lineWidth?: number): void;
  export function drawAllOnCanvas(canvas: HTMLCanvasElement, sddList: Array<SeismogramDisplayData>, xScaleList: Array<LuxonTimeScale>, yScaleList: Array<ScaleLinear<number, number, never>>, colornameList: Array<string>, lineWidth?: number, connectSegments?: boolean, maxSamplePerPixelForLineDraw?: number): void;
  export function drawSeismogramAsLine(sdd: SeismogramDisplayData, context: CanvasRenderingContext2D, width: number, xScale: LuxonTimeScale, yScale: ScaleLinear<number, number, never>, color: string, lineWidth?: number, connectSegments?: boolean, maxSamplePerPixelForLineDraw?: number): void;
  export type XYLine = {
      x: Array<number>;
      y: Array<number>;
      samplesPerPixel: number;
      maxSamplePerPixelForLineDraw: number;
  };
  export function seismogramSegmentAsLine(segment: SeismogramSegment | null, width: number, xScale: LuxonTimeScale, yScale: ScaleLinear<number, number, never>, maxSamplePerPixelForLineDraw?: number): XYLine;
  export function pushPoint(out: XYLine, xpixel: number, ypixel: number): void;
  export function rgbaForColorName(name: string): Uint8ClampedArray;

}
declare module 'seisplotjs/src/sorting' {
  import type { DateTime } from "luxon";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  export const SORT_NONE = "none";
  export const SORT_DISTANCE = "distance";
  export const SORT_AZIMUTH = "azimuth";
  export const SORT_BACKAZIMUTH = "backazimuth";
  export const SORT_ALPHABETICAL = "alphabetical";
  export const SORT_STARTTIME = "starttime";
  export const SORT_ORIGINTIME = "origin";
  export function sort(seisData: Array<SeismogramDisplayData>, key: string): SeismogramDisplayData[];
  export function createSortValueFunction(key: string): (sdd: SeismogramDisplayData) => number | string | DateTime;
  /**
   * Reorders array of SeismogramDisplayData in "best effort" xyz order. The data
   * should all be from the same station and represent three components of
   * motion. A common grouping of Z,N,E seismograms in any order would be returned as
   * E,N,Z. If the grouping was Z,1,2 where the azimuth of 1 is 90 degrees ccw
   * from 2, then the returned order would be 2,1,Z.
   *
   * @param  sddList  array representing compoment of motion
   * @returns     sorted array in x,y,z order
   */
  export function reorderXYZ(sddList: Array<SeismogramDisplayData>): Array<SeismogramDisplayData>;

}
declare module 'seisplotjs/src/spectraplot' {
  import { FFTResult } from "seisplotjs/src/fft";
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { Complex } from "seisplotjs/src/oregondsputil";
  export const SPECTRA_ELEMENT = "sp-spectra";
  /**
   * Similar to FFTResult, but used for plotting non-fft generated data.
   * This allows the frequencies to be, for example, evenly distrubuted
   * in log instead of linearly for plotting PolesZeros stages.
   */
  export class FreqAmp {
      freq: Float32Array;
      values: Array<InstanceType<typeof Complex>>;
      /** optional units of the original data for display purposes. */
      inputUnits: string;
      seismogramDisplayData: null | SeismogramDisplayData;
      constructor(freq: Float32Array, values: Array<InstanceType<typeof Complex>>);
      frequencies(): Float32Array;
      amplitudes(): Float32Array;
      phases(): Float32Array;
      get numFrequencies(): number;
      get minFrequency(): number;
      get maxFrequency(): number;
      get fundamentalFrequency(): number;
  }
  /**
   * Defualt CSS for styling fft plots.
   */
  export const spectra_plot_css = "\n:host {\n  display: block\n}\n\ndiv.wrapper {\n  height: 100%;\n  min-height: 100px;\n}\npath.fftpath {\n  stroke: skyblue;\n  fill: none;\n  stroke-width: 1px;\n}\n\nsvg.spectra_plot {\n  height: 100%;\n  width: 100%;\n  min-height: 100px;\n  display: block;\n}\nsvg.spectra_plot text.title {\n  font-size: larger;\n  font-weight: bold;\n  fill: black;\n  color: black;\n}\n\nsvg.spectra_plot text.sublabel {\n  font-size: smaller;\n}\n\n/* links in svg */\nsvg.spectra_plot text a {\n  fill: #0000EE;\n  text-decoration: underline;\n}\n\n";
  export const AMPLITUDE = "amplitude";
  export const PHASE = "phase";
  export const LOGFREQ = "logfreq";
  export const KIND = "kind";
  /**
   * A amplitude or phase plot of fft data. The 'kind' attribute controls whether
   * 'amplitude' or 'phase' is plotted and the 'logfreq' attribute controls
   * whether frequency, x axis, is linear or log.
   * Setting the seismogramConfig changes
   * the plot configuration, althought not all values are used.
   * The data as an array of FFTResult or FreqAmp
   * sets the data to be plotted.
   * Amplitude is plotted with y axis log, phase is y axis linear.
   *
   */
  export class SpectraPlot extends HTMLElement {
      _seismographConfig: SeismographConfig;
      _fftResults: Array<FFTResult | FreqAmp>;
      constructor(fftResults?: Array<FFTResult | FreqAmp>, seismographConfig?: SeismographConfig);
      get fftResults(): Array<FFTResult | FreqAmp>;
      set fftResults(fftResults: Array<FFTResult | FreqAmp>);
      get seismographConfig(): SeismographConfig;
      set seismographConfig(seismographConfig: SeismographConfig);
      get kind(): string;
      set kind(val: string);
      get logfreq(): boolean;
      set logfreq(val: boolean);
      connectedCallback(): void;
      static get observedAttributes(): string[];
      attributeChangedCallback(_name: string, _oldValue: string, _newValue: string): void;
      draw(): void;
  }

}
declare module 'seisplotjs/src/spelement' {
  import { SeismogramDisplayData } from "seisplotjs/src/seismogram";
  import { SeismographConfig } from "seisplotjs/src/seismographconfig";
  export class SeisPlotElement extends HTMLElement {
      _seisDataList: Array<SeismogramDisplayData>;
      _seismographConfig: SeismographConfig;
      onRedraw: (el: SeisPlotElement) => void;
      _throttleRedraw: ReturnType<typeof requestAnimationFrame> | null;
      constructor(seisData?: SeismogramDisplayData | Array<SeismogramDisplayData>, seisConfig?: SeismographConfig);
      get seisData(): Array<SeismogramDisplayData>;
      set seisData(seisData: Array<SeismogramDisplayData>);
      get seismographConfig(): SeismographConfig;
      set seismographConfig(seismographConfig: SeismographConfig);
      addStyle(css: string, id?: string): HTMLStyleElement;
      /**
       * Notification to the element that something about the current seismogram
       * data has changed. This could be that the actual waveform data has been updated
       * or that auxillary data like quake or channel has been added or that the
       * configuration has changed. This should trigger a redraw.
       */
      seisDataUpdated(): void;
      connectedCallback(): void;
      /**
       * Redraw the element. This implements a throttle so that many redraws
       * are coelsced into a single actual draw if they occur before the
       * next animation frame.
       */
      redraw(): void;
      /**
       * Draw the element, overridden by subclasses. Generally outside callers
       * should prefer calling redraw() as it handles throttling and calls the
       * onRedraw callback.
       */
      draw(): void;
      getShadowRoot(autoAdd?: boolean): ShadowRoot;
  }
  export function addStyleToElement(element: HTMLElement, css: string, id?: string): HTMLStyleElement;

}
declare module 'seisplotjs/src/stationxml' {
  import { Complex } from "seisplotjs/src/oregondsputil";
  import { FDSNSourceId, NetworkSourceId, StationSourceId, NslcId } from "seisplotjs/src/fdsnsourceid";
  import { DateTime, Interval } from "luxon";
  /** xml namespace for stationxml */
  export const STAML_NS = "http://www.fdsn.org/xml/station/1";
  export const COUNT_UNIT_NAME = "count";
  export const FIX_INVALID_STAXML = true;
  export const INVALID_NUMBER = -99999;
  export const FAKE_START_DATE: DateTime<true> | DateTime<false>;
  /** a fake, completely empty stationxml document in case of no data. */
  export const FAKE_EMPTY_XML = "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?> <FDSNStationXML xmlns=\"http://www.fdsn.org/xml/station/1\" schemaVersion=\"1.0\" xsi:schemaLocation=\"http://www.fdsn.org/xml/station/1 http://www.fdsn.org/xml/station/fdsn-station-1.0.xsd\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:iris=\"http://www.fdsn.org/xml/station/1/iris\"> </FDSNStationXML>";
  export const CHANNEL_CLICK_EVENT = "channelclick";
  export const STATION_CLICK_EVENT = "stationclick";
  /**
   * Utility function to create CustomEvent for clicking on a Channel, for example
   * in a map or table.
   *
   * @param  sta          Channel clicked on
   * @param  mouseclick original mouse click Event
   * @returns            CustomEvent populated with channel field in detail.
   */
  export function createChannelClickEvent(sta: Channel, mouseclick: Event): CustomEvent;
  /**
   * Utility function to create CustomEvent for clicking on a Station, for example
   * in a map or table.
   *
   * @param  sta          Station clicked on
   * @param  mouseclick original mouse click Event
   * @returns            CustomEvent populated with station field in detail.
   */
  export function createStationClickEvent(sta: Station, mouseclick: Event): CustomEvent;
  export class Network {
      networkCode: string;
      _startDate: DateTime;
      _endDate: DateTime | null;
      restrictedStatus: string;
      description: string;
      totalNumberStations: number | null;
      stations: Array<Station>;
      constructor(networkCode: string);
      get sourceId(): NetworkSourceId;
      get startDate(): DateTime;
      set startDate(value: DateTime | string);
      get endDate(): null | DateTime;
      set endDate(value: DateTime | string | null);
      get timeRange(): Interval;
      codes(): string;
      isActiveAt(d?: DateTime): boolean;
      isTempNet(): boolean;
  }
  export class Station {
      network: Network;
      stationCode: string;
      sourceID: string | null;
      /** @private */
      _startDate: DateTime;
      /** @private */
      _endDate: DateTime | null;
      restrictedStatus: string;
      name: string;
      latitude: number;
      longitude: number;
      elevation: number;
      waterLevel: number | null;
      comments: Array<Comment>;
      equipmentList: Array<Equipment>;
      dataAvailability: DataAvailability | null;
      identifierList: Array<string>;
      description: string;
      geology: string;
      vault: string;
      channels: Array<Channel>;
      constructor(network: Network, stationCode: string);
      get sourceId(): StationSourceId;
      get startDate(): DateTime;
      set startDate(value: DateTime | string);
      get endDate(): DateTime | null;
      set endDate(value: DateTime | string | null);
      get timeRange(): Interval;
      get networkCode(): string;
      isActiveAt(d?: DateTime): boolean;
      codes(sep?: string): string;
  }
  export class Channel {
      station: Station;
      /** @private */
      _locationCode: string;
      channelCode: string;
      /** @private */
      _sourceId: FDSNSourceId | undefined;
      /** @private */
      _startDate: DateTime;
      /** @private */
      _endDate: DateTime | null;
      restrictedStatus: string;
      latitude: number;
      longitude: number;
      elevation: number;
      depth: number;
      azimuth: number;
      dip: number;
      sampleRate: number;
      waterLevel: number | null;
      comments: Array<Comment>;
      equipmentList: Array<Equipment>;
      dataAvailability: DataAvailability | null;
      identifierList: Array<string>;
      description: string;
      response: Response | null;
      sensor: Equipment | null;
      preamplifier: Equipment | null;
      datalogger: Equipment | null;
      constructor(station: Station, channelCode: string, locationCode: string);
      get sourceId(): FDSNSourceId;
      get nslcId(): NslcId;
      get startDate(): DateTime;
      set startDate(value: DateTime | string);
      get endDate(): null | DateTime;
      set endDate(value: DateTime | string | null);
      get timeRange(): Interval;
      get locationCode(): string;
      set locationCode(value: string);
      get stationCode(): string;
      get networkCode(): string;
      /**
       * Checks if this channel has sensitivity defined, within the response.
       *
       * @returns          true if instrumentSensitivity exits
       */
      hasInstrumentSensitivity(): boolean;
      set instrumentSensitivity(value: InstrumentSensitivity);
      get instrumentSensitivity(): InstrumentSensitivity;
      /**
       * return network, station, location and channels codes as one string.
       *
       * @returns net.sta.loc.chan
       */
      get nslc(): string;
      /**
       * return network, station, location and channels codes as one string.
       *
       * @param sep separator, defaults to dot '.'
       * @returns net.sta.loc.chan
       */
      codes(sep?: string): string;
      isActiveAt(d?: DateTime): boolean;
  }
  export class InstrumentSensitivity {
      sensitivity: number;
      frequency: number;
      inputUnits: string;
      outputUnits: string;
      constructor(sensitivity: number, frequency: number, inputUnits: string, outputUnits: string);
  }
  export class Equipment {
      resourceId: string;
      type: string;
      description: string;
      manufacturer: string;
      vendor: string;
      model: string;
      serialNumber: string;
      installationDate: DateTime | null;
      removalDate: DateTime | null;
      calibrationDateList: Array<DateTime>;
      constructor();
  }
  export class Response {
      instrumentSensitivity: InstrumentSensitivity | null;
      stages: Array<Stage>;
      constructor(instrumentSensitivity?: InstrumentSensitivity, stages?: Array<Stage>);
  }
  export class Stage {
      filter: AbstractFilterType | null;
      decimation: Decimation | null;
      gain: Gain;
      constructor(filter: AbstractFilterType | null, decimation: Decimation | null, gain: Gain);
  }
  export class AbstractFilterType {
      inputUnits: string;
      outputUnits: string;
      name: string;
      description: string;
      constructor(inputUnits: string, outputUnits: string);
  }
  export class PolesZeros extends AbstractFilterType {
      pzTransferFunctionType: string;
      normalizationFactor: number;
      normalizationFrequency: number;
      zeros: Array<InstanceType<typeof Complex>>;
      poles: Array<InstanceType<typeof Complex>>;
      constructor(inputUnits: string, outputUnits: string);
  }
  export class FIR extends AbstractFilterType {
      symmetry: string;
      numerator: Array<number>;
      constructor(inputUnits: string, outputUnits: string);
  }
  export class CoefficientsFilter extends AbstractFilterType {
      cfTransferFunction: string;
      numerator: Array<number>;
      denominator: Array<number>;
      constructor(inputUnits: string, outputUnits: string);
  }
  export class Decimation {
      inputSampleRate: number;
      factor: number;
      offset: number | null | undefined;
      delay: number | null | undefined;
      correction: number | null | undefined;
      constructor(inputSampleRate: number, factor: number);
  }
  export class Gain {
      value: number;
      frequency: number;
      constructor(value: number, frequency: number);
  }
  export class Span {
      interval: Interval;
      numberSegments: number;
      maximumTimeTear: number | null;
      constructor(interval: Interval);
  }
  export class DataAvailability {
      extent: Interval | null;
      spanList: Array<Span>;
      constructor();
  }
  export class Comment {
      id: string | null;
      subject: string | null;
      value: string;
      beginEffectiveTime: DateTime | null;
      endEffectiveTime: DateTime | null;
      authorList: Array<Author>;
      constructor(value: string);
  }
  export class Author {
      name: string | null;
      agency: string | null;
      email: string | null;
      phone: string | null;
  }
  /**
   * Parses the FDSN StationXML returned from a query.
   *
   * @param rawXml parsed xml to extract objects from
   * @returns an Array of Network objects.
   */
  export function parseStationXml(rawXml: Document): Array<Network>;
  /**
   * Parses a FDSNStationXML Network xml element into a Network object.
   *
   * @param xml the network xml Element
   * @returns Network instance
   */
  export function convertToNetwork(xml: Element): Network;
  /**
   * Parses a FDSNStationXML Station xml element into a Station object.
   *
   * @param network the containing network
   * @param xml the station xml Element
   * @returns Station instance
   */
  export function convertToStation(network: Network, xml: Element): Station;
  /**
   * Parses a FDSNStationXML Channel xml element into a Channel object.
   *
   * @param station the containing staton
   * @param xml the channel xml Element
   * @returns Channel instance
   */
  export function convertToChannel(station: Station, xml: Element): Channel;
  export function convertToDataAvailability(xml: Element): DataAvailability;
  export function convertToComment(xml: Element): Comment;
  export function convertToAuthor(xml: Element): Author;
  export function convertToEquipment(xml: Element): Equipment;
  /**
   * Parses a FDSNStationXML Response xml element into a Response object.
   *
   * @param responseXml the response xml Element
   * @returns Response instance
   */
  export function convertToResponse(responseXml: Element): Response;
  /**
   * Parses a FDSNStationXML InstrumentSensitivity xml element into a InstrumentSensitivity object.
   *
   * @param xml the InstrumentSensitivity xml Element
   * @returns InstrumentSensitivity instance
   */
  export function convertToInstrumentSensitivity(xml: Element): InstrumentSensitivity;
  /**
   * Parses a FDSNStationXML Stage xml element into a Stage object.
   *
   * @param stageXml the Stage xml Element
   * @returns Stage instance
   */
  export function convertToStage(stageXml: Element): Stage;
  /**
   * Parses a FDSNStationXML Decimation xml element into a Decimation object.
   *
   * @param decXml the Decimation xml Element
   * @returns Decimation instance
   */
  export function convertToDecimation(decXml: Element): Decimation;
  /**
   * Parses a FDSNStationXML Gain xml element into a Gain object.
   *
   * @param gainXml the Gain xml Element
   * @returns Gain instance
   */
  export function convertToGain(gainXml: Element): Gain;
  export function createInterval(start: DateTime, end: null | DateTime): Interval;
  /**
   * Extracts a complex number from an stationxml element.
   *
   * @param   el xml element
   * @returns     Complex instance
   */
  export function extractComplex(el: Element): InstanceType<typeof Complex>;
  /**
   * Generator function to access all stations within all networks in the array.
   *
   * @param      networks array of Networks
   * @yields           generator yeiding stations
   */
  export function allStations(networks: Array<Network>): Generator<Station, void, unknown>;
  /**
   * Generator function to access all channels within all stations
   * within all networks in the array.
   *
   * @param      networks array of Networks
   * @yields           generator yeiding channels
   */
  export function allChannels(networks: Array<Network>): Generator<Channel, void, unknown>;
  /**
   * Extract all channels from all stations from all networks in the input array.
   * Regular expressions may be used instaed of exact code matchs.
   *
   * @param   networks Array of networks.
   * @param   netCode network code to match
   * @param   staCode station code to match
   * @param   locCode location code to match
   * @param   chanCode channel code to match
   * @yields           Array of channels.
   */
  export function findChannels(networks: Array<Network>, netCode: string, staCode: string, locCode: string, chanCode: string): Generator<Channel, void, unknown>;
  export function uniqueSourceIds(channelList: Iterable<Channel>): Array<FDSNSourceId>;
  export function uniqueStations(channelList: Iterable<Channel>): Array<Station>;
  export function uniqueNetworks(channelList: Iterable<Channel>): Array<Network>;
  /**
   * Fetches and parses StationXML from a URL. This can be used in instances where
   * a static stationXML file is available on a web site instead of via a web
   * service with query paramters.
   * @param  url          the url to download from
   * @param  timeoutSec   timeout in case of failed connection
   * @param nodata        http code for no data
   * @returns             Promise to parsed StationXML as an Network array
   */
  export function fetchStationXml(url: string | URL, timeoutSec?: number, nodata?: number): Promise<Array<Network>>;
  export const parseUtil: {
      _grabFirstEl: (xml: Element | null | void, tagName: string) => Element | null;
      _grabFirstElText: (xml: Element | null | void, tagName: string) => string | null;
      _grabFirstElFloat: (xml: Element | null | void, tagName: string) => number | null;
      _grabFirstElInt: (xml: Element | null | void, tagName: string) => number | null;
      _grabAttribute: (xml: Element | null | void, tagName: string) => string | null;
      _requireAttribute: (xml: Element | null | void, tagName: string) => string;
      _grabAttributeNS: (xml: Element | null | void, namespace: string, tagName: string) => string | null;
  };

}
declare module 'seisplotjs/src/taper' {
  import { Seismogram } from "seisplotjs/src/seismogram";
  /**
   * Applies a time domain taper of the given type.
   *
   * @param   seis      Seismogram to taper
   * @param   width     width as a fraction of seismogram length, default is 0.05
   * @param   taperType taper type, default is HANNING, others are HAMMING and COSINE
   * @returns            copy of seismogram with taper applied.
   */
  export function taper(seis: Seismogram, width?: number, taperType?: string): Seismogram;
  /**
   * Calculates the coefficients for tapering, [omega, f0, f1]
   *
   * @param type HANNING, HAMMING or COSINE
   * @param length number of data points
   * @returns tapering coefficients as an array of length 3
   */
  export function getCoefficients(type: string, length: number): Array<number>;
  export const HANNING = "HANNING";
  export const HAMMING = "HAMMING";
  export const COSINE = "COSINE";

}
declare module 'seisplotjs/src/textformat' {
  export const latlonFormat: Intl.NumberFormat;
  export const magFormat: Intl.NumberFormat;
  export const depthNoUnitFormat: Intl.NumberFormat;
  export const depthFormat: Intl.NumberFormat;
  export const depthMeterFormat: Intl.NumberFormat;

}
declare module 'seisplotjs/src/transfer' {
  import { FFTResult } from "seisplotjs/src/fft";
  import { SeismogramSegment } from "seisplotjs/src/seismogramsegment";
  import { Seismogram } from "seisplotjs/src/seismogram";
  import { SacPoleZero } from "seisplotjs/src/sacpolezero";
  import { Response, PolesZeros } from "seisplotjs/src/stationxml";
  import { Complex } from "seisplotjs/src/oregondsputil";
  import { AllMeasures, AllMeasuresSystems, AllMeasuresUnits } from 'convert-units/definitions/all';
  /**
   * Applies response, poles and zeros along with overall gain to the seismogram.
   * Should produce results similar to the sac command:
   * transfer from polezero to none
   *
   * @param   seis     seismogram to instrument correct
   * @param   response response to apply
   * @param   lowCut   low cut
   * @param   lowPass  low pass
   * @param   highPass high pass
   * @param   highCut  high cut
   * @returns           instrument corrected seismogram
   */
  export function transfer(seis: Seismogram, response: Response, lowCut: number, lowPass: number, highPass: number, highCut: number): Seismogram;
  export function transferSacPZ(seis: Seismogram, sacPoleZero: SacPoleZero, lowCut: number, lowPass: number, highPass: number, highCut: number): Seismogram;
  export function transferSacPZSegment(seis: SeismogramSegment, sacPoleZero: SacPoleZero, lowCut: number, lowPass: number, highPass: number, highCut: number): SeismogramSegment;
  export function calcResponse(response: Response, numPoints: number, sampleRate: number, unit: string): FFTResult;
  /**
   * Caclulates the frequency response from the given poles and zeros.
   *
   * @param   sacPoleZero poles and zeros
   * @param   numPoints   number of points in the output fft
   * @param   sampleRate  sample rate to compute at
   * @returns             frequency response
   */
  export function calcResponseFromSacPoleZero(sacPoleZero: SacPoleZero, numPoints: number, sampleRate: number): FFTResult;
  /**
   * Applies poles and zeros to the fft of a time series. Modifies the freqValues
   * in place.
   *
   * @param   freqValues  fft of a timeseries
   * @param   sampFreq    sampling frequency
   * @param   sacPoleZero poles and zeros
   * @param   lowCut      low cut
   * @param   lowPass     low pass
   * @param   highPass    high pass
   * @param   highCut     high cut
   * @returns             input freq values, with poles and zeros applied
   */
  export function combine(freqValues: Float32Array, sampFreq: number, sacPoleZero: SacPoleZero, lowCut: number, lowPass: number, highPass: number, highCut: number): Float32Array;
  /**
   * Evaluates the poles and zeros at the given value. The return value is
   * 1/(pz(s) to avoid divide by zero issues. If there is a divide by zero
   * situation, then the response is set to be 0+0i.
   *
   * @param sacPoleZero SAC PoleZero response
   * @param freq frequency to evaluate
   * @returns complex frequency domain value for this frequency
   */
  export function evalPoleZeroInverse(sacPoleZero: SacPoleZero, freq: number): InstanceType<typeof Complex>;
  /**
   * Calculates the frequency taper for the given parameters.
   *
   * @param   freq     frequency
   * @param   lowCut   low cut
   * @param   lowPass  low pass
   * @param   highPass high pass
   * @param   highCut  high cut
   * @returns           taper value at the frequency
   */
  export function calcFreqTaper(freq: number, lowCut: number, lowPass: number, highPass: number, highCut: number): number;
  /**
   * Applies the frequency taper to the fft of the time series.
   *
   * @param   fftResult  fft of time series
   * @param   sampleRate sample rate
   * @param   lowCut     low cut
   * @param   lowPass    low pass
   * @param   highPass   high pass
   * @param   highCut    high cut
   * @returns            fft with taper applied
   */
  export function applyFreqTaper(fftResult: FFTResult, sampleRate: number, lowCut: number, lowPass: number, highPass: number, highCut: number): FFTResult;
  /**
   * commonly used units
   */
  export const METER: import("convert-units").Conversion<AllMeasures, AllMeasuresSystems, AllMeasuresUnits> | null;
  export const METER_PER_SECOND: import("convert-units").Conversion<AllMeasures, AllMeasuresSystems, AllMeasuresUnits> | null;
  export const METER_PER_SECOND_PER_SECOND: import("convert-units").Conversion<AllMeasures, AllMeasuresSystems, AllMeasuresUnits> | null;
  export function calcGamma(unit: string): number;
  export function calcScaleUnit(unit: string): number;
  /**
   * Converts a StationXML response to SAC PoleZero style. This
   * converts the analog to digital stage (usually 0) along
   * with the overall gain, but does not include later FIR stages.
   * To maintain compatibility with SAC, this includes extra zeros
   * if needed to convert to displacement. The number of extra zeros
   * added is kept as gamma in the result.
   *
   * @param response stationxml Response to convert
   * @returns SAC PoleZero style version of the response
   */
  export function convertToSacPoleZero(response: Response): SacPoleZero;
  export function convertPoleZeroToSacStyle(polesZeros: PolesZeros, sensitivity: number, sensitivity_freq: number, gamma: number): SacPoleZero;
  export function calc_A0(poles: Array<InstanceType<typeof Complex>>, zeros: Array<InstanceType<typeof Complex>>, ref_freq: number): number;

}
declare module 'seisplotjs/src/traveltime' {
  import { FDSNCommon } from "seisplotjs/src/fdsncommon";
  import { Station, Channel } from "seisplotjs/src/stationxml";
  import { Quake } from "seisplotjs/src/quakeml";
  export const IRIS_HOST = "service.iris.edu";
  export const TEXT_FORMAT = "text";
  export const JSON_FORMAT = "json";
  export const SVG_FORMAT = "svg";
  /**
   * Type for json returned by iris traveltime web service
   *
   */
  export type TraveltimeJsonType = {
      model: string;
      sourcedepth: number;
      receiverdepth: number;
      phases: Array<string>;
      arrivals: Array<TraveltimeArrivalType>;
  };
  export type TraveltimeArrivalType = {
      distdeg: number;
      phase: string;
      time: number;
      rayparam: number;
      takeoff: number;
      incident: number;
      puristdist: number;
      puristname: string;
  };
  /**
   * Verifies that JSON matches the types we expect, for typescript.
   *
   * @param  v JSON object, usually from the traveltime web service
   * @returns   true if matches expected structure
   */
  export function isValidTraveltimeJsonType(v: unknown): v is TraveltimeJsonType;
  export function isValidTraveltimeArrivalType(v: unknown): v is TraveltimeArrivalType;
  /**
   * converts a text line from the text format into an
   *  TraveltimeArrivalType object like what is returned by the json format.
   *
   *  @param ttimeline travel time output line for an arrival
   *  @returns parsed travel time arrival
   */
  export function convertTravelTimeLineToObject(ttimeline: string): TraveltimeArrivalType;
  /**
   * Creates a fake arrival for the origin time, useful to display a flag
   * at origin time similar to the P and S arrival.
   * @param  distdeg earthquake to station distance, in degrees
   * @returns      an arrival for the origin
   */
  export function createOriginArrival(distdeg: number): TraveltimeArrivalType;
  /**
   * Query to the IRIS traveltime webservice, based on the TauP Toolkit. See
   * https://service.iris.edu/irisws/traveltime/1/ and
   * https://www.seis.sc.edu/TauP/
   *
   * @param host optional host to connect to, defaults to IRIS
   */
  export class TraveltimeQuery extends FDSNCommon {
      /** @private */
      _evdepth: number;
      /** @private */
      _distdeg: Array<number> | undefined;
      /** @private */
      _model: string | undefined;
      /** @private */
      _phases: string | undefined;
      /** @private */
      _stalat: number | undefined;
      /** @private */
      _stalon: number | undefined;
      /** @private */
      _receiverdepth: number | undefined;
      /** @private */
      _evlat: number | undefined;
      /** @private */
      _evlon: number | undefined;
      /** @private */
      _format: string;
      /** @private */
      _noheader: boolean;
      constructor(host?: string | null);
      protocol(value?: string): TraveltimeQuery;
      getProtocol(): string | undefined;
      host(value?: string): TraveltimeQuery;
      getHost(): string;
      /**
       * Gets/Sets the remote port to connect to.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      port(value?: number): TraveltimeQuery;
      getPort(): number | undefined;
      /**
       * Gets/Sets the nodata parameter, usually 404 or 204 (default), controlling
       * the status code when no matching data is found by the service.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      nodata(value?: number): TraveltimeQuery;
      getNodata(): number | undefined;
      specVersion(value?: string): TraveltimeQuery;
      getSpecVersion(): string | undefined;
      evdepth(value?: number): TraveltimeQuery;
      evdepthInMeter(value?: number): TraveltimeQuery;
      getEvdepth(): number | undefined;
      distdeg(value?: number | Array<number>): TraveltimeQuery;
      getDistdeg(): Array<number> | undefined;
      model(value?: string): TraveltimeQuery;
      getModel(): string | undefined;
      phases(value?: string): TraveltimeQuery;
      getPhases(): string | undefined;
      stalat(value?: number): TraveltimeQuery;
      getStalat(): number | undefined;
      stalon(value?: number): TraveltimeQuery;
      getStalon(): number | undefined;
      latLonFromStation(station: Station): TraveltimeQuery;
      receiverdepth(value?: number): TraveltimeQuery;
      receiverdepthInMeter(value?: number): TraveltimeQuery;
      receiverdepthFromChannel(channel: Channel): TraveltimeQuery;
      getReceiverdepth(): number | undefined;
      evlat(value?: number): TraveltimeQuery;
      getEvlat(): number | undefined;
      evlon(value?: number): TraveltimeQuery;
      getEvlon(): number | undefined;
      latLonFromQuake(quake: Quake): TraveltimeQuery;
      format(value?: string): TraveltimeQuery;
      getFormat(): string | undefined;
      noheader(value?: boolean): TraveltimeQuery;
      getNoheader(): boolean | undefined;
      /**
       * Get/Set the timeout in seconds for the request. Default is 30.
       *
       * @param value optional new value if setting
       * @returns new value if getting, this if setting
       */
      timeout(value?: number): TraveltimeQuery;
      getTimeout(): number | undefined;
      queryText(): Promise<string>;
      queryJson(): Promise<TraveltimeJsonType>;
      querySvg(): Promise<Element>;
      queryWadl(): Promise<Document>;
      query(): Promise<TraveltimeJsonType | Element | string>;
      formBaseURL(): string;
      formURL(): string;
      queryTauPVersion(): Promise<string>;
      formTauPVersionURL(): string;
      formWadlURL(): string;
  }
  export const FAKE_EMPTY_TEXT_MODEL = "Model: ";
  export const FAKE_EMPTY_TEXT_HEADERS = "\nDistance   Depth   Phase   Travel    Ray Param  Takeoff  Incident  Purist    Purist\n  (deg)     (km)   Name    Time (s)  p (s/deg)   (deg)    (deg)   Distance   Name\n-----------------------------------------------------------------------------------\n";
  export function createEmptyTraveltimeJson(ttquery: TraveltimeQuery): TraveltimeJsonType;
  export const FAKE_EMPTY_SVG = "\n<svg version=\"1.1\" baseProfile=\"full\" xmlns=\"http://www.w3.org/2000/svg\" width=\"500\" height=\"500\" viewBox=\"0 0 14016.2 14016.2\">\n<!--\n This script will plot ray paths generated by TauP using SVG. -->\n<defs>\n    <style type=\"text/css\"><![CDATA[\n        circle {\n            vector-effect: non-scaling-stroke;\n            stroke: grey;\n            fill: none;\n        }\n        polyline {\n            vector-effect: non-scaling-stroke;\n            stroke: black;\n            fill: none;\n        }\n    ]]></style>\n</defs>\n<g transform=\"translate(7008.1,7008.1)\" >\n<!-- draw surface and label distances.-->\n<!-- tick marks every 30 degrees.-->\n  <polyline points=\"    0.00  -6371.00,     0.00  -6689.55\" />\n  <polyline points=\" 3185.50  -5517.45,  3344.78  -5793.32\" />\n  <polyline points=\" 5517.45  -3185.50,  5793.32  -3344.77\" />\n  <polyline points=\" 6371.00      0.00,  6689.55      0.00\" />\n  <polyline points=\" 5517.45   3185.50,  5793.32   3344.77\" />\n  <polyline points=\" 3185.50   5517.45,  3344.78   5793.32\" />\n  <polyline points=\"    0.00   6371.00,     0.00   6689.55\" />\n  <polyline points=\"-3185.50   5517.45, -3344.77   5793.32\" />\n  <polyline points=\"-5517.45   3185.50, -5793.32   3344.77\" />\n  <polyline points=\"-6371.00      0.00, -6689.55      0.00\" />\n  <polyline points=\"-5517.45  -3185.50, -5793.32  -3344.78\" />\n  <polyline points=\"-3185.50  -5517.45, -3344.78  -5793.32\" />\n  <circle cx=\"0.0\" cy=\"0.0\" r=\"6371.0\" />\n  </g>\n</svg>\n";

}
declare module 'seisplotjs/src/usgsgeojson' {
  import { Quake, EventParameters } from "seisplotjs/src/quakeml";
  import type { Feature, Point } from "geojson";
  export const hourSummarySignificantUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_hour.geojson";
  export const hourSummaryM4_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/4.5_hour.geojson";
  export const hourSummaryM2_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/2.5_hour.geojson";
  export const hourSummaryM1_0Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/1.0_hour.geojson";
  export const hourSummaryAllUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_hour.geojson";
  export const daySummarySignificantUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_day.geojson";
  export const daySummaryM4_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/4.5_day.geojson";
  export const daySummaryM2_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/2.5_day.geojson";
  export const daySummaryM1_0Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/1.0_day.geojson";
  export const daySummaryAllUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_day.geojson";
  export const weekSummarySignificantUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_week.geojson";
  export const weekSummaryM4_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/4.5_week.geojson";
  export const weekSummaryM2_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/2.5_week.geojson";
  export const weekSummaryM1_0Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/1.0_week.geojson";
  export const weekSummaryAllUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_week.geojson";
  export const monthSummarySignificantUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/significant_month.geojson";
  export const monthSummaryM4_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/4.5_month.geojson";
  export const monthSummaryM2_5Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/2.5_month.geojson";
  export const monthSummaryM1_0Url = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/1.0_month.geojson";
  export const monthSummaryAllUrl = "https://earthquake.usgs.gov/earthquakes/feed/v1.0/summary/all_month.geojson";
  export function loadHourSummarySignificant(): Promise<Array<Quake>>;
  export function loadHourSummaryM4_5(): Promise<Array<Quake>>;
  export function loadHourSummaryM2_5(): Promise<Array<Quake>>;
  export function loadHourSummaryM1_0(): Promise<Array<Quake>>;
  export function loadHourSummaryAll(): Promise<Array<Quake>>;
  export function loadDaySummarySignificant(): Promise<Array<Quake>>;
  export function loadDaySummaryM4_5(): Promise<Array<Quake>>;
  export function loadDaySummaryM2_5(): Promise<Array<Quake>>;
  export function loadDaySummaryM1_0(): Promise<Array<Quake>>;
  export function loadDaySummaryAll(): Promise<Array<Quake>>;
  export function loadWeekSummarySignificant(): Promise<Array<Quake>>;
  export function loadWeekSummaryM4_5(): Promise<Array<Quake>>;
  export function loadWeekSummaryM2_5(): Promise<Array<Quake>>;
  export function loadWeekSummaryM1_0(): Promise<Array<Quake>>;
  export function loadWeekSummaryAll(): Promise<Array<Quake>>;
  export function loadMonthSummarySignificant(): Promise<Array<Quake>>;
  export function loadMonthSummaryM4_5(): Promise<Array<Quake>>;
  export function loadMonthSummaryM2_5(): Promise<Array<Quake>>;
  export function loadMonthSummaryM1_0(): Promise<Array<Quake>>;
  export function loadMonthSummaryAll(): Promise<Array<Quake>>;
  export function loadUSGSSummary(url: string): Promise<Array<Quake>>;
  export function loadUSGSGeoJsonSummary(url: string): Promise<EventParameters>;
  export function loadRawUSGSGeoJsonSummary(url: string): Promise<USGSGeoJsonSummary>;
  /**
   * Parses geojson from USGS feeds. Not all fields are parsed, just
   * basic origin and magnitude along with the id.
   * See https://earthquake.usgs.gov/earthquakes/feed/v1.0/geojson.php
   * @param  geojson  text from USGS feed
   * @returns EventParameters, which holds an array of Quake objects
   */
  export function parseGeoJSON(geojson: USGSGeoJsonSummary): EventParameters;
  /**
   * Parses a single GeoJson feature into a Quake.
   * @param  feature  from USGS style geojson
   * @returns Quake with origin and magnitude
   */
  export function parseFeatureAsQuake(feature: USGSGeoJsonFeature): Quake;
  export interface USGSGeoJsonMetaData {
      generated: number;
      url: string;
      title: string;
      api: string;
      count: number;
      status: number;
  }
  export interface USGSGeoJsonProperties {
      mag: number;
      place: string;
      time: number;
      updated: number;
      tz: number;
      url: string;
      detail: string;
      felt: number;
      cdi: number;
      mmi: number;
      alert: string;
      status: string;
      tsunami: number;
      sig: number;
      net: string;
      code: string;
      ids: string;
      sources: string;
      types: string;
      nst: number;
      dmin: number;
      rms: number;
      gap: number;
      magType: string;
      type: string;
      title: string;
  }
  export interface USGSGeoJsonFeature extends Feature<Point, USGSGeoJsonProperties> {
  }
  export interface USGSGeoJsonSummary {
      type: "FeatureCollection";
      metadata: USGSGeoJsonMetaData;
      features: Array<USGSGeoJsonFeature>;
  }
  export function isValidUSGSGeoJsonSummary(jsonValue: unknown): jsonValue is USGSGeoJsonSummary;
  export function isValidUSGSGeoJsonQuake(jsonValue: unknown): jsonValue is USGSGeoJsonFeature;

}
declare module 'seisplotjs/src/util' {
  import { DateTime, Duration, Interval, FixedOffsetZone } from "luxon";
  export const XML_MIME = "application/xml";
  export const JSON_MIME = "application/json";
  export const JSONAPI_MIME = "application/vnd.api+json";
  export const SVG_MIME = "image/svg+xml";
  export const TEXT_MIME = "text/plain";
  export const BINARY_MIME = "application/octet-stream";
  export const UTC_OPTIONS: {
      zone: FixedOffsetZone;
  };
  export function hasArgs(value: unknown): boolean;
  export function hasNoArgs(value: unknown): boolean;
  export function isStringArg(value: unknown): value is string;
  export function isNumArg(value: unknown): value is number;
  export function isNonEmptyStringArg(value: unknown): value is string;
  export function isObject(obj: unknown): obj is object;
  export function isDef<Value>(value: Value | undefined | null): value is Value;
  export function reErrorWithMessage(err: unknown, message: string): Error;
  export interface StringDictionary {
      [index: string]: unknown;
  }
  export function asStringDictionary(inobj: unknown): StringDictionary;
  export function doStringGetterSetter(inobj: unknown, field: string, value?: string): unknown;
  export function doBoolGetterSetter(inobj: unknown, field: string, value?: boolean): unknown;
  export function doIntGetterSetter(inobj: unknown, field: string, value?: number): unknown;
  export function doFloatGetterSetter(inobj: unknown, field: string, value?: number): StringDictionary;
  export function doMomentGetterSetter(inobj: unknown, field: string, value?: DateTime | string): StringDictionary;
  /**
   * Converts entire DataView to a string as utf-8.
   *
   * @param   dataView bytes to convert
   * @returns           the string
   */
  export function dataViewToString(dataView: DataView): string;
  /**
   * Log a message to the console. Put here to limit lint console errors
   * for the times we really do want to use console.log. Will also append a
   * p tag to a div#debug if it exists.
   *
   * @param   msg the message to log
   */
  export function log(msg: string): void;
  /**
   * typescript-y check if Error.
   *
   * @param error object that might be an Error
   * @returns true if Error object
   */
  export function isError(error: unknown): error is Error;
  /**
   * typescript-y convert errors.
   *
   * @param maybeError obejct that might be an Error object
   * @returns an Error object
   */
  export function toError(maybeError: unknown): Error;
  /**
   * Log a warning message to the console. Put here to limit lint console errors
   * for the times we really do want to use console.log. Will also append a
   * p tag to a div#debug if it exists.
   *
   * @param   msg the message to log
   */
  export function warn(msg: string): void;
  /**
   * String representation of input. This is kind of dumb but makes
   *  flow happier.
   *
   * @param value any kind of thing that can be turned into a string
   * @returns a string
   */
  export function stringify(value: unknown): string;
  export function isoToDateTime(val: string): DateTime;
  /**
   * Create a luxon Interval from a start and end.
   *
   * @param  start         start of the interval as iso string or DateTime
   * @param  end         end of the interval as string or DateTime
   * @returns          the interval
   */
  export function startEnd(start: string | DateTime, end: string | DateTime): Interval;
  /**
   * Create a luxon Interval from a start and a duration. If the duration is negative, the
   * start time will become the end time. This differs from luxon Interval.after which
   * will return an invalid Interval instead.
   *
   * @param  start         start of the interval as iso string or DateTime
   * @param  duration      duration of the interval as iso string, number of seconds, or Duration
   * @returns          the interval
   */
  export function startDuration(start: string | DateTime, duration: string | Duration | number): Interval;
  /**
   * Create a luxon Interval from a duration and a end. If the duration is negative, the
   * end time will become the start time. This differs from luxon Interval.before which
   * will return an invalid Interval instead.
   *
   * @param  duration      duration of the interval as iso string, number of seconds, or Duration
   * @param  end         end of the interval as string or DateTime
   * @returns          the interval
   */
  export function durationEnd(duration: string | Duration | number, end: string | DateTime): Interval;
  /**
   * Calculates offset of remote server versus local time. It is assumed that the
   * argument was acquired as close in time to calling this as possible.
   *
   * @param  serverTimeUTC now as reported by remote server
   * @returns offset in seconds to now on local machine
   */
  export function calcClockOffset(serverTimeUTC: DateTime): number;
  export const WAY_FUTURE: DateTime;
  /**
   * converts the input value is a DateTime, throws Error if not
   * a string, Date or DateTime. Zero length string or "now" return
   * current time.
   *
   * @param d 'now', string time, Date, number of milliseconds since epoch, or DateTime
   * @returns DateTime created from argument
   */
  export function checkStringOrDate(d: string | Date | DateTime): DateTime;
  /**
   * Converts name and value into a html query parameter, with appending ampersand.
   *
   * @param   name parameter name
   * @param   val  parameter value
   * @returns      formated query parameter
   */
  export function makeParam(name: string, val: unknown): string;
  /**
   * Converts name and value into a parameter line, with appending newline,
   * for including in POST body.
   *
   * @param   name parameter name
   * @param   val  parameter value
   * @returns      formated query parameter
   */
  export function makePostParam(name: string, val: unknown): string;
  /**
   * converts to ISO8601 but removes the trailing Z as FDSN web services
   * do not allow that.
   *
   * @param  date DateTime to convert to string
   * @returns ISO8601 without timezone Z
   */
  export function toIsoWoZ(date: DateTime): string;
  /**
   * Extracts a valid starting DateTime from interval.
   * Throws Error if interval is not valid.
   * @param  interval              luxon Interval
   * @returns          start DateTime
   */
  export function validStartTime(interval: Interval): DateTime;
  /**
   * Extracts a valid ending DateTime from interval.
   * Throws Error if interval is not valid.
   * @param  interval              luxon Interval
   * @returns          end DateTime
   */
  export function validEndTime(interval: Interval): DateTime;
  /**
   * Converts a luxon DateTime to a Javascript Date, checking for null,
   * undefined and isValid first. Throws Error in that case.
   *
   * @param  d  luxon DateTime
   * @returns   Javascript Date
   */
  export function toJSDate(d: DateTime | null | undefined): Date;
  /**
   * Check a Luxon DateTime, Interval or Duration for valid.
   * Throws Error if not. THis is to avoid globally setting
   * luxon's Settings.throwOnInvalid = true;
   * but still throw/catch on invalid dates.
   * @param  d                 luxon object
   * @param  msg               optional message to add to error
   * @returns  passed in object if valid
   */
  export function checkLuxonValid(d: null | DateTime | Interval | Duration, msg?: string): Duration<boolean> | DateTime<boolean> | Interval<boolean>;
  /**
   * @returns the protocol, http: or https: for the document if possible.
   * Note this includes the colon.
   */
  export function checkProtocol(): string;
  export interface FetchInitObject {
      cache: string;
      redirect: string;
      mode: string;
      referrer: string;
      headers: Record<string, string>;
      signal?: AbortSignal;
  }
  /**
   * Create default fetch init object with the given mimeType. Sets
   * no-cache, follow redirects, cors mode, referrer as seisplotjs and
   * mimetype as a header. Note that redirect with POST may fail due to
   * POST being changed to GET on a 301. Fetching with POST may wish
   * to use redirect: "manual" to handle the 301 correctly by POSTing to
   * the new URL.
   *
   *
   * @param   mimeType requested mime type
   * @returns           object with fetch configuration parameters
   */
  export function defaultFetchInitObj(mimeType?: string): RequestInit;
  export function cloneFetchInitObj(fetchInit: RequestInit): RequestInit;
  export function errorFetch(_url: URL | RequestInfo, _init?: RequestInit | undefined): Promise<Response>;
  export let default_fetch: null | ((url: URL | RequestInfo, init?: RequestInit | undefined) => Promise<Response>);
  export function setDefaultFetch(fetcher: (url: URL | RequestInfo, init?: RequestInit | undefined) => Promise<Response>): void;
  export function getFetch(): (url: URL | RequestInfo, init?: RequestInit | undefined) => Promise<Response>;
  /**
   * Does a fetch, but times out if it takes too long.
   *
   * @param   url        url to retrieve
   * @param   fetchInit  fetch configuration, initialization
   * @param   timeoutSec maximum time to wait in seconds
   * @param   fetcher optional fetch to use instead of global fetch
   * @returns             promise to the result
   * @throws Error if time out or other failure
   */
  export function doFetchWithTimeout(url: string | URL, fetchInit?: RequestInit, timeoutSec?: number, fetcher?: (url: URL | RequestInfo, init?: RequestInit | undefined) => Promise<Response>): Promise<Response>;
  /**
   * Allows downloading of in memory data, as ArrayBuffer, to file as if
   * the user clicked a download link.
   *
   * @param  data               ArrayBuffer to download
   * @param  filename          default filename
   * @param  mimeType      mimeType, default application/octet-stream
   */
  export function downloadBlobAsFile(data: ArrayBuffer, filename: string, mimeType?: string): void;
  /**
   * Recursively calculates the mean of a slice of an array. This helps with
   * very long seismograms to equally weight each sample point without overflowing.
   *
   * @param   dataSlice slice of a seismogram
   * @param   totalPts  number of points in the original seismogram
   * @returns            sum of slice data points divided by totalPts
   */
  export function meanOfSlice(dataSlice: Int32Array | Float32Array | Float64Array, totalPts: number): number;
  export const SVG_NS = "http://www.w3.org/2000/svg";
  export const XHTML_NS = "http://www.w3.org/1999/xhtml";
  export function createSVGElement(name: string): SVGElement;
  export function updateVersionText(selector?: string): void;

}
declare module 'seisplotjs/src/vector' {
  import { Seismogram } from "seisplotjs/src/seismogram";
  /**
   * const for degrees to radians, pi/180
   */
  export const DtoR: number;
  /**
   * Result of rotation for 2 seismograms.
   */
  export class RotatedSeismograms {
      radial: Seismogram;
      transverse: Seismogram;
      azimuthRadial: number;
      azimuthTransverse: number;
      rotation: number;
      constructor(radial: Seismogram, azimuthRadial: number, transverse: Seismogram, azimuthTransverse: number, rotation: number);
  }
  /**
   * Rotates the given seismograms from their given azimuths so the output radial
   * is along the new azimuth and the output transverse is perpendicular to it.
   *
   * @param   seisA    first seismogram
   * @param   azimuthA azimuth of first seismogram
   * @param   seisB    second seismogram
   * @param   azimuthB azimuth of second seismogram
   * @param   azimuth  output radial azimuth to rotate to
   * @returns radial and transverse seismograms
   */
  export function rotate(seisA: Seismogram, azimuthA: number, seisB: Seismogram, azimuthB: number, azimuth: number): RotatedSeismograms;
  /**
   * creates a new Seismogram where the value at each sample is the
   * vector magnitude of the 3 corresponding data points from each seismogram.
   * Each of the 3 seismograms are assumed to be mutually perpendicular so
   * that each set of samples gives a vector in 3-dimensional space. In particular
   * all three seismograms must have the same number of samples and sample rate.
   * It is assumed, but not checked, that they will be the three components of
   * motion at a station (ie matching network, station and location codes)
   * and have the same start time.
   *
   * @param   seisA first seismogram
   * @param   seisB second seismogram
   * @param   seisC third seismogram
   * @param   orientCode optional orientation code for resulting seismogram, defaults to M
   * @returns Seismogram of vector magnitudes
   */
  export function vectorMagnitude(seisA: Seismogram, seisB: Seismogram, seisC: Seismogram, orientCode?: string | null): Seismogram;

}
declare module 'seisplotjs/src/version' {
  export const version = "3.1.4";

}
declare module 'seisplotjs/test/checkProtocol.test' {
  export {};

}
declare module 'seisplotjs/test/distaz.spec' {
  export {};

}
declare module 'seisplotjs/test/fdsnavailability.test' {
  export {};

}
declare module 'seisplotjs/test/fdsndatacenters.test' {
  export {};

}
declare module 'seisplotjs/test/fdsndataselect/fdsndataselect.test' {
  export {};

}
declare module 'seisplotjs/test/fdsnevent/fdsnevent.test' {
  export {};

}
declare module 'seisplotjs/test/fdsnsourceid.test' {
  export {};

}
declare module 'seisplotjs/test/fdsnstation/fdsnstation.test' {
  export {};

}
declare module 'seisplotjs/test/fdsnstation/util.test' {
  export {};

}
declare module 'seisplotjs/test/filter/cut.test' {
  export {};

}
declare module 'seisplotjs/test/filter/differentiate.test' {
  import '../jestRatioMatchers';

}
declare module 'seisplotjs/test/filter/fft.test' {
  import '../jestRatioMatchers';

}
declare module 'seisplotjs/test/filter/filter.test' {
  export {};

}
declare module 'seisplotjs/test/filter/hilbert.test' {
  export {};

}
declare module 'seisplotjs/test/filter/integrate.test' {
  import '../jestRatioMatchers';

}
declare module 'seisplotjs/test/filter/rtrend.test' {
  export {};

}
declare module 'seisplotjs/test/filter/sacfile' {
  import { SacPoleZero } from 'seisplotjs/src/sacpolezero';
  import { Seismogram } from 'seisplotjs/src/seismogram';
  import { DateTime } from 'luxon';
  export const NVHDR_OFFSET: number;
  export const NPTS_OFFSET: number;
  export const DATA_OFFSET = 632;
  export const YEAR_OFFSET: number;
  export const DAY_OF_YEAR_OFFSET: number;
  export const HOUR_OFFSET: number;
  export const MIN_OFFSET: number;
  export const SEC_OFFSET: number;
  export const MSEC_OFFSET: number;
  export type sacType = {
      littleEndian: boolean;
      delta: number;
      npts: number;
      start: DateTime;
      y: Float32Array;
  };
  export function readDataView(filename: string): Promise<DataView>;
  export function readSac(filename: string): Promise<sacType>;
  export function asSeismogram(sac: sacType): Seismogram;
  export function readSeismogram(filename: string): Promise<Seismogram>;
  export function parseSac(dataView: DataView): sacType;
  export function replaceYData(dataView: DataView, yData: Float32Array): DataView;
  export function writeSac(sacDataView: DataView, filename: string): Promise<void>;
  export function readSacPoleZero(filename: string): Promise<SacPoleZero>;

}
declare module 'seisplotjs/test/filter/sacpolezero.test' {
  export {};

}
declare module 'seisplotjs/test/filter/taper.test' {
  import '../jestRatioMatchers';

}
declare module 'seisplotjs/test/filter/transfer.test' {
  import '../jestRatioMatchers';

}
declare module 'seisplotjs/test/irisfedcatalog.test' {
  export {};

}
declare module 'seisplotjs/test/jestRatioMatchers' {
  global {
      namespace jest {
          interface Matchers<R> {
              toBeCloseToRatio(argument: number, digits: number, closeZero?: number, closeZeroDigits?: number): R;
              arrayToBeCloseTo(argument: Array<number> | Int32Array | Float32Array | Float64Array, digits: number): R;
              arrayToBeCloseToRatio(argument: Array<number> | Int32Array | Float32Array | Float64Array, digits: number, closeZero?: number, closeZeroDigits?: number): R;
          }
      }
  }
  export {};

}
declare module 'seisplotjs/test/metatest.test' {
  import './jestRatioMatchers';

}
declare module 'seisplotjs/test/miniseed/headerTimes.test' {
  export {};

}
declare module 'seisplotjs/test/miniseed/miniseed_file.test' {
  export {};

}
declare module 'seisplotjs/test/mseed3/mseed3_file.test' {
  export {};

}
declare module 'seisplotjs/test/mseed3/mseed3eh.test' {
  export {};

}
declare module 'seisplotjs/test/quakeml/quakeml.test' {
  export {};

}
declare module 'seisplotjs/test/ringserverweb.test' {
  export {};

}
declare module 'seisplotjs/test/scale.test' {
  export {};

}
declare module 'seisplotjs/test/seismogram.test' {
  export {};

}
declare module 'seisplotjs/test/seismographconfig.test' {
  export {};

}
declare module 'seisplotjs/test/seismographutil.test' {
  export {};

}
declare module 'seisplotjs/test/stationxml/network.test' {
  export {};

}
declare module 'seisplotjs/test/stationxml/parsestationxml.test' {
  export {};

}
declare module 'seisplotjs/test/stationxml/station.test' {
  export {};

}
declare module 'seisplotjs/test/traveltime/formUrl.test' {
  export {};

}
declare module 'seisplotjs/test/usgs_geojson.test' {
  export {};

}
declare module 'seisplotjs/test/util.test' {
  export {};

}
declare module 'seisplotjs/test/vector.test' {
  import './jestRatioMatchers';

}
declare module 'seisplotjs/testremotes/fdsnavailability_livetest' {
  export {};

}
declare module 'seisplotjs/testremotes/fdsnstation_livetest' {
  export {};

}
declare module 'seisplotjs/testremotes/irisfedcatalog_livetest' {
  export {};

}
declare module 'seisplotjs/testremotes/ringserver_livetest' {
  export {};

}
declare module 'seisplotjs/testremotes/seismogramload_livetest' {
  export {};

}
declare module 'seisplotjs/testremotes/traveltime.livetest' {
  export {};

}
declare module 'seisplotjs/testremotes/usgsgeojson_livetest' {
  export {};

}
declare module 'seisplotjs' {
  import main = require('seisplotjs/src/index_node');
  export = main;
}