package edu.sc.seis.sod.process.waveform.vector;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;

import java.io.DataInputStream;

import org.junit.Test;

import edu.iris.Fissures.model.SamplingImpl;
import edu.iris.Fissures.model.TimeInterval;
import edu.iris.Fissures.model.UnitImpl;
import edu.iris.Fissures.seismogramDC.LocalSeismogramImpl;
import edu.sc.seis.fissuresUtil.freq.Cmplx;
import edu.sc.seis.fissuresUtil.mockFissures.IfSeismogramDC.MockSeismogram;
import edu.sc.seis.fissuresUtil.sac.FissuresToSac;
import edu.sc.seis.seisFile.sac.SacTimeSeries;
import edu.sc.seis.sod.process.waveform.OregonDspFFT;

// JUnitDoclet end import
/**
 * Generated by JUnitDoclet, a tool provided by ObjectFab GmbH under LGPL.
 * Please see www.junitdoclet.org, www.gnu.org and www.objectfab.de for
 * informations about the tool, the licence and the authors.
 */
public class IterDeconTest
{

    IterDecon iterdecon = null;

    public IterDecon createInstance() throws Exception {
        return new IterDecon(100, true, .0001f, 3);
    }

    protected void setUp() throws Exception {
        iterdecon = createInstance();
    }

    protected void tearDown() throws Exception {
        iterdecon = null;
    }
    
    public static float[] createTestData() {
        float[] data = new float[1024];
        data[3] = 6;
        data[23] = 9;
        data[67] = 10;
        data[100] = 2;
        data[200] = -1.5f;
        data[300] = .25f;
        return data;
    }

    @Test
    public void testUseOregonDSPFFT() {
        float[] data = createTestData();
        IterDecon.useOregonDSPFFT = true;
        float[] oregonFFT = IterDecon.forwardFFT(data);
        float[] oregonFFTDirect = OregonDspFFT.forward(data);
        // oregonDSP uses opposite sign convention on imag part, so mul -1
        for (int i = oregonFFTDirect.length/2+1; i < oregonFFTDirect.length; i++) {
            oregonFFTDirect[i] *= -1;
        }
        assertArrayEquals(oregonFFTDirect, oregonFFT, 0.0001f);
    }
    
    @Test
    public void testForwardFFT() throws Exception {
        float[] data = createTestData();
        IterDecon.useOregonDSPFFT = false;
        float[] origFFT = IterDecon.forwardFFT(data);
        IterDecon.useOregonDSPFFT = true;
        float[] oregonFFT = IterDecon.forwardFFT(data);
        Cmplx[] cData = Cmplx.fft(data);
        assertEquals("length ", origFFT.length, oregonFFT.length);
        assertEquals(origFFT[0], oregonFFT[0], 0.000001f);
        assertEquals(origFFT[1], oregonFFT[data.length/2], 0.000001f);
        for (int i = 1; i < data.length/2; i++) {
            assertEquals("real "+i, cData[i].r, origFFT[2*i], 0.000001f);
            assertEquals("imag "+i, cData[i].i, origFFT[2*i+1], 0.000001f);
            assertEquals("real "+i, cData[i].r, oregonFFT[i], 0.00001f);
            assertEquals("imag "+i, cData[i].i, oregonFFT[data.length-i], 0.00001f);
            assertEquals("real "+i, origFFT[2*i], oregonFFT[i], 0.00001f);
            assertEquals("imag "+i, origFFT[2*i+1], oregonFFT[data.length-i], 0.00001f);
        }
    }
    
    @Test
    public void testInverseFFT() throws Exception {
        float[] data = createTestData();
        IterDecon.useOregonDSPFFT = false;
        float[] forward = IterDecon.forwardFFT(data);
        float[] origInverse = IterDecon.inverseFFT(forward);
        IterDecon.useOregonDSPFFT = true;
        forward = IterDecon.forwardFFT(data);
        float[] oregoninverse = IterDecon.inverseFFT(forward);
        assertEquals("length ", data.length, origInverse.length);
        assertEquals("length ", data.length, oregoninverse.length);
        assertArrayEquals(data, origInverse, 0.001f);
        assertArrayEquals(data, oregoninverse, 0.001f);
    }
    
    @Test
    public void testNoGauussProcess() throws Exception {
        float delta = 0.05f;
        float[] numData = new float[2048];
        numData[100] = 2;
        numData[200] = -1.5f;
        numData[300] = .25f;
        float[] denomData = new float[2048];
        denomData[100] = .5f;
        // without gaussian filter
        IterDecon zeroGauss = new IterDecon(3, true, 0.001f, 0.0f);
        IterDeconResult result = zeroGauss.process(numData, denomData, delta);
        float[] pred = result.getPredicted();
        int[] s = result.getShifts();
        float[] a = result.getAmps();
        assertEquals("zeroGauss spike 0", 0, s[0]);
        assertEquals("zeroGauss amp 0", 4 / delta, a[0], 0.0001f);
        assertEquals("zeroGauss spike 1", 100, s[1]);
        assertEquals("zeroGauss amp 1", -3 / delta, a[1], 0.0001f);
        assertEquals("zeroGauss spike 2 a=" + a[2], 200, s[2]);
        assertEquals("zeroGauss amp 2", .5f / delta, a[2], 0.0001f);
        assertEquals("zeroGauss pred 0", 4 / delta, pred[0], 0.0001f);
        assertEquals("zeroGauss pred 1", 0, pred[1], 0.0001f);
        assertEquals("zeroGauss pred 100", -3 / delta, pred[100], 0.0001f);
        assertEquals("zeroGauss pred 101", 0, pred[3], 0.0001f);
        assertEquals("zeroGauss pred 200", .5f / delta, pred[200], 0.0001f);
        assertEquals("zeroGauss pred 201", 0, pred[5], 0.0001f);
    }

    @Test
    public void testWithGauussProcess() throws Exception {
        float delta = 0.05f;
        IterDecon withGaussIterdecon = new IterDecon(200, true, .001f, 2.5f);
        // with gaussian filter
        float[] numData = new float[2048];
        numData[100] = 2;
        numData[200] = -1.5f;
        numData[300] = .25f;
        float[] denomData = new float[numData.length];
        denomData[100] = .5f;
        SamplingImpl sampling = new SamplingImpl(1, new TimeInterval(delta, UnitImpl.SECOND));
        LocalSeismogramImpl fakeNum = MockSeismogram.createTestData("num");
        fakeNum.setData(numData);
        fakeNum.channel_id.channel_code = "BHR";
        fakeNum.sampling_info = sampling;
        LocalSeismogramImpl fakeDenom = MockSeismogram.createTestData("denom");
        fakeDenom.setData(denomData);
        fakeDenom.channel_id.channel_code = "BHZ";
        fakeDenom.sampling_info = sampling;
        SacTimeSeries sac = FissuresToSac.getSAC(fakeNum);
        sac.write("withGauss.BHR.sac");
        sac = FissuresToSac.getSAC(fakeDenom);
        sac.write("withGauss.BHZ.sac");
        sac = null;
        IterDeconResult result = withGaussIterdecon.process(numData, denomData, delta);
        float[] pred = result.getPredicted();
        pred = IterDecon.phaseShift(pred, 5, delta);
        LocalSeismogramImpl predSeis = MockSeismogram.createTestData("denom");
        predSeis.setData(pred);
        predSeis.channel_id.channel_code = "OUT";
        sac = FissuresToSac.getSAC(predSeis);
        sac.write("withGauss.ITR.sac");
        DataInputStream in = new DataInputStream(this.getClass()
                .getClassLoader()
                .getResourceAsStream("edu/sc/seis/sod/process/waveform/vector/iterDeconTest/withGauss.predicted.sac"));
        sac.read(in);
        float[] fortranData = sac.getY();
        int[] s = result.getShifts();
        float[] a = result.getAmps();
        assertEquals("gauss spike 0", 0, s[0]);
        assertEquals("gauss amp 0", 4 / delta, a[0], 0.001f);
        assertEquals("gauss spike 1", 100, s[1]);
        assertEquals("gauss amp 1", -3 / delta, a[1], 0.0001f);
        assertEquals("gauss spike 2 a=" + a[2], 200, s[2]);
        assertEquals("gauss amp 2", .5f / delta, a[2], 0.0001f);
        assertEquals("position 0 " + fortranData[0] + "  " + pred[0] + "  ratio=" + (fortranData[0] / pred[0]),
                     fortranData[0],
                     pred[0],
                     0.0001f);
        assertEquals("position 100 " + fortranData[100] + "  " + pred[100] + "  ratio="
                + (fortranData[100] / pred[100]), fortranData[100], pred[100], 0.0001f);
        assertEquals("position 200 " + fortranData[200] + "  " + pred[200] + "  ratio="
                + (fortranData[200] / pred[200]), fortranData[200], pred[200], 0.0001f);
        assertEquals("position 300 " + fortranData[300] + "  " + pred[300] + "  ratio="
                + (fortranData[300] / pred[300]), fortranData[300], pred[300], 0.0001f);
        assertArrayEquals("data from fortran", fortranData, pred, 0.0001f);
    }

    @Test
    public void testFakeCrustProcess() throws Exception {
        float delta = 0.05f;
        // with more complex demon
        float[] denomData = new float[2048];
        denomData[100] = .15f;
        denomData[101] = .5f;
        denomData[102] = .9f;
        denomData[103] = 1.1f;
        denomData[104] = .8f;
        denomData[105] = .4f;
        denomData[106] = .1f;
        denomData[107] = -.3f;
        denomData[108] = -.6f;
        denomData[109] = -.4f;
        denomData[110] = -.1f;
        denomData[111] = .1f;
        // create fake crust with Vp=6 and Vs=3.5, h=30
        float alpha = 6;
        float beta = 3.5f;
        float h = 30;
        float p = 7.6f / 111.19f;
        float etaP = (float)Math.sqrt(1 / (alpha * alpha) - p * p);
        float etaS = (float)Math.sqrt(1 / (beta * beta) - p * p);
        float timePs = h * (etaS - etaP);
        float timePpPs = h * (etaS + etaP);
        float timePsPs = h * (2 * etaS);
        // System.out.println("timePs="+timePs+"  timePpPs="+timePpPs+"  timePsPs="+timePsPs);
        float[] numData = new float[denomData.length];
        System.arraycopy(denomData, 0, numData, 0, denomData.length);
        // scale num by 1/3
        for (int i = 0; i < numData.length; i++) {
            numData[i] *= .33f;
        }
        float[] temp = new float[numData.length];
        System.arraycopy(denomData, 0, temp, 0, denomData.length);
        temp = IterDecon.phaseShift(temp, timePs, delta);
        // scale num by 1/5
        for (int i = 0; i < temp.length; i++) {
            numData[i] += .33f * .50f * temp[i];
        }
        System.arraycopy(denomData, 0, temp, 0, denomData.length);
        temp = IterDecon.phaseShift(temp, timePpPs, delta);
        // scale num
        for (int i = 0; i < temp.length; i++) {
            numData[i] += .33f * .3f * temp[i];
        }
        System.arraycopy(denomData, 0, temp, 0, denomData.length);
        temp = IterDecon.phaseShift(temp, timePsPs, delta);
        // scale num
        for (int i = 0; i < temp.length; i++) {
            numData[i] += .33f * .2f * temp[i];
        }
        IterDecon iterdecon = new IterDecon(100, true, .0001f, 3);
        IterDeconResult result = iterdecon.process(numData, denomData, delta);
        int[] s = result.getShifts();
        float[] a = result.getAmps();
        assertEquals("fake data spike 0", 0, s[0]);
        assertEquals("fake data amp 0", .33 / delta, a[0], 0.0001f);
        assertEquals("fake data spike 1", Math.round(timePs / delta), s[1], 0.1f);
        assertEquals("fake data amp 1", .33f * .5f / delta, a[1], 0.01f);
        assertEquals("fake data spike 2 a=" + a[2], Math.round(timePpPs / delta), s[2], 0.1f);
        assertEquals("fake data amp 2", .33f * .3f / delta, a[2], 0.01f);
        assertEquals("fake data spike 3 a=" + a[3], Math.round(timePsPs / delta), s[3], 0.1f);
        assertEquals("fake data amp 3", .33f * .2f / delta, a[3], 0.01f);
        // JUnitDoclet end method process
    }

    @Test
    public void testESK1999_312_16_45_41_6() throws Exception {
        SacTimeSeries sac = new SacTimeSeries();
        DataInputStream in = new DataInputStream(this.getClass()
                .getClassLoader()
                .getResourceAsStream("edu/sc/seis/sod/process/waveform/vector/iterDeconTest/ESK1999_312_16.predicted.sac"));
        sac.read(in);
        in.close();
        float[] fortranData = sac.getY();
        in = new DataInputStream(this.getClass()
                .getClassLoader()
                .getResourceAsStream("edu/sc/seis/sod/process/waveform/vector/iterDeconTest/ESK_num.sac"));
        sac.read(in);
        in.close();
        float[] num = sac.getY();
        in = new DataInputStream(this.getClass()
                .getClassLoader()
                .getResourceAsStream("edu/sc/seis/sod/process/waveform/vector/iterDeconTest/ESK_denom.sac"));
        sac.read(in);
        in.close();
        float[] denom = sac.getY();
        IterDecon iterdecon = new IterDecon(100, true, .0001f, 3);
        IterDeconResult result = iterdecon.process(num, denom, sac.getHeader().getDelta());
        float[] pred = result.getPredicted();
        pred = IterDecon.phaseShift(pred, 5, sac.getHeader().getDelta());
        int[] s = result.getShifts();
        float[] a = result.getAmps();
        int i = 0;
        // spikes from fortran are in time, div delta to get index
        // output from fortran iterdecon_tjo is:
        // The maximum spike delay is 102.40012
        //
        // File Spike amplitude Spike delay Misfit Improvement
        // r001 0.384009242E+00 0.100 48.98% 51.0211%
        // r002 -0.132486761E+00 16.250 42.91% 6.0732%
        // r003 0.116493061E+00 2.250 38.21% 4.6952%
        // r004 -0.988256037E-01 10.800 34.83% 3.3792%
        // r005 -0.606716201E-01 15.450 33.56% 1.2736%
        // r006 -0.635700300E-01 20.650 32.16% 1.3983%
        // r007 -0.568093359E-01 41.350 31.04% 1.1166%
        // r008 0.520336218E-01 3.950 30.11% 0.9368%
        // r009 0.494165495E-01 1.000 29.26% 0.8449%
        // r010 -0.416982807E-01 79.850 28.66% 0.6015%
        // ... snip ...
        // r100 0.105094928E-01 3.500 14.55% 0.0382%
        //
        // Last Error Change = 0.0382%
        //
        // Number of bumps in final result: 100
        // The final deconvolution reproduces 85.4% of the signal.
        assertEquals("spike " + i, 0.100 / sac.getHeader().getDelta(), s[i], 0.1f);
        assertEquals("amp   " + i, 0.384009242 / sac.getHeader().getDelta(), a[i], 0.001f);
        i++;
        assertEquals("spike " + i, 16.250 / sac.getHeader().getDelta(), s[i], 0.1f);
        assertEquals("amp   " + i, -0.132486761 / sac.getHeader().getDelta(), a[i], 0.001f);
        i++;
        assertEquals("spike " + i, 2.250 / sac.getHeader().getDelta(), s[i], 0.1f);
        assertEquals("amp   " + i, 0.116493061 / sac.getHeader().getDelta(), a[i], 0.001f);
        i++;
        assertEquals("spike " + i, 10.800 / sac.getHeader().getDelta(), s[i], 0.1f);
        assertEquals("amp   " + i, -0.0988256037 / sac.getHeader().getDelta(), a[i], 0.001f);
        i++;
        assertEquals("spike " + i, 15.450 / sac.getHeader().getDelta(), s[i], 0.1f);
        assertEquals("amp   " + i, -0.0606716201 / sac.getHeader().getDelta(), a[i], 0.001f);
        i++;
        assertArrayEquals("fortran predicted", fortranData, pred, 0.000001f);
        assertEquals("percent match", 85.4, result.getPercentMatch(), 0.1f);
    }

    @Test
    public void testZeroPhaseShift() throws Exception {
        float[] data = new float[1024];
        data[10] = 1;
        float[] out = IterDecon.phaseShift(data, 0.00f, 0.05f);
        IterDecon.useOregonDSPFFT = false;
         float[] oldout = IterDecon.phaseShift(data, 0.0f, 0.05f);
         IterDecon.useOregonDSPFFT = true;
         for ( int i=0; i<data.length; i++) {
             if (data[i] > 0.01f || out[i] > 0.001f || oldout[i] > 0.001f) {
         System.out.println(i+" data="+data[i]+"  out="+out[i]+"  oldout="+oldout[i]);
             }
         } // end of for ()
        // expected actual
        assertArrayEquals("no shift", data, out, 0.001f);
    }

    @Test
    public void testOnePhaseShift() throws Exception {
        // JUnitDoclet begin method phaseShift
        float[] data = new float[1024];
        data[10] = 1;
        float[] out = IterDecon.phaseShift(data, 0.05f, 0.05f);
        // float[] oldout = iterdecon.oldphaseShift(data, 1.0f, 1.0f);
        // for ( int i=0; i<data.length; i++) {
        // System.out.println("data="+data[i]+"  out="+out[i]);
        // } // end of for ()
        // expected actual
        assertEquals("9 shifts to 10", data[9], out[10], .001);
        assertEquals("10 shifts to 11", data[10], out[11], .001);
        assertEquals("11 shifts to 12", data[11], out[12], .001);
        // JUnitDoclet end method phaseShift
    }

    @Test
    public void testFivePhaseShift() throws Exception {
        // JUnitDoclet begin method phaseShift
        float[] data = new float[1024];
        data[10] = 1;
        data[11] = 2;
        data[12] = 1.1f;
        float[] out = IterDecon.phaseShift(data, 5f, 0.05f);
        // expected actual
        assertEquals("9 shifts to 109", data[9], out[109], .001);
        assertEquals("10 shifts to 110", data[10], out[110], .001);
        assertEquals("11 shifts to 111", data[11], out[111], .001);
        assertEquals("12 shifts to 112", data[11], out[111], .001);
        assertEquals("13 shifts to 113", data[11], out[111], .001);
        // JUnitDoclet end method phaseShift
    }

    @Test
    public void testNextPowerTwo() throws Exception {
        // JUnitDoclet begin method phaseShift
        assertEquals(IterDecon.nextPowerTwo(3), 4);
        assertEquals(IterDecon.nextPowerTwo(4), 4);
        assertEquals(IterDecon.nextPowerTwo(1024), 1024);
        assertEquals(IterDecon.nextPowerTwo(1025), 2048);
        // JUnitDoclet end method phaseShift
    }

    /**
     * Gaussian filter of constant should do nothing.
     */
    @Test
    public void testGaussianFilter() throws Exception {
        SacTimeSeries sac = new SacTimeSeries();
        DataInputStream in = new DataInputStream(this.getClass()
                .getClassLoader()
                .getResourceAsStream("edu/sc/seis/sod/process/waveform/vector/iterDeconTest/gauss1024.sac"));
        sac.read(in);
        float[] data = new float[sac.getHeader().getNpts()];
        data[100] = 1 / sac.getHeader().getDelta();
        float[] sacData = sac.getY();
        IterDecon.useOregonDSPFFT = false;
        float[] out = IterDecon.gaussianFilter(data, 2.5f, sac.getHeader().getDelta());
        assertArrayEquals("gaussian filter", sacData, out, 0.0001f);
        IterDecon.useOregonDSPFFT = true;
        float[] outOregon = IterDecon.gaussianFilter(data, 2.5f, sac.getHeader().getDelta());
        assertArrayEquals("gaussian filter", out, outOregon, 0.0001f);
        assertArrayEquals("gaussian filter", sacData, outOregon, 0.011f);
        if (IterDecon.useNativeFFT) {
            // test non-native as well
            IterDecon.useNativeFFT = false;
            out = IterDecon.gaussianFilter(data, 2.5f, sac.getHeader().getDelta());
            assertArrayEquals("gaussian filter", sacData, out, 0.01f);
            IterDecon.useNativeFFT = true;
        }
    }

    /*
     * public void testNativeVsJavaFFT() { float[] data = new float[1024];
     * data[100] = 1; float[] nativeFFT = new float[data.length];
     * System.arraycopy(data, 0, nativeFFT, 0, data.length);
     * NativeFFT.forward(nativeFFT);
     * 
     * float[] javaFFT = new float[data.length*2]; for(int i = 0; i <
     * data.length; i++) { javaFFT[2*i] = data[i]; } javaFFT =
     * Cmplx.four1Forward(javaFFT); assertEquals(2*data.length, javaFFT.length);
     * javaFFT = IterDecon.shortenFFT(javaFFT); assertEquals(nativeFFT.length,
     * javaFFT.length); ArrayAssert.assertEquals(nativeFFT, javaFFT, 0.0001f);
     * for(int i = 0; i < nativeFFT.length; i++) { assertEquals(""+i,
     * nativeFFT[i], javaFFT[i], 0.00001f); } }
     */
    @Test
    public void testNativeVsJavaInverseFFT() {
        float[] data = new float[32];
        data[10] = 1;
        float[] javaFFT = new float[data.length];
        System.arraycopy(data, 0, javaFFT, 0, data.length);
        javaFFT = IterDecon.shortenFFT(Cmplx.four1Forward(javaFFT));
        float[] invJavaFFT = Cmplx.four1Inverse(IterDecon.lengthenFFT(javaFFT));
        assertArrayEquals(data, invJavaFFT, 0.00001f);
        for (int i = 0; i < data.length; i++) {
            assertEquals("" + i, data[i], invJavaFFT[i], 0.00001f);
        }
    }

    @Test
    public void testGetMinIndex() {
        float[] data = {3, 4, -5, 0, 4, 4, 0, -5, 4, 3};
        int index = IterDecon.getMinIndex(data);
        assertEquals("min index", 2, index);
        index = IterDecon.getMaxIndex(data);
        assertEquals("max index", 1, index);
        index = IterDecon.getAbsMaxIndex(data);
        assertEquals("abs max index", 2, index);
    }

    @Test
    public void testPower() {
        float[] data = {0, 2, 3, -1, -2, 0};
        assertEquals(18f, IterDecon.power(data), 0.00001f);
    }

    @Test
    public void testCorrelationNorm() {
        float[] fData = {0, 0, 2, 0, 0, 0, 0, 0};
        float[] gData = {0, 2, 0, 0, 0, 0, 0, 0};
        float[] corr = IterDecon.correlateNorm(fData, gData);
        assertEquals("lag 0", 0f, corr[0], 0.00001f);
        assertEquals("lag 1", 1f, corr[1], 0.00001f);
        assertEquals("lag 2", 0f, corr[2], 0.00001f);
        assertEquals("lag 3", 0f, corr[3], 0.00001f);
    }

    @Test
    public void testIterDeconIdentity() throws Exception {
        // JUnitDoclet begin method phaseShift
        float[] data = new float[128];
        data[49] = .5f;
        IterDecon iterdecon = new IterDecon(100, true, .0001f, 3);
        IterDeconResult out = iterdecon.process(data, data, 1.0f);
        IterDecon.gaussianFilter(out.predicted, 3.0f, 1.0f);
        // for ( int i=0; i<out.predicted.length; i++) {
        // System.out.println("predicted "+i+"  data="+data[i]+"  out="+out.predicted[i]);
        // assertEquals("predicted "+i+"  data="+data[i]+"  out="+out.predicted[i],
        // data[i], out.predicted[i], 0.001);
        // } // end of for ()
        /*
         * these values come from running New_Decon_Process on a impulse
         * generated with sac's fg impulse command (100 datapoints, 1 at 49) The
         * receiver function of data from itself should be unity at lag 0 and
         * zero elsewhere, of course the gaussian tends to smear it out.
         * 
         * piglet 51>../New_Decon_Process/iterdecon_tjo
         * 
         * Program iterdeconfd - Version 1.0X, 1997-98 Chuck Ammon, Saint Louis
         * University
         * 
         * impulse100.sac impulse100.sac output 100 10 .001 3.0 1 0 output
         * 
         * The maximum spike delay is 64.00000
         * 
         * File Spike amplitude Spike delay Misfit Improvement r001
         * 0.100000012E+01 0.000 0.00% 100.0000% r002 -0.126299312E-06 0.000
         * 0.00% 0.0000%
         * 
         * Last Error Change = 0.0000%
         * 
         * Hit the min improvement tolerance - halting. Number of bumps in final
         * result: 1 The final deconvolution reproduces 100.0% of the signal.
         */
        assertEquals(0.9156569, out.predicted[0], .000001);
        assertEquals(0.04999885, out.predicted[1], .000001);
        assertEquals(-0.01094833, out.predicted[2], .000001);
        assertEquals(0.004774094, out.predicted[3], .000001);
        assertEquals(-0.002670953, out.predicted[4], .000001);
        // JUnitDoclet end method phaseShift
    }
}
