/*
 * Decompiled with CFR 0.152.
 */
package gov.usgs.winston.db;

import gov.usgs.earthworm.TraceBuf;
import gov.usgs.util.CurrentTime;
import gov.usgs.util.Time;
import gov.usgs.util.Util;
import gov.usgs.winston.db.Admin;
import gov.usgs.winston.db.WinstonDatabase;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TimeZone;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.collections.map.LRUMap;
import org.apache.commons.collections.set.MapBackedSet;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class InputEW {
    private WinstonDatabase winston;
    private DateFormat dateFormat;
    private Logger logger;
    private static Set<String> checkTableCache;
    private static Map<String, double[]> channelTimeSpans;
    private Map<String, SortedMap<Double, double[]>> channelHelicorderRows;
    private int maxRows = 300;
    private int numRowsToDelete = 60;

    public InputEW(WinstonDatabase w) {
        this.setWinston(w);
        checkTableCache = MapBackedSet.decorate(Collections.synchronizedMap(new LRUMap(w.getCacheCap() / 2, true)));
        channelTimeSpans = Collections.synchronizedMap(new HashMap());
        this.channelHelicorderRows = new HashMap<String, SortedMap<Double, double[]>>();
        this.dateFormat = new SimpleDateFormat("yyyy_MM_dd");
        this.dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
    }

    public void setWinston(WinstonDatabase db) {
        this.winston = db;
        this.logger = this.winston.getLogger();
    }

    public void setRowParameters(int mr, int nd) {
        this.maxRows = mr;
        this.numRowsToDelete = nd;
    }

    private List<String> getDayTables(String code) {
        ArrayList<String> list = new ArrayList<String>(10);
        try {
            ResultSet rs = this.winston.getStatement().executeQuery("SHOW TABLES");
            while (rs.next()) {
                list.add(rs.getString(1));
            }
            rs.close();
            Collections.sort(list);
            ArrayList<String> dayList = new ArrayList<String>(list.size() / 2);
            for (String table : list) {
                String day = table.substring(table.indexOf("$$") + 2);
                if (day.length() != 10 || day.charAt(4) != '_' || day.charAt(7) != '_' || !Character.isDigit(day.charAt(0)) || !Character.isDigit(day.charAt(9))) continue;
                dayList.add(table);
            }
            return dayList;
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Could not get list of tables for channel: " + code, e);
            return null;
        }
    }

    public void purgeTables(String channel, int days) {
        if (days <= 0) {
            return;
        }
        if (!this.winston.checkConnect()) {
            return;
        }
        if (!this.winston.useDatabase(channel)) {
            return;
        }
        List<String> list = this.getDayTables(channel);
        if (list == null) {
            return;
        }
        Date now = CurrentTime.getInstance().nowDate();
        Date then = new Date(now.getTime() - (long)days * 86400000L);
        String thenString = Time.format((String)"yyyy_MM_dd", (Date)then);
        this.winston.getLogger().info("Purging '" + channel + "' tables before: " + thenString);
        boolean deleted = false;
        boolean setTime = false;
        for (String table : list) {
            String[] ss = table.split("\\$\\$");
            if (thenString.compareTo(ss[1]) > 0) {
                try {
                    checkTableCache.remove(table);
                    this.winston.getStatement().execute("DROP TABLE " + table);
                    this.winston.getStatement().execute("DROP TABLE " + ss[0] + "$$H" + ss[1]);
                    deleted = true;
                    this.winston.getLogger().info("Deleted table: " + table);
                }
                catch (Exception e) {
                    this.winston.getLogger().severe("Could not drop old table: " + channel + ".  Are permissions set properly?");
                }
                continue;
            }
            if (!deleted) break;
            try {
                String nextLowestTable = table;
                ResultSet rs = this.winston.getStatement().executeQuery("SELECT MIN(st) FROM " + nextLowestTable);
                rs.next();
                double t1 = rs.getDouble(1);
                this.setTimeSpan(channel, t1, Double.NaN);
                rs.close();
                setTime = true;
            }
            catch (Exception e) {
                this.winston.getLogger().severe("Could not update span after dropping table: " + channel);
            }
            break;
        }
        if (deleted && !setTime) {
            this.winston.getLogger().info("Permanently deleting channel: " + channel);
            new Admin(this.winston).deleteChannel(channel);
        }
    }

    private boolean createDayTable(String code, String date) {
        try {
            double prevDayJ2k = Util.dateToJ2K((Date)this.dateFormat.parse(date)) - 86400.0;
            String prevDate = this.dateFormat.format(Util.j2KToDate((double)prevDayJ2k));
            String waveTable = code + "$$" + date;
            String heliTable = code + "$$H" + date;
            String waveTableLast = code + "$$" + prevDate;
            String heliTableLast = code + "$$H" + prevDate;
            String waveTableall = code + "$$" + "past2days";
            String heliTableall = code + "$$H" + "past2days";
            this.winston.getStatement().execute("CREATE TABLE " + waveTable + " (st DOUBLE PRIMARY KEY, et DOUBLE, sr DOUBLE, " + "datatype CHAR(3), tracebuf BLOB)");
            this.winston.getStatement().execute("CREATE TABLE " + heliTable + " (j2ksec DOUBLE PRIMARY KEY, smin INT, smax INT, " + "rcnt INT, rsam DOUBLE)");
            this.winston.getLogger().log(Level.INFO, "Creating VIEWs for VAlarm: " + heliTableall);
            String sql = "CREATE or REPLACE VIEW " + waveTableall + " AS SELECT * FROM " + waveTable;
            if (this.tableExists(waveTableLast)) {
                sql = sql + " UNION ALL select * from " + waveTableLast;
            }
            this.winston.getStatement().execute(sql);
            sql = "CREATE or REPLACE VIEW " + heliTableall + " AS SELECT * FROM " + heliTable;
            if (this.tableExists(heliTableLast)) {
                sql = sql + " UNION ALL select * from " + heliTableLast;
            }
            this.winston.getStatement().execute(sql);
            return true;
        }
        catch (Exception ex) {
            this.winston.getLogger().log(Level.SEVERE, "Could not create day table: '" + code + "$" + date + "'.", ex);
            return false;
        }
    }

    private boolean tableExists(String code, String date) {
        return this.tableExists(code + "$$" + date);
    }

    private boolean tableExists(String table) {
        if (checkTableCache.contains(table)) {
            return true;
        }
        try {
            ResultSet rs = this.winston.getStatement().executeQuery("SHOW TABLES LIKE '" + table + "'");
            boolean result = rs.next();
            if (result) {
                checkTableCache.add(table);
                rs.close();
            }
            return result;
        }
        catch (Exception exception) {
            return false;
        }
    }

    private void setTimeSpan(String channel, double st, double et) throws SQLException {
        double[] d = channelTimeSpans.get(channel);
        if (!Double.isNaN(st)) {
            if (d != null) {
                d[0] = st;
            }
            this.winston.getStatement().execute("UPDATE " + this.winston.getDatabasePrefix() + "_ROOT.channels SET st=" + st + " WHERE code='" + channel + "'");
        }
        if (!Double.isNaN(et)) {
            if (d != null) {
                d[1] = et;
            }
            this.winston.getStatement().execute("UPDATE " + this.winston.getDatabasePrefix() + "_ROOT.channels SET et=" + et + " WHERE code='" + channel + "'");
        }
    }

    private double[] getTimeSpan(String channel) {
        double[] d = channelTimeSpans.get(channel);
        if (d != null) {
            return d;
        }
        try {
            ResultSet rs = this.winston.getStatement().executeQuery("SELECT st, et FROM " + this.winston.getDatabasePrefix() + "_ROOT.channels WHERE code='" + channel + "'");
            d = new double[]{Double.NaN, Double.NaN};
            if (rs.next()) {
                d[0] = rs.getDouble(1);
                d[1] = rs.getDouble(2);
            }
            rs.close();
            channelTimeSpans.put(channel, d);
            return d;
        }
        catch (Exception e) {
            this.winston.getLogger().log(Level.SEVERE, "Could not get time span for channel: " + channel, e);
            return null;
        }
    }

    private double[] getHelicorderRow(String channel, double j2ksec, boolean useDB) {
        double[] d;
        SortedMap<Double, double[]> rows = this.channelHelicorderRows.get(channel);
        if (rows == null) {
            rows = new TreeMap<Double, double[]>();
            this.channelHelicorderRows.put(channel, rows);
        }
        if ((d = (double[])rows.get(j2ksec)) != null) {
            return d;
        }
        String date = this.dateFormat.format(Util.j2KToDate((double)j2ksec));
        String table = channel + "$$H" + date;
        if (useDB) {
            try {
                ResultSet rs = this.winston.getStatement().executeQuery("SELECT j2ksec, smin, smax, rcnt, rsam FROM " + table + " WHERE j2ksec=" + j2ksec);
                if (rs.next()) {
                    d = new double[]{rs.getDouble(1), rs.getDouble(2), rs.getDouble(3), rs.getDouble(4), rs.getDouble(5), 0.0, 0.0, 0.0};
                }
                rs.close();
            }
            catch (Exception e) {
                this.logger.warning("Could not get helicorder row: " + e.getMessage());
            }
        }
        if (d == null) {
            d = new double[]{j2ksec, 2.147483647E9, -2.147483648E9, 0.0, 0.0, 0.0, 0.0, 0.0};
        }
        rows.put(j2ksec, d);
        if (rows.size() > this.maxRows) {
            for (int i = 0; i < this.numRowsToDelete; ++i) {
                rows.remove(rows.firstKey());
            }
        }
        return d;
    }

    private double getRSAMMu(String channel, double j2ksec, int delta, int duration) {
        SortedMap<Double, double[]> rows = this.channelHelicorderRows.get(channel);
        if (rows == null) {
            return Double.NaN;
        }
        delta = -delta;
        double t = 0.0;
        double n = 0.0;
        for (int k = delta - duration; k < delta; ++k) {
            double[] hr = (double[])rows.get(j2ksec + (double)k);
            if (hr == null) continue;
            t += hr[5] * hr[3];
            n += hr[3];
        }
        if (n != 0.0) {
            return t / n;
        }
        return 0.0;
    }

    private PreparedStatement getInputStatement(String table, TraceBuf tb) {
        try {
            PreparedStatement insert = this.winston.getPreparedStatement("INSERT INTO " + table + " VALUES (?,?,?,?,?);");
            if (insert == null) {
                this.logger.severe("Just got a null ps");
            } else if (tb == null) {
                this.logger.severe("null tb");
            }
            insert.setDouble(1, tb.getStartTimeJ2K());
            insert.setDouble(2, tb.getEndTimeJ2K());
            insert.setDouble(3, tb.samplingRate);
            insert.setString(4, tb.dataType);
            byte[] compressed = Util.compress((byte[])tb.bytes, (int)1, (int)0, (int)(tb.bytes.length - 1));
            insert.setBytes(5, compressed);
            return insert;
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Could not create prepared statement: " + tb, e);
            this.logger.severe("TOMP " + e.getMessage());
            return null;
        }
    }

    private void updateHelicorderData(Set<Double> modifiedRows, String channel, String date, TraceBuf tb, boolean computeRsam, int delta, int duration, boolean useDB) {
        double fst = Math.floor(tb.getStartTimeJ2K());
        double cet = Math.ceil(tb.getEndTimeJ2K());
        double[][] heliList = new double[(int)Math.round(cet - fst) + 1][];
        int j = 0;
        for (int i = (int)Math.round(fst); i <= (int)Math.round(cet); ++i) {
            modifiedRows.add(Double.valueOf(i));
            heliList[j] = this.getHelicorderRow(channel, i, useDB);
            if (computeRsam) {
                double mu;
                heliList[j][6] = mu = this.getRSAMMu(channel, i, delta, duration);
            }
            ++j;
        }
        double st = tb.getStartTimeJ2K();
        double dt = 1.0 / tb.samplingRate;
        for (int i = 0; i < tb.numSamples; ++i) {
            j = (int)(Math.floor(st) - fst);
            double[] d = heliList[j];
            int sample = tb.data[i];
            d[1] = Math.min(d[1], (double)sample);
            d[2] = Math.max(d[2], (double)sample);
            if (computeRsam) {
                d[4] = (d[4] * d[3] + (double)Math.abs(sample)) / (d[3] + 1.0);
                d[5] = (d[5] * d[3] + (double)sample) / (d[3] + 1.0);
                d[7] = (d[7] * d[3] + Math.abs((double)sample - d[6])) / (d[3] + 1.0);
                d[3] = d[3] + 1.0;
            }
            st += dt;
        }
    }

    private double writeHelicorderData(String channel, Set<Double> modifiedRows) {
        for (double j2k : modifiedRows) {
            String date = this.dateFormat.format(Util.j2KToDate((double)j2k));
            String table = channel + "$$H" + date;
            double[] row = this.getHelicorderRow(channel, j2k, false);
            String sql = String.format("INSERT INTO %s (j2ksec, smin, smax, rcnt, rsam) VALUES (%f,%d,%d,%d,%f) ON DUPLICATE KEY UPDATE smin=VALUES(smin), smax=VALUES(smax), rcnt=VALUES(rcnt), rsam=VALUES(rsam)", table, j2k, (int)row[1], (int)row[2], (int)row[3], row[7]);
            try {
                this.winston.getStatement().execute(sql);
            }
            catch (SQLException ex) {
                this.logger.warning("Could not write helicorder row: " + ex.getMessage());
                return j2k;
            }
        }
        return Double.NaN;
    }

    public boolean rederive(List<TraceBuf> tbs, boolean computeRsam, int delta, int duration) {
        if (tbs == null || tbs.size() == 0) {
            return false;
        }
        if (!this.winston.checkConnect()) {
            return false;
        }
        String channel = tbs.get(0).toWinstonString();
        if (!this.winston.useDatabase(channel)) {
            return false;
        }
        TreeSet<Double> modifiedHeliRows = new TreeSet<Double>();
        for (TraceBuf tb : tbs) {
            if (tb == null || !tb.toWinstonString().equals(channel)) continue;
            double ts = tb.getStartTimeJ2K();
            String date = this.dateFormat.format(Util.j2KToDate((double)ts));
            this.updateHelicorderData(modifiedHeliRows, channel, date, tb, computeRsam, delta, duration, false);
        }
        return Double.isNaN(this.writeHelicorderData(channel, modifiedHeliRows));
    }

    private List<InputResult> getError(InputResult.Code code) {
        ArrayList<InputResult> list = new ArrayList<InputResult>(1);
        list.add(new InputResult(code, null));
        return list;
    }

    public List<InputResult> inputTraceBufs(List<TraceBuf> tbs, boolean computeRsam, int delta, int duration) {
        if (tbs == null || tbs.size() == 0) {
            return this.getError(InputResult.Code.ERROR_INPUT);
        }
        if (!this.winston.checkConnect()) {
            return this.getError(InputResult.Code.ERROR_NO_WINSTON);
        }
        String channel = tbs.get(0).toWinstonString();
        double[] span = this.getTimeSpan(channel);
        if (span == null) {
            return this.getError(InputResult.Code.ERROR_TIME_SPAN);
        }
        double stBefore = span[0];
        if (!this.winston.useDatabase(channel)) {
            return this.getError(InputResult.Code.ERROR_DATABASE);
        }
        ArrayList<InputResult> results = new ArrayList<InputResult>(tbs.size() + 1);
        TreeSet<Double> modifiedHeliRows = new TreeSet<Double>();
        Iterator<TraceBuf> it = tbs.iterator();
        while (it.hasNext()) {
            boolean tableCreated = false;
            TraceBuf tb = it.next();
            InputResult result = new InputResult(InputResult.Code.NO_CODE, tb);
            if (tb == null) {
                result.code = InputResult.Code.ERROR_NULL_TRACEBUF;
            }
            if (!tb.toWinstonString().equals(channel)) {
                result.code = InputResult.Code.ERROR_CHANNEL;
            }
            if (result.code != InputResult.Code.NO_CODE) continue;
            double ts = tb.getStartTimeJ2K();
            String date = this.dateFormat.format(Util.j2KToDate((double)ts));
            String endDate = this.dateFormat.format(Util.j2KToDate((double)(tb.getEndTimeJ2K() + 1.0)));
            String table = channel + "$$" + date;
            try {
                if (!this.tableExists(channel, date)) {
                    this.createDayTable(channel, date);
                    tableCreated = true;
                }
                if (!this.tableExists(channel, endDate)) {
                    this.createDayTable(channel, endDate);
                    tableCreated = true;
                }
                PreparedStatement insert = this.getInputStatement(table, tb);
                try {
                    insert.executeUpdate();
                }
                catch (SQLException ex) {
                    if (ex.getMessage().startsWith("Duplicate entry")) {
                        result.code = InputResult.Code.ERROR_DUPLICATE;
                    }
                    throw ex;
                }
                span[0] = Math.min(span[0], tb.getStartTimeJ2K());
                span[1] = Math.max(span[1], tb.getEndTimeJ2K());
                if (result.code != InputResult.Code.ERROR_DUPLICATE) {
                    this.updateHelicorderData(modifiedHeliRows, channel, date, tb, computeRsam, delta, duration, true);
                }
            }
            catch (SQLException ex) {
                result.code = InputResult.Code.ERROR_DATABASE;
                this.logger.log(Level.SEVERE, "Could not insert trace buf: ", ex);
            }
            if (result.code == InputResult.Code.NO_CODE) {
                result.code = tableCreated ? InputResult.Code.SUCCESS_CREATED_TABLE : InputResult.Code.SUCCESS;
            }
            results.add(result);
            tableCreated = false;
        }
        InputResult heliResult = new InputResult(InputResult.Code.SUCCESS_HELICORDER, null);
        double failed = this.writeHelicorderData(channel, modifiedHeliRows);
        if (!Double.isNaN(failed)) {
            heliResult.code = InputResult.Code.ERROR_HELICORDER;
            heliResult.failedHeliJ2K = failed;
        }
        results.add(heliResult);
        InputResult spanResult = new InputResult(InputResult.Code.ERROR_TIME_SPAN, null);
        try {
            if (span[0] == stBefore) {
                this.setTimeSpan(channel, Double.NaN, span[1]);
            } else {
                this.setTimeSpan(channel, span[0], span[1]);
            }
            spanResult.code = InputResult.Code.SUCCESS_TIME_SPAN;
        }
        catch (SQLException ex) {
            this.logger.log(Level.SEVERE, "Could not set time span for channel: " + channel, ex);
        }
        results.add(spanResult);
        return results;
    }

    public static class InputResult {
        public Code code;
        public TraceBuf traceBuf;
        public double failedHeliJ2K;

        public InputResult(Code c, TraceBuf tb) {
            this.code = c;
            this.traceBuf = tb;
        }

        /*
         * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
         */
        public static enum Code {
            NO_CODE,
            ERROR_INPUT,
            ERROR_NULL_TRACEBUF,
            ERROR_DUPLICATE,
            ERROR_UNKNOWN,
            ERROR_DATABASE,
            ERROR_NO_WINSTON,
            ERROR_CHANNEL,
            ERROR_TIME_SPAN,
            ERROR_HELICORDER,
            SUCCESS,
            SUCCESS_CREATED_TABLE,
            SUCCESS_HELICORDER,
            SUCCESS_TIME_SPAN;

        }
    }
}

