/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.util;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.eclipse.jetty.util.component.AbstractLifeCycle;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class PathWatcher
extends AbstractLifeCycle
implements Runnable {
    private static final boolean IS_WINDOWS;
    private static final Logger LOG;
    private static final Logger NOISY_LOG;
    private static final WatchEvent.Kind<?>[] WATCH_EVENT_KINDS;
    private final WatchService watcher;
    private final WatchEvent.Modifier[] watchModifiers;
    private final boolean nativeWatchService;
    private Map<WatchKey, Config> keys = new HashMap<WatchKey, Config>();
    private List<Listener> listeners = new ArrayList<Listener>();
    private List<PathWatchEvent> pendingAddEvents = new ArrayList<PathWatchEvent>();
    private long updateQuietTimeDuration = 1000L;
    private TimeUnit updateQuietTimeUnit = TimeUnit.MILLISECONDS;
    private Thread thread;

    protected static <T> WatchEvent<T> cast(WatchEvent<?> event) {
        return event;
    }

    public PathWatcher() throws IOException {
        this.watcher = FileSystems.getDefault().newWatchService();
        WatchEvent.Modifier[] modifiers = null;
        boolean nativeService = true;
        try {
            ClassLoader cl = Thread.currentThread().getContextClassLoader();
            Class<?> pollingWatchServiceClass = Class.forName("sun.nio.fs.PollingWatchService", false, cl);
            if (pollingWatchServiceClass.isAssignableFrom(this.watcher.getClass())) {
                nativeService = false;
                LOG.info("Using Non-Native Java {}", pollingWatchServiceClass.getName());
                Class<?> c = Class.forName("com.sun.nio.file.SensitivityWatchEventModifier");
                Field f = c.getField("HIGH");
                modifiers = new WatchEvent.Modifier[]{(WatchEvent.Modifier)f.get(c)};
            }
        }
        catch (Throwable t) {
            LOG.ignore(t);
        }
        this.watchModifiers = modifiers;
        this.nativeWatchService = nativeService;
    }

    public void addDirectoryWatch(final Config baseDir) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Watching directory {}", baseDir);
        }
        Files.walkFileTree(baseDir.dir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                FileVisitResult result = FileVisitResult.SKIP_SUBTREE;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("preVisitDirectory: {}", dir);
                }
                if (!baseDir.isExcluded(dir)) {
                    if (baseDir.isIncluded(dir)) {
                        PathWatchEvent event = new PathWatchEvent(dir, PathWatchEventType.ADDED);
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Pending {}", event);
                        }
                        PathWatcher.this.pendingAddEvents.add(event);
                    }
                    PathWatcher.this.register(dir, baseDir);
                    if (baseDir.shouldRecurseDirectory(dir) || baseDir.dir.equals(dir)) {
                        result = FileVisitResult.CONTINUE;
                    }
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("preVisitDirectory: result {}", new Object[]{result});
                }
                return result;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (baseDir.matches(file)) {
                    PathWatchEvent event = new PathWatchEvent(file, PathWatchEventType.ADDED);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Pending {}", event);
                    }
                    PathWatcher.this.pendingAddEvents.add(event);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public void addFileWatch(Path file) throws IOException {
        Path abs;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Watching file {}", file);
        }
        if (!(abs = file).isAbsolute()) {
            abs = file.toAbsolutePath();
        }
        Config config = new Config(abs.getParent());
        config.addIncludeGlobRelative("");
        config.addIncludeGlobRelative(file.getFileName().toString());
        this.addDirectoryWatch(config);
    }

    public void addListener(Listener listener) {
        this.listeners.add(listener);
    }

    private void appendConfigId(StringBuilder s) {
        ArrayList<Path> dirs = new ArrayList<Path>();
        for (Config config : this.keys.values()) {
            dirs.add(config.dir);
        }
        Collections.sort(dirs);
        s.append("[");
        if (dirs.size() > 0) {
            s.append(dirs.get(0));
            if (dirs.size() > 1) {
                s.append(" (+").append(dirs.size() - 1).append(")");
            }
        } else {
            s.append("<null>");
        }
        s.append("]");
    }

    @Override
    protected void doStart() throws Exception {
        StringBuilder threadId = new StringBuilder();
        threadId.append("PathWatcher-Thread");
        this.appendConfigId(threadId);
        this.thread = new Thread((Runnable)this, threadId.toString());
        this.thread.start();
        super.doStart();
    }

    @Override
    protected void doStop() throws Exception {
        this.watcher.close();
        super.doStop();
    }

    public Iterator<Listener> getListeners() {
        return this.listeners.iterator();
    }

    public long getUpdateQuietTimeMillis() {
        return TimeUnit.MILLISECONDS.convert(this.updateQuietTimeDuration, this.updateQuietTimeUnit);
    }

    protected void notifyOnPathWatchEvent(PathWatchEvent event) {
        for (Listener listener : this.listeners) {
            try {
                listener.onPathWatchEvent(event);
            }
            catch (Throwable t) {
                LOG.warn(t);
            }
        }
    }

    protected void register(Path dir, Config root) throws IOException {
        LOG.debug("Registering watch on {}", dir);
        if (this.watchModifiers != null) {
            WatchKey key = dir.register(this.watcher, WATCH_EVENT_KINDS, this.watchModifiers);
            this.keys.put(key, root.asSubConfig(dir));
        } else {
            WatchKey key = dir.register(this.watcher, WATCH_EVENT_KINDS);
            this.keys.put(key, root.asSubConfig(dir));
        }
    }

    public boolean removeListener(Listener listener) {
        return this.listeners.remove(listener);
    }

    @Override
    public void run() {
        HashMap<Path, PathWatchEvent> pendingUpdateEvents = new HashMap<Path, PathWatchEvent>();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Starting java.nio file watching with {}", this.watcher);
        }
        block5: while (true) {
            Config config;
            WatchKey key = null;
            try {
                if (!this.pendingAddEvents.isEmpty()) {
                    for (PathWatchEvent event : this.pendingAddEvents) {
                        this.notifyOnPathWatchEvent(event);
                    }
                    this.pendingAddEvents.clear();
                }
                if (pendingUpdateEvents.isEmpty()) {
                    if (NOISY_LOG.isDebugEnabled()) {
                        NOISY_LOG.debug("Waiting for take()", new Object[0]);
                    }
                    key = this.watcher.take();
                } else {
                    if (NOISY_LOG.isDebugEnabled()) {
                        NOISY_LOG.debug("Waiting for poll({}, {})", new Object[]{this.updateQuietTimeDuration, this.updateQuietTimeUnit});
                    }
                    if ((key = this.watcher.poll(this.updateQuietTimeDuration, this.updateQuietTimeUnit)) == null) {
                        Iterator<PathWatchEvent> iterator = new HashSet(pendingUpdateEvents.keySet()).iterator();
                        while (true) {
                            if (!iterator.hasNext()) continue block5;
                            Path path = (Path)((Object)iterator.next());
                            PathWatchEvent pending = (PathWatchEvent)pendingUpdateEvents.get(path);
                            if (!pending.isQuiet(this.updateQuietTimeDuration, this.updateQuietTimeUnit)) continue;
                            this.notifyOnPathWatchEvent(pending);
                            pendingUpdateEvents.remove(path);
                        }
                    }
                }
            }
            catch (ClosedWatchServiceException e) {
                return;
            }
            catch (InterruptedException e) {
                if (this.isRunning()) {
                    LOG.warn(e);
                } else {
                    LOG.ignore(e);
                }
                return;
            }
            if ((config = this.keys.get(key)) == null) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("WatchKey not recognized: {}", key);
                continue;
            }
            for (WatchEvent<?> event : key.pollEvents()) {
                WatchEvent.Kind<?> kind = event.kind();
                WatchEvent<Path> ev = PathWatcher.cast(event);
                Path name = (Path)ev.context();
                Path child = config.dir.resolve(name);
                if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                    if (Files.isDirectory(child, LinkOption.NOFOLLOW_LINKS)) {
                        try {
                            this.addDirectoryWatch(config.asSubConfig(child));
                        }
                        catch (IOException e) {
                            LOG.warn(e);
                        }
                        continue;
                    }
                    if (!config.matches(child)) continue;
                    this.notifyOnPathWatchEvent(new PathWatchEvent(child, ev));
                    continue;
                }
                if (!config.matches(child)) continue;
                if (kind == StandardWatchEventKinds.ENTRY_MODIFY) {
                    PathWatchEvent pending = (PathWatchEvent)pendingUpdateEvents.get(child);
                    if (pending == null) {
                        pendingUpdateEvents.put(child, new PathWatchEvent(child, ev));
                        continue;
                    }
                    if (pending.isQuiet(this.updateQuietTimeDuration, this.updateQuietTimeUnit)) {
                        this.notifyOnPathWatchEvent(pending);
                        pendingUpdateEvents.remove(child);
                        continue;
                    }
                    pending.incrementCount(ev.count());
                    continue;
                }
                this.notifyOnPathWatchEvent(new PathWatchEvent(child, ev));
            }
            if (key.reset()) continue;
            this.keys.remove(key);
            if (this.keys.isEmpty()) break;
        }
    }

    public void setUpdateQuietTime(long duration, TimeUnit unit) {
        long desiredMillis = unit.toMillis(duration);
        if (!this.nativeWatchService && desiredMillis < 5000L) {
            LOG.warn("Quiet Time is too low for non-native WatchService [{}]: {} < 5000 ms (defaulting to 5000 ms)", this.watcher.getClass().getName(), desiredMillis);
            this.updateQuietTimeDuration = 5000L;
            this.updateQuietTimeUnit = TimeUnit.MILLISECONDS;
            return;
        }
        if (IS_WINDOWS && desiredMillis < 1000L) {
            LOG.warn("Quiet Time is too low for Microsoft Windows: {} < 1000 ms (defaulting to 1000 ms)", desiredMillis);
            this.updateQuietTimeDuration = 1000L;
            this.updateQuietTimeUnit = TimeUnit.MILLISECONDS;
            return;
        }
        this.updateQuietTimeDuration = duration;
        this.updateQuietTimeUnit = unit;
    }

    public String toString() {
        StringBuilder s = new StringBuilder(this.getClass().getName());
        this.appendConfigId(s);
        return s.toString();
    }

    static {
        String os = System.getProperty("os.name");
        if (os == null) {
            IS_WINDOWS = false;
        } else {
            String osl = os.toLowerCase(Locale.ENGLISH);
            IS_WINDOWS = osl.contains("windows");
        }
        LOG = Log.getLogger(PathWatcher.class);
        NOISY_LOG = Log.getLogger(PathWatcher.class.getName() + ".Noisy");
        WATCH_EVENT_KINDS = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY};
    }

    public static enum PathWatchEventType {
        ADDED,
        DELETED,
        MODIFIED,
        UNKNOWN;

    }

    public static class PathWatchEvent {
        private final Path path;
        private final PathWatchEventType type;
        private int count;
        private long timestamp;
        private long lastFileSize = -1L;

        public PathWatchEvent(Path path, PathWatchEventType type) {
            this.path = path;
            this.count = 0;
            this.type = type;
            this.timestamp = System.currentTimeMillis();
        }

        public PathWatchEvent(Path path, WatchEvent<Path> event) {
            this.path = path;
            this.count = event.count();
            this.type = event.kind() == StandardWatchEventKinds.ENTRY_CREATE ? PathWatchEventType.ADDED : (event.kind() == StandardWatchEventKinds.ENTRY_DELETE ? PathWatchEventType.DELETED : (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY ? PathWatchEventType.MODIFIED : PathWatchEventType.UNKNOWN));
            this.timestamp = System.currentTimeMillis();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            PathWatchEvent other = (PathWatchEvent)obj;
            if (this.path == null ? other.path != null : !this.path.equals(other.path)) {
                return false;
            }
            return this.type == other.type;
        }

        public boolean isQuiet(long expiredDuration, TimeUnit expiredUnit) {
            long now = System.currentTimeMillis();
            long pastdue = this.timestamp + expiredUnit.toMillis(expiredDuration);
            this.timestamp = now;
            try {
                long fileSize = Files.size(this.path);
                boolean fileSizeChanged = this.lastFileSize != fileSize;
                this.lastFileSize = fileSize;
                if (now > pastdue && !fileSizeChanged) {
                    return true;
                }
            }
            catch (IOException e) {
                LOG.debug("Cannot read file size: " + this.path, e);
            }
            return false;
        }

        public int getCount() {
            return this.count;
        }

        public Path getPath() {
            return this.path;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        public PathWatchEventType getType() {
            return this.type;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (this.path == null ? 0 : this.path.hashCode());
            result = 31 * result + (this.type == null ? 0 : this.type.hashCode());
            return result;
        }

        public void incrementCount(int num) {
            this.count += num;
        }

        public String toString() {
            return String.format("PathWatchEvent[%s|%s,count=%d]", new Object[]{this.type, this.path, this.count});
        }
    }

    public static interface Listener {
        public void onPathWatchEvent(PathWatchEvent var1);
    }

    public static class Config {
        private static final String PATTERN_SEP;
        protected final Path dir;
        protected int recurseDepth = 0;
        protected List<PathMatcher> includes;
        protected List<PathMatcher> excludes;
        protected boolean excludeHidden = false;

        public Config(Path path) {
            this.dir = path;
            this.includes = new ArrayList<PathMatcher>();
            this.excludes = new ArrayList<PathMatcher>();
        }

        public void addExclude(PathMatcher matcher) {
            this.excludes.add(matcher);
        }

        public void addExclude(String syntaxAndPattern) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding exclude: [{}]", syntaxAndPattern);
            }
            this.addExclude(this.dir.getFileSystem().getPathMatcher(syntaxAndPattern));
        }

        public void addExcludeGlobRelative(String pattern) {
            this.addExclude(this.toGlobPattern(this.dir, pattern));
        }

        public void addExcludeHidden() {
            if (!this.excludeHidden) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Adding hidden files and directories to exclusions", new Object[0]);
                }
                this.excludeHidden = true;
                this.addExclude("regex:^.*" + PATTERN_SEP + "\\..*$");
                this.addExclude("regex:^.*" + PATTERN_SEP + "\\..*" + PATTERN_SEP + ".*$");
            }
        }

        public void addExcludes(List<String> syntaxAndPatterns) {
            for (String syntaxAndPattern : syntaxAndPatterns) {
                this.addExclude(syntaxAndPattern);
            }
        }

        public void addInclude(PathMatcher matcher) {
            this.includes.add(matcher);
        }

        public void addInclude(String syntaxAndPattern) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding include: [{}]", syntaxAndPattern);
            }
            this.addInclude(this.dir.getFileSystem().getPathMatcher(syntaxAndPattern));
        }

        public void addIncludeGlobRelative(String pattern) {
            this.addInclude(this.toGlobPattern(this.dir, pattern));
        }

        public void addIncludes(List<String> syntaxAndPatterns) {
            for (String syntaxAndPattern : syntaxAndPatterns) {
                this.addInclude(syntaxAndPattern);
            }
        }

        public Config asSubConfig(Path dir) {
            Config subconfig = new Config(dir);
            subconfig.includes = this.includes;
            subconfig.excludes = this.excludes;
            subconfig.recurseDepth = this.recurseDepth - 1;
            return subconfig;
        }

        public int getRecurseDepth() {
            return this.recurseDepth;
        }

        private boolean hasMatch(Path path, List<PathMatcher> matchers) {
            for (PathMatcher matcher : matchers) {
                if (!matcher.matches(path)) continue;
                return true;
            }
            return false;
        }

        public boolean isExcluded(Path dir) throws IOException {
            if (this.excludeHidden && Files.isHidden(dir)) {
                if (NOISY_LOG.isDebugEnabled()) {
                    NOISY_LOG.debug("isExcluded [Hidden] on {}", dir);
                }
                return true;
            }
            if (this.excludes.isEmpty()) {
                return false;
            }
            boolean matched = this.hasMatch(dir, this.excludes);
            if (NOISY_LOG.isDebugEnabled()) {
                NOISY_LOG.debug("isExcluded [{}] on {}", matched, dir);
            }
            return matched;
        }

        public boolean isIncluded(Path dir) {
            if (this.includes.isEmpty()) {
                if (NOISY_LOG.isDebugEnabled()) {
                    NOISY_LOG.debug("isIncluded [All] on {}", dir);
                }
                return true;
            }
            boolean matched = this.hasMatch(dir, this.includes);
            if (NOISY_LOG.isDebugEnabled()) {
                NOISY_LOG.debug("isIncluded [{}] on {}", matched, dir);
            }
            return matched;
        }

        public boolean matches(Path path) {
            try {
                return !this.isExcluded(path) && this.isIncluded(path);
            }
            catch (IOException e) {
                LOG.warn("Unable to match path: " + path, e);
                return false;
            }
        }

        public void setRecurseDepth(int depth) {
            this.recurseDepth = depth;
        }

        public boolean shouldRecurseDirectory(Path child) {
            if (!child.startsWith(child)) {
                return false;
            }
            int childDepth = this.dir.relativize(child).getNameCount();
            return childDepth <= this.recurseDepth;
        }

        private String toGlobPattern(Path path, String subPattern) {
            StringBuilder s = new StringBuilder();
            s.append("glob:");
            boolean needDelim = false;
            Path root = path.getRoot();
            if (root != null) {
                if (NOISY_LOG.isDebugEnabled()) {
                    NOISY_LOG.debug("Path: {} -> Root: {}", path, root);
                }
                for (Object c : (Object)root.toString().toCharArray()) {
                    if (c == 92) {
                        s.append(PATTERN_SEP);
                        continue;
                    }
                    s.append((char)c);
                }
            } else {
                needDelim = true;
            }
            for (Path segment : path) {
                if (needDelim) {
                    s.append(PATTERN_SEP);
                }
                s.append(segment);
                needDelim = true;
            }
            if (subPattern != null && subPattern.length() > 0) {
                if (needDelim) {
                    s.append(PATTERN_SEP);
                }
                for (Object c : (Object)subPattern.toCharArray()) {
                    if (c == 47) {
                        s.append(PATTERN_SEP);
                        continue;
                    }
                    s.append((char)c);
                }
            }
            return s.toString();
        }

        public String toString() {
            StringBuilder s = new StringBuilder();
            s.append(this.dir);
            if (this.recurseDepth > 0) {
                s.append(" [depth=").append(this.recurseDepth).append("]");
            }
            return s.toString();
        }

        static {
            String sep = File.separator;
            if (File.separatorChar == '\\') {
                sep = "\\\\";
            }
            PATTERN_SEP = sep;
        }
    }
}

